/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef RING_H
#define RING_H

#include <libpandora/global.h>

#include <libpandora/error.h>

template<class elt_t>
class Ring
{
private:
  typedef Ring<elt_t> ring_t;

private:
  elt_t	*elts;
  int	nelts;

  int	ptr;
  elt_t sum;

public:
  Ring(void)  : elts(NULL) 		{ init(0); }
  Ring(int n) : elts(NULL)		{ init(n); }
  ~Ring(void) 				{ clear(); }
  
  inline int size(void) const 		{ return nelts; }
  void init(int n);

  bool put(const elt_t &elt);
  inline elt_t last(void);
  inline elt_t average(void);

private:
  void clear(void);
};

template <class elt_t>
void Ring<elt_t>::init(int n)
{
  clear();
  if (n <= 0) return;
  nelts = n;
  elts = new elt_t[nelts];
  for (int i = 0; i < nelts; ++i) elts[i] = elt_t();
}

template <class elt_t>
void Ring<elt_t>::clear(void)
{
  __DELETE_ARRAY(elts);
  nelts = 0;
  ptr = 0;
  sum = elt_t();
}

template <class elt_t>
bool Ring<elt_t>::put(const elt_t &elt)
{
  if (elts == NULL) return false;

  sum = sum - elts[ptr] + elt;
  elts[ptr] = elt;
  ++ptr;

  if (ptr == nelts) ptr = 0;
  return true;
}

template <class elt_t>
elt_t Ring<elt_t>::last(void)
{
  return (ptr == 0) ? elts[nelts-1] : elts[ptr-1];
}

template <class elt_t>
elt_t Ring<elt_t>::average(void)
{
  return (elt_t)(sum / nelts);
}

#endif /* RING_H */


