/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#define RESOURCE_MANAGER_DEBUG 0

#include <libpandora/resourcemanager.h>
#include <libpandora/resource.h>
#include <libpandora/resource_entry.h>

ResourceManager::~ResourceManager(void)
{
  order.clear();
  resource_t *res = NULL;
  valuesDo(resources, res) __DELETE(res);
  resources.clear();
}

bool ResourceManager::addResource(const text &uri, int pri)
{
  long key = (long)uri;
  resource_t *res = resources.atOrNil(key);

  if (res == NULL) {
    resource_t *res = resource_t::parse(uri);
    if (res == NULL) {
      pandora_warning("invalid resource: " << uri);
      return false;
    }
#if RESOURCE_MANAGER_DEBUG
    pandora_debug("[resource add]      " << res->getURI() 
		 << " (" << pri << ")");
#endif
    resources.atPut(key, res);

    res->setPriority(pri);
    order.insert(elt_t(pri, key));
    return true;
  } else {
    return setPriority(uri, pri);
  }
}

bool ResourceManager::setPriority(const text &uri, int pri)
{
  long key = (long)uri;
  resource_t *res = resources.atOrNil(key);
  if (res == NULL) return false;
  int old = res->getPriority();

  map_iter_t ptr = multimap_find(order, old, key);
  if (ptr == order.end()) return false;
  order.erase(ptr);
  order.insert(elt_t(pri, key));

  res->setPriority(pri);

#if RESOURCE_MANAGER_DEBUG
  pandora_debug("[resource priority] " << res->getURI() << " (" 
	       << old << " -> " << pri << ")");
#endif
  return true;
}

bool ResourceManager::deleteResource(const text &uri)
{
  long key = (long)uri;
  resource_t *res = resources.atOrNil(key);
  if (res == NULL) {
    pandora_warning("resource is not present: " << uri);
    return false;
  }

  map_iter_t ptr = multimap_find(order, res->getPriority(), key);
  if (ptr == order.end()) return false;
  order.erase(ptr);

  resources.removeKey(key);
#if RESOURCE_MANAGER_DEBUG
  pandora_debug("[resource del]      " << res->getURI());
#endif
  __DELETE(res);
  return true;
}

int ResourceManager::listResources(resource_entry_t *re, int max_res)
{
  int i = 0;

  for (map_iter_t ptr = order.begin(); ptr != order.end(); ++ptr) {
    resource_t *res = resources.atOrNil(ptr->second);
    if (res == NULL) continue;
    if (i >= max_res) break;
    res->fillEntry(&(re[i]));
    ++i;
  }
  
  return i;
}

bool ResourceManager::update(void)
{
  for (multimap<int,long>::reverse_iterator ptr = order.rbegin(); 
       ptr != order.rend(); 
       ++ptr) {
    resource_t *res = resources.atOrNil(ptr->second);
    if (res == NULL) break;
    if (res->getPriority() < 0) break;
    if (check(res)) return true;
  }

  return false;
}

bool ResourceManager::check(const text &uri)
{
  resource_t *res = resources.atOrNil((long)uri);
  if (res == NULL) return false;
  return check(res);
}

bool ResourceManager::check(resource_t *res)
{
#if RESOURCE_MANAGER_DEBUG
  pandora_debug("[resource check]    " << res->getURI());
#endif
  if (res == NULL) return false;
  if (!res->isModified()) return false;

#if RESOURCE_MANAGER_DEBUG
  pandora_debug("[resource fetch]    " << res->getURI());
#endif

  char *str = NULL;
  if (!res->fetch(&str)) return false;

  bool ret = action(str, data);

  __FREE(str);

  return ret;
}

