/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef PANDORA_KEY_H
#define PANDORA_KEY_H

#include <libpandora/global.h>

#include <libpandora/text.h>
#include <libpandora/util.h>

class PandoraKey {
private:
  enum pkey_t { undefined = -1, single, pair, quad, textual, full };
  
  pkey_t	type;
  long 		k1, k2, k3, k4;
  text 		ks;

public:
  inline PandoraKey(void) : type(undefined), k1(0), k2(0), k3(0), k4(0) { }
  inline ~PandoraKey(void) { }

  inline PandoraKey(const PandoraKey &);
  inline PandoraKey& operator=(const PandoraKey &);

  inline operator long(void) const;
  inline bool operator ==(const PandoraKey &x) const;
  inline bool operator <(const PandoraKey &x) const;
  inline bool operator !=(const PandoraKey &x) const { return !operator==(x); }

  inline void set(long);
  inline void set(long, long);
  inline void set(long, long, long, long);
  inline void set(const text &);
  inline void set(const text &, long, long, long, long);
};

PandoraKey::PandoraKey(const PandoraKey& x) 
  : type(x.type), k1(x.k1), k2(x.k2), k3(x.k3), k4(x.k4), ks(x.ks)
{ 
}

PandoraKey& PandoraKey::operator =(const PandoraKey& x) 
{
  type = x.type; k1 = x.k1; k2 = x.k2; k3 = x.k3; k4 = x.k4; ks = x.ks;
  return *this;
}

#define ROT(v, r) ((v) << (r) | ((unsigned long)(v))>>(8 * SIZEOF_LONG - (r)))
PandoraKey::operator long(void) const
{
  switch (type) {
  case full:		return (string_hash(ks.data()) 
				^ (ROT(k1, 13) ^ ROT(k2, 23) 
				   ^ (k3 | (k4 << 16))));
  case textual:		return string_hash(ks.data());
  case quad:		return (ROT(k1, 13) ^ ROT(k2, 23) ^ (k3 | (k4 << 16)));
  case pair:		return (k1 ^ k2);
  case single:		return k1;
  case undefined:	break;
  }

  return 0;
}
#undef ROT

bool PandoraKey::operator ==(const PandoraKey &x) const 
{
  if (type != x.type) return false;

  switch (type) {
  case textual:		                return (ks == x.ks); 
  case full:		if (ks != x.ks) return false;        
  case quad:		if (k4 != x.k4) return false; 
    			if (k3 != x.k3) return false;        
  case pair:		if (k2 != x.k2) return false;        
  case single:		if (k1 != x.k1) return false;        
  case undefined:	                break;               
  }

  return true;
}

bool PandoraKey::operator <(const PandoraKey &x) const 
{
  if ((int)type < (int)x.type) return true;

  switch (type) {
  case textual:		               return (ks < x.ks);
  case full:		if (ks < x.ks) return true;
    			if (ks > x.ks) return false;
  case quad:		if (k4 < x.k4) return true;
    			if (k4 > x.k4) return false;
    			if (k3 < x.k3) return true;
    			if (k3 > x.k3) return false;
  case pair:		if (k2 < x.k2) return true;
    			if (k2 > x.k2) return false;
  case single:			       return (k1 < x.k1);
  case undefined:	               break;      
  }

  return false;
}

void PandoraKey::set(long xk1) 
{
  type = single;
  k1 = xk1; 
}

void PandoraKey::set(long xk1, long xk2) 
{
  type = pair;
  k1 = xk1; k2 = xk2; 
}

void PandoraKey::set(long xk1, long xk2, long xk3, long xk4) 
{
  type = quad;
  k1 = xk1; k2 = xk2; k3 = xk3; k4 = xk4;
}

void PandoraKey::set(const text &xks) 
{
  type = textual;
  ks = xks;
}

void PandoraKey::set(const text &xks, long xk1, long xk2, long xk3, long xk4) 
{
  type = full;
  ks = xks; k1 = xk1; k2 = xk2; k3 = xk3; k4 = xk4; 
}

#endif /* PANDORA_KEY_H */
