/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#define	ATOMIC_ALLOC  0
#define	ERRCHK_CTOR   0
#define ERRCHK_MALLOC 0

extern "C" {
#include <stdlib.h>

#if ATOMIC_ALLOC
#include <libpandora/thread.h>
#endif
	   }

#include <iostream>
#include <libpandora/error.h>

#if PANDORA_ALLOC

# if 1
#  define PRINTF(X)
# else
#  define PRINTF(X)	pandora_debug(X)
# endif

# define HDR_T u_int32_t


# if ATOMIC_ALLOC
static Mutex alloc_mx;
# endif


void *operator new(size_t lbs)
{
# if ATOMIC_ALLOC
  alloc_mx.lock();
# endif

# if ERRCHK_CTOR
  char *q= (char *)malloc(lbs + sizeof(HDR_T));
  while (q == 0) {
    pandora_warning("MEMORY EXHAUSTED (SLEEPING)");
    sleep(1);
    q= (char *)malloc(lbs + sizeof(HDR_T));
  }
  char *p= q + sizeof(HDR_T);
  *(HDR_T *)q= 0xd00df00d;
  PRINTF((void *)p << " malloc (new)");
# else
  char *p= (char *)malloc(lbs);
# endif

# if ATOMIC_ALLOC
  alloc_mx.unlock();
# endif
  return (void *)p;
}

void *operator new[](size_t lbs)
{
# if ATOMIC_ALLOC
  alloc_mx.lock();
# endif

# if ERRCHK_CTOR
  char *q= (char *)malloc(lbs + sizeof(HDR_T));
  while (q == 0) {
    pandora_warning("MEMORY EXHAUSTED (SLEEPING)");
    sleep(1);
    q= (char *)malloc(lbs + sizeof(HDR_T));
  }
  char *p= q + sizeof(HDR_T);
  *(HDR_T *)q= 0xd00dfeed;
  PRINTF((void *)p << " malloc (new[])");
# else
  char *p= (char *)malloc(lbs);  
# endif

# if ATOMIC_ALLOC
  alloc_mx.unlock();
# endif
  return (void *)p;
}

void operator delete(void *p)
{
  if (p != 0) {
# if ATOMIC_ALLOC
    alloc_mx.lock();
# endif

# if ERRCHK_CTOR
    char *q= (char *)p - sizeof(HDR_T);
    HDR_T hdr= *(HDR_T *)q;
    if (hdr == 0xdeadbeef) { pandora_error("DELETE DELETE"); }
    if (hdr != 0xd00df00d) { pandora_error("DELETE GARBAGE"); }
    *(HDR_T *)q= 0xdeadbeef;
    PRINTF(p << " free (delete)");
    free((void *)q);
# else
    free((void *)p);
# endif

# if ATOMIC_ALLOC
   alloc_mx.unlock();
# endif
  }
}

void operator delete[](void *p)
{
  if (p != 0) {
# if ATOMIC_ALLOC
    pthread_mutex_lock(&alloc_mx);
# endif
    
# if ERRCHK_CTOR
    char *q= (char *)p - sizeof(HDR_T);
    HDR_T hdr= *(HDR_T *)q;
    if (hdr == 0xdeadbeef) { pandora_error("DELETE DELETE"); }
    if (hdr != 0xd00dfeed) { pandora_error("DELETE GARBAGE"); }
    *(HDR_T *)q= 0xdeadbeef;
    PRINTF(p << " free (delete[])");
    free((void *)q);
# else
    free((void *)p);
# endif

# if ATOMIC_ALLOC
    alloc_mx.unlock();
# endif
  }
}

extern "C" {
void *xmalloc(size_t size)
{
# if ATOMIC_ALLOC
  alloc_mx.lock();
# endif

# if ERRCHK_MALLOC
  char *q = (char *) malloc(size + sizeof(HDR_T));
  while (q == 0) {
    pandora_warning("MEMORY EXHAUSTED (SLEEPING)");
    sleep(1);
    q = (char *)malloc(size + sizeof(HDR_T));
  }
  char *p = q + sizeof(HDR_T);
  *(HDR_T *)q = 0xd00df00d;
  PRINTF((void *)p << " malloc");
# else   
  char *p = (char *) malloc(size);
# endif
  
# if ATOMIC_ALLOC
  alloc_mx.unlock();
# endif

  if (p == NULL)
    pandora_error("out of memory (allocating " << (int) size << " bytes)");
  return (void *)p;
} 

void *xrealloc(void *p, size_t new_size)
{
  void *new_ptr; 
  if (p == NULL)     { return xmalloc(new_size); }
  if (new_size == 0) { xfree(p); return NULL; }

# if ATOMIC_ALLOC
  alloc_mx.lock();
# endif

# if ERRCHK_MALLOC
  char *q = (char *)p - sizeof(HDR_T);
  HDR_T hdr = *(HDR_T *)q;
  if (hdr == 0xdeadb00f) { pandora_error("REALLOC FREE"); }
  if (hdr != 0xd00df00d) { pandora_error("REALLOC GARBAGE"); }

  q = (char *) realloc(p, new_size + sizeof(HDR_T));
  while (q == 0) {
    pandora_warning("MEMORY EXHAUSTED (SLEEPING)");
    sleep(1);
    q = (char *)malloc(new_size + sizeof(HDR_T));
  }
  new_ptr = q + sizeof(HDR_T);
  *(HDR_T *)q = 0xd00df00d;
  PRINTF((void *)new_ptr << " realloc");

  *(HDR_T *)q = 0xdeadb00f;
  PRINTF(p << " realloc");
# else  
  new_ptr = realloc(p, new_size);
#endif  

# if ATOMIC_ALLOC
  alloc_mx.unlock();
# endif

  if (new_ptr == NULL)
    pandora_error("out of memory (new_size " << (int) new_size << " bytes)");
  return new_ptr;
} 

void xfree(void *p)
{
  if (p == NULL) {
#if 0
    pandora_error("NULL pointer given as argument");
#else
    return;
#endif
  }
# if ATOMIC_ALLOC
  alloc_mx.lock();
# endif

# if ERRCHK_MALLOC
    char *q = (char *)p - sizeof(HDR_T);
    HDR_T hdr = *(HDR_T *)q;
    if (hdr == 0xdeadb00f) { pandora_error("FREE FREE"); }
    if (hdr != 0xd00df00d) { pandora_error("FREE GARBAGE"); }
    *(HDR_T *)q = 0xdeadb00f;
    PRINTF(p << " free");
    free((void *)q);
# else  
  free(p);
# endif  

# if ATOMIC_ALLOC
  alloc_mx.lock();
# endif
} 

char *xstrdup(const char *str)
{
  if (str != NULL) {
#if (defined HAVE_STRDUP) && (!defined ERRCHK_MALLOC)
    char *dup = strdup(str);
#else
    char *dup = (char *)xmalloc (strlen (str) + 1);
    strcpy (dup, str);
#endif
    return dup;
  } else {
    return NULL;
  }
}

}

#endif

# undef HDR_T
