/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef PANDORA_H
#define PANDORA_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
#include <libpandora/conf/strings.h>
}

#include <libpandora/map.h>
#include <libpandora/pipe.h>
#include <libpandora/thread.h>
#include <libpandora/text.h>
#include <iostream>

class StackEntry;
class CompEntry;
class OptionEntry;
class Dispatcher;
class Component;
class MultiValue;
class Packet;

class Pandora {
private:
  struct disp_t {
    Dispatcher  *dispatcher;
    text 	id;
    bool 	root;

    disp_t(stack_handle_t, const text &, bool);
    ~disp_t(void);
  };

public:
  static long 		  		grm_id;

private:
  static stack_handle_t   		curhandle;
  static Mutex 		  		running_mx;
  Map<stack_handle_t, disp_t *>		running;
  Map<text, StackEntry *> 		repository;
  Pipe<stack_handle_t> 	  		toClean;

  int nbDisp, nbRunning;

public:
  Pandora(void);
  ~Pandora(void);

  bool init(const char *defs);
  bool update(void);

  bool getName(stack_handle_t, text &);

  stack_handle_t start(const text &id, bool threaded, bool root);

  bool stop(stack_handle_t);
  stack_handle_t stop(const text &);
  void stop(void);
  bool stop_async(stack_handle_t);
  void stop_async(void);

  bool suspend(stack_handle_t);
  bool resume(stack_handle_t);

  bool clean(stack_handle_t);
  void clean(void);
  void clean_async(void);
  void dclean(void) 			{ dispCleanup(false); }

  int getNbDefined(void) const		{ return repository.size(); }
  int getNbRunning(void) const		{ return nbRunning; }

  int listDefined(text *ids, int max_ids);  
  int listRunning(stack_handle_t *shs, int max_ids); 
 
  bool setStack(const StackEntry &se, bool dynupdate);

  bool setOption(const text&, const text&, const text&, const MultiValue &);
  bool setOption(stack_handle_t, const text&, const text&, const MultiValue &);
  bool getOption(stack_handle_t, const text &, const text &, MultiValue *);
  bool query(stack_handle_t, const text &, const text &, MultiValue *);

  stack_handle_t use(const text &id);
  bool use(stack_handle_t);
  bool release(stack_handle_t);
  bool push(stack_handle_t, Packet *);
  inline Dispatcher *dispatcher(stack_handle_t);

  StackEntry *getStackEntry(const text &id);
  
  void finished(stack_handle_t);
  
  int poll(bool blocking = true);
  int poll(stack_handle_t sh, bool blocking = true);

  bool expand(const text &sid, bool lup = false);

  void dump(ostream *f = &cout);

private:
  bool setup(Dispatcher *, StackEntry *, MultiValue *, int);
  bool expand(StackEntry *, bool lup);
  int  parseValues(const char *, MultiValue *, int);
  void dispCleanup(bool);
  inline disp_t *disp_entry(stack_handle_t);
  inline bool checkHandle(stack_handle_t);
  inline stack_handle_t nextHandle(void) 	{ return ++curhandle; }
};

extern Pandora *pandora;

extern "C" int pandora_init(void);
extern "C" int pandora_fini(void);

Pandora::disp_t *Pandora::disp_entry(stack_handle_t handle)
{
  return running.at(handle);
}

Dispatcher *Pandora::dispatcher(stack_handle_t handle)
{
  return disp_entry(handle)->dispatcher;
}

bool Pandora::checkHandle(stack_handle_t handle)
{
  return ((handle != NIL_STACK_HANDLE) && (running.includesKey(handle)));
}

#endif /* PANDORA_H */
