/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef PACKET_H
#define PACKET_H

#include <libpandora/global.h>

extern "C" {
#include <stdio.h>
#include <libpandora/conf/time.h>
}

#include <iostream>
#include <libpandora/map.h>
#include <libpandora/multimap.h>
#include <libpandora/util.h>
#include <libpandora/error.h>
#include <libpandora/dynloader.h>
#include <libpandora/clock.h>

class PacketFactory;

#define PACKET_SYMBOL_PREFIX	__pandora_packet_

#define packet_init(cl, vers)					\
  cl(void) { }							\
  static symbol_id_t packet_id;					\
  virtual symbol_id_t getID(void) { return this->packet_id; }	\
  virtual const char *getName(void) { return #cl; }		\
  const static int __version = vers;				\
  virtual Packet *clone(void) { return new cl(*this); }

#define packet_export(cl, from)						\
  extern "C" Packet *pandora_make_sym(packet,cl)(void)			\
    { return new cl(); }						\
  symbol_id_t cl::packet_id = DynLoader::make_id(#cl);			\
  static const bool __##cl##_from__ =					\
    ((add_pkt_map(&Packet::pkt_from, cl::packet_id, #from)), true);	\
  static const bool __##cl##_name__ =					\
    ((Packet::names.atPut(cl::packet_id, #cl)),				\
     true);								\
  EXPORT_SYMBOL(packet, cl)

#define locatePacket(type, var, start)				\
  type *(var) = NULL;						\
  do {								\
    Packet *__tmp__pkt__ = (start);				\
    if (__tmp__pkt__ != NULL) {					\
      __tmp__pkt__ = __tmp__pkt__->up;    			\
      while(__tmp__pkt__ != NULL 				\
	    && __tmp__pkt__->getID() != type::packet_id)	\
	__tmp__pkt__ = __tmp__pkt__->up;			\
      (var) = static_cast<type *>(__tmp__pkt__); 		\
    }								\
  } while(0)

#define locatePacket0(type, var, start)				\
  type *(var) = NULL;						\
  do {								\
    Packet *tmp = (start);					\
    while(tmp != NULL && tmp->getID() != type::packet_id) 	\
      tmp = tmp->up; 						\
    (var) = static_cast<type *>(tmp);				\
  } while(0)

#define cleanPacket(pkt) do { Packet::clean(pkt); pkt = NULL; } while(0)

/** Packet base class.
    This is the base class for each packet type used with any component.
*/
class Packet {
public:
  static Map<symbol_id_t, text> names;
  static MultiMap<symbol_id_t, symbol_id_t> pkt_from;

private:
  int refCount;

public:
  Packet *up;
  time_t curTime;
  timeval timeStamp;
  int branch;
#ifndef NDEBUG
  static int  created, alive;
#endif

protected:
  inline Packet(void);
  inline virtual ~Packet(void);

public:
  inline Packet(const Packet& x);
  inline Packet& operator= (const Packet& x);
  inline static void clean(Packet*);

  inline void packetSetUp(Packet *);
  inline void packetUnsetUp(void);
  inline void packetSetUpSkip(Packet *);

  inline void log(void) { print(&cerr); }
  inline virtual void print(ostream *) { }
  
  inline virtual symbol_id_t getID(void);
  inline virtual const char *getName(void);
  inline virtual Packet *clone(void);

  inline void refer(void) { ++refCount; }
  inline void release(void){ --refCount; }
  inline bool inUse(void) const { return (refCount > 0); }

  virtual size_t write(char *, size_t, int);
  virtual size_t read(const char *, int);

  inline void setTimeStamp(void);
  inline void setTimeStamp(const timeval &ts);

  static const char *getPrefix(void) { return cpp_quote(PACKET_SYMBOL_PREFIX);}

  static text get_name(symbol_id_t id) { return names.atOrNil(id); }
  static bool extends(symbol_id_t dst, symbol_id_t src);

  friend class PacketFactory;
};

extern void serialize(char *, size_t &, const size_t , Packet **);
extern void unserialize(const char *, size_t &, Packet **);

extern ostream &operator<< (ostream &, Packet&);
extern void add_pkt_map(MultiMap<symbol_id_t, symbol_id_t> *,
			symbol_id_t, const char *);

Packet::Packet(void) 
  : refCount(0), up(NULL), curTime(0), branch(0)
{
  timeStamp.tv_sec = (time_t) 0;
#ifndef NDEBUG
  ++created; ++alive;
#endif
}  

Packet::Packet(const Packet& x) 
  : refCount(0), curTime(x.curTime), timeStamp(x.timeStamp), branch(x.branch)
{
#if DEBUG_PKT
  pandora_debug("copy() packet: " << typeid(x).name()
	       << "\t[" << this << "]");
#endif
  if (x.up != NULL) packetSetUp((x.up)->clone());
  else up = NULL;
#ifndef NDEBUG
  ++created; ++alive;
#endif
}

Packet& Packet::operator= (const Packet& x) 
{
#if DEBUG_PKT
  pandora_debug("copy= packet: " << typeid(x).name()
	       << "\t[" << this << "]");
#endif
  refCount = 0;
  clean(up);
  if (x.up != NULL) packetSetUp((x.up)->clone());
  else up = NULL;
  curTime = x.curTime; 
  timeStamp = x.timeStamp; 
  branch = x.branch;
  return *this; 
}

Packet::~Packet(void) 
{ 
#if 0
  pandora_debug("delete packet: " << typeid(*this).name() 
	       << "\t[" << this << "]");
#endif
  pandora_assert(refCount == 0);
  Packet *old_up = up;
  packetUnsetUp();
  clean(old_up);
#ifndef NDEBUG
  --alive; 
#endif
}

void Packet::clean(Packet *pkt) 
{
#if DEBUG_PKT
  if (pkt != NULL) {
    pandora_debug("clean packet(" << pkt->refCount<< "): " 
		 << typeid(*pkt).name() 
		 << "\t[" << pkt << "]");
  }
#endif
  if (pkt == NULL) return;
  if (!(pkt->inUse())) __DELETE(pkt);
}

void Packet::packetSetUp(Packet *u) 
{
  if (u != NULL) {
    up = u;
    up->refer();
  }
}

void Packet::packetUnsetUp(void) 
{
  if (up != NULL) {
    up->release();
    up = NULL;
  }
}

void Packet::packetSetUpSkip(Packet *x) 
{
  packetSetUp(x->up);
  x->packetUnsetUp();
}

symbol_id_t Packet::getID(void)
{
  pandora_warning("getting plain Packet id");
  return 0;
}

const char *Packet::getName(void)
{
  pandora_warning("getting plain Packet name");
  return "Packet";
}

Packet *Packet::clone(void)
{
  pandora_warning("cloning plain Packet");
  return new Packet(*this);
}

void Packet::setTimeStamp(void) 
{ 
  wallclock->gettime(&timeStamp);
}

void Packet::setTimeStamp(const timeval &ts) 
{ 
  timeStamp = ts;
}

#endif /* PACKET_H */
