/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <sys/socket.h>
}

#include <libpandora/packet.h>
#include <libpandora/packetfactory.h>
#include <libpandora/serialize.h>

#define DEBUG_PKT 0

#if DEBUG_PKT
#include <typeinfo>
#endif

Map<symbol_id_t, text> Packet::names;
MultiMap<symbol_id_t, symbol_id_t> Packet::pkt_from;

#ifndef NDEBUG
int Packet::created = 0;
int Packet::alive = 0;
#endif

size_t Packet::write(char *, size_t, int)
{
  print(&cout);
  pandora_error("method not implemented");
  return 0;
}

size_t Packet::read(const char *, int)
{
  pandora_error("read function not implemented for this packet type");
  return 0;
}

static void _serialize(char *str, size_t &count, const size_t maxlen, 
		       Packet *pkt, int level) 
{
  if (pkt == NULL) {
    serialVar(level);
    return;
  }

  _serialize(str, count, maxlen, pkt->up, level+1);

  symbol_id_t ptype = pkt->getID();
  serialVar(ptype);
  serialVar(pkt->timeStamp);
  count += pkt->write(str + count, maxlen - count, level);
}

void serialize(char *str, size_t &count, const size_t maxlen, Packet **var)
{
  int level = 0;
  if (var == NULL || *var == NULL) {
    int l = -1;
    serialVar(l);
  }
  _serialize(str, count, maxlen, *var, level);
}

static Packet *_unserialize(const char *str, size_t &count, 
			    Packet *down, int level)
{
  symbol_id_t ptype;
  unserialVar(ptype);

  Packet *pkt = dynloader->createPacket(ptype);
  if (pkt == NULL)
    pandora_error("invalid packet type: #0x" << hex << ptype);
  unserialVar(pkt->timeStamp);
  count += pkt->read(str + count, level);

  if (down != NULL) pkt->packetSetUp(down);

  if (level <= 0) return pkt;

  return _unserialize(str, count, pkt, level-1);
}

void unserialize(const char *str, size_t &count, Packet **var)
{
  if (var == NULL) return;

  int level = 0;
  unserialVar(level);

  if (*var != NULL) cleanPacket(*var);
  if (level < 0) {
    *var = NULL;
    return;
  }
  Packet *pkt = _unserialize(str, count, NULL, level-1);
  *var = pkt;
}

ostream& operator<<(ostream& f, Packet &p)
{
  p.print(&f);
  return f;
}

bool Packet::extends(symbol_id_t dst, symbol_id_t src)
{
  if (dst == src) return true;
  symbol_id_t ids[16];
  int n = Packet::pkt_from.atOrNil(dst, ids, 16);
  for (int i = 0; i < n; ++i) 
    if (extends(ids[i], src)) 
      return true;
  return false;
}

void add_pkt_map(MultiMap<symbol_id_t, symbol_id_t> *mm,
		 symbol_id_t id, const char *str)
{
  if (mm == NULL) return;
  char *s2 = xstrdup(str);
  const char *delims = "+| \t\r\n";
  char *pkt = strtok(s2, delims);
  while(pkt != NULL) {
    symbol_id_t id2 = ((strncmp(pkt, "Packet", 6) == 0) 
		       ? (symbol_id_t) -1
		       : DynLoader::make_id(pkt));
    //pandora_debug(id << " -> " << id2);
    mm->atPut(id, id2);
    pkt = strtok(NULL, delims);
  }
  __FREE(s2);
}
