/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef MULTI_VALUE_H
#define MULTI_VALUE_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
	   }

#include <iostream>
#include <libpandora/serialize.h>
#include <libpandora/util.h>
#include <libpandora/error.h>

struct MultiValue {
public:
  enum multi_value_t {
    undefined = -1, 
    integer, 
    boolean, 
    floating,
    textual, 
    pointer
  } type;

  union multi_value {
    int   d;	// integer
    bool  b;	// boolean
    float f;	// float
    char *s;	// string
    void *v;	// pointer
  } value;
  
  MultiValue(void) : type(undefined)  			{ value.s = 0; }
  MultiValue(int d) : type(undefined)                   { set(d);      }
  MultiValue(bool b) : type(undefined)                  { set(b);      }
  MultiValue(float f): type(undefined)                  { set(f);      }
  MultiValue(const char *s): type(undefined)            { set(s);      }
  MultiValue(const char *s, int len) : type(undefined)  { set(s, len); }
  inline MultiValue(const MultiValue &x);
  inline MultiValue &operator= (const MultiValue &x);
  inline ~MultiValue(void);

  inline operator long() const;
  inline bool operator ==(const MultiValue &) const;
  inline bool operator !=(const MultiValue &x) { return !operator ==(x); }
  
  
  void print(ostream *f) const;

  inline void set(int d);
  inline void set(bool b);
  inline void set(float f);
  inline void set(const char *s);
  inline void set(const char *s, int len);
  inline void set(void *v);

  inline bool get(int *d);
  inline bool get(bool *b);
  inline bool get(float *f);
  inline bool get(char **s);
  inline bool get(void **v);

  inline void reset(void) 		{ cleanup(); type = undefined; }

private:
  inline void cleanup(void);

  friend void unserialize(const char *str, size_t &count, MultiValue *var);
};

extern ostream &operator<< (ostream &, const MultiValue &);

MultiValue::~MultiValue(void)
{
  cleanup();
  type = undefined;
}

MultiValue::MultiValue(const MultiValue &x) 
  : type(x.type)
{
  if (type == textual)  value.s = xstrdup(x.value.s);
  else value = x.value;
}

MultiValue &MultiValue::operator=(const MultiValue &x)
{
  cleanup();
  type = x.type;
  if (type == textual)  value.s = xstrdup(x.value.s);
  else value = x.value;
  return *this;
}


MultiValue::operator long() const
{
  switch(type) {
  case integer:	 	return value.d;
  case boolean:	 	return value.b;
  case floating: 	return (long)(value.f * (float)(2 << 16));
  case textual:	 	return string_hash(value.s);
  case pointer: 	return (long)value.v;
  case undefined:	break;
  }
  return -1;
}

bool MultiValue::operator ==(const MultiValue &x) const
{
  if (type != x.type) return false;
  switch(type) {
  case integer:	 	return value.d == x.value.d;
  case boolean:	 	return value.b == x.value.b;
  case floating: 	return value.f == x.value.f;
  case textual:	 	return string_eq(value.s, x.value.s);
  case pointer: 	return value.v == x.value.v;
  case undefined:	break;
  }
  return true;
}

void MultiValue::cleanup(void) 
{
  if (type == textual) {
    __FREE(value.s);
  } else {
    value.v = 0;
  }
}

serialEnum(MultiValue::multi_value_t);

inline void serialize(char *str, size_t &count, 
		      const size_t maxlen, const MultiValue *var)
{
  serialVar(var->type);

  switch(var->type) {
  case MultiValue::integer:	serialVar(var->value.d); break;
  case MultiValue::boolean:	serialVar(var->value.b); break;
  case MultiValue::floating:	serialVar(var->value.f); break;
  case MultiValue::textual:	serialVar(var->value.s); break;
  case MultiValue::undefined:    break;
  case MultiValue::pointer: 
    pandora_warning("cannot serialize this kind of MultiValue"); break;
  default: 
    pandora_warning("unknown MultiValue type: " << (int)var->type); break;
  }
}

inline void unserialize(const char *str, size_t &count, MultiValue *var)
{
  var->cleanup();
  unserialVar(var->type);

  switch(var->type) {
  case MultiValue::integer:	 unserialVar(var->value.d); break;
  case MultiValue::boolean:	 unserialVar(var->value.b); break;
  case MultiValue::floating:	 unserialVar(var->value.f); break;
  case MultiValue::textual:	 unserialVar(var->value.s); break;
  case MultiValue::undefined:    break;
  case MultiValue::pointer: 
    pandora_warning("cannot unserialize this kind of MultiValue"); break;
  default: 
    pandora_warning("unknown MultiValue type: " << (int)var->type); break;
  }
}

void MultiValue::set(int d)	
{ 
  cleanup(); 
  type = integer;  
  value.d = d; 
}

void  MultiValue::set(bool b)	
{ 
  cleanup(); 
  type = boolean;  
  value.b = b; 
}

void  MultiValue::set(float f)	
{ 
  cleanup(); 
  type = floating; 
  value.f = f; 
}

void  MultiValue::set(const char *str)	
{ 
  cleanup(); 
  type = textual;  
  value.s = (str != NULL) ? xstrdup(str) : NULL; 
}

void  MultiValue::set(const char *str, int len)	
{ 
  cleanup(); 
  type = textual;  
  value.s = (char *)xmalloc((len+1)*sizeof(char));
  memcpy(value.s, str, len*sizeof(char));
  (value.s)[len] = '\0';
}

void  MultiValue::set(void *v)	
{ 
  cleanup(); 
  type = pointer; 
  value.v = v; 
}

bool MultiValue::get(int *d)	
{ 
  if (type != integer) return false;
  *d = value.d;
  return true;
}

bool MultiValue::get(bool *b)	
{ 
  if (type != boolean) return false;
  *b = value.b;
  return true;
}

bool MultiValue::get(float *f)	
{ 
  if (type != floating) return false;
  *f = value.f;
  return true;
}

bool MultiValue::get(char **s)	
{ 
  if (type != textual) return false;
  __FREE(*s);
  *s = (value.s != NULL) ? xstrdup(value.s) : NULL;
  return true;
}

bool MultiValue::get(void **v)	
{ 
  if (type != pointer) return false;
  *v = value.v;
  return true;
}

#endif /* MULTI_VALUE_H */
