/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef MAP_H
#define MAP_H

#include <libpandora/global.h>

#include <libpandora/set.h>
#include <libpandora/association.h>

#define GENERIC	template<class Key, class Val> inline
#define MAP	Map<Key, Val>
#define ASSOC	Association<Key, Val>

template<class Key, class Val>
class Map : public Set<ASSOC>
{
public:
  typedef ASSOC assoc;

  Map(const assoc undefined= assoc(Key(), Val()), long nElements= 4)
    : Set<assoc>(undefined, nElements) {}

  Map(const Key undefined, long nElements= 4)
    : Set<assoc>(assoc(undefined, Val()), nElements) {}
#if 0
  Map()
    : Set<assoc>(assoc(Key(), Val()), 512) {}
#endif
  virtual ~Map(void) { }

public:
  assoc	      &associationAt(const Key &key) const;
  Val	      &atOrNil(const Key &key) const;
  Val	      &at(const Key &key) const;
  const long  findKeyOrNil(const Key &key) const;
  const long  scanForKey(const Key &anObject) const;
  void	       atPut(const Key &key, const Val &anObject);
  bool	       includesKey(const Key &key) const;
  void	       add(const assoc &anAssociation);
  Val	       removeKey(const Key &key);
  void	       rehash();
  Key	       &iterNextKey(set_iterator_t &iter);
  Val	       &iterNextValue(set_iterator_t &iter);

protected:
//virtual const Key &keyAt(const long index) const;
  virtual void       noCheckAdd(const assoc &assoc);
  virtual const long  scanFor(const assoc &assoc) const;
};


//
// INLINE IMPLEMENTATION
//

// Answer the association with the given key.

GENERIC ASSOC &MAP::associationAt(const Key &key) const
{
  long index;
  ASSOC &assoc= _array[index= findKeyOrNil(key)];
  pandora_assert(assoc != _undefined);
  return assoc;
}


GENERIC Val &MAP::atOrNil(const Key &key) const
{
  long index;
  ASSOC &assoc= _array[index= findKeyOrNil(key)];
  return assoc.value;
}

GENERIC Val &MAP::at(const Key &key) const
{
  long index;
  ASSOC &assoc= _array[index= findKeyOrNil(key)];
  pandora_assert(assoc != _undefined);
  return assoc.value;
}


// Set the value at key to be anObject.  If key is not found, create a new
// entry for key and set is value to anObject.

GENERIC void MAP::atPut(const Key &key, const Val &anObject)
{
  long index;
  ASSOC &element= _array[index= findKeyOrNil(key)];
  if (element == _undefined)
    atNewIndexPut(index, ASSOC(key, anObject));
  else
    element.value= anObject;
}


// Answer whether the receiver has a key equal to the argument, key.

GENERIC bool MAP::includesKey(const Key &key) const
{
  long index;

  index= findKeyOrNil(key);
  return _array[index] != _undefined;
}


// Add anAssociation as an element in the map.

GENERIC void MAP::add(const ASSOC &anAssociation)
{
  long index;
  ASSOC &element= _array[index= findElementOrNil(anAssociation)];
  if (element == _undefined)
    atNewIndexPut(index, anAssociation);
  else
    element.value= anAssociation.value;
}


// Remove key (and its associated value) from the map.

GENERIC Val MAP::removeKey(const Key &key)
{
  long index= findKeyOrNil(key);
  pandora_assert(_array[index] != _undefined);
  Val value = (_array[index]).value;
  _array[index]= _undefined;
  --_tally;
  fixCollisionsFrom(index);
  return value;
}


//GENERIC const ASSOC &MAP::keyAt(long index) const
//{
//  return _array[index];
//}


// Must be defined separately for Map because findElementOrNil
// expects a key, not an association.

GENERIC void MAP::noCheckAdd(const ASSOC &assoc)
{
  _array[findElementOrNil(assoc)]= assoc;
  ++_tally;
}


GENERIC void MAP::rehash(void)
{
  ASSOC *oldElements= _array;
  newArray(_capacity);
  //putchar('#'); fflush(stdout);
  for (long i= 0; i < _capacity; ++i)
    if (oldElements[i] != _undefined)
      noCheckAdd(oldElements[i]);
  __DELETE_ARRAY(oldElements);
}


// Scan the key array for the first slot containing either a nil
// (indicating an empty slot) or an element that matches anObject.  Answer
// the index of that slot or -1 if no slot is found.  This method will
// be overridden in various subclasses that have different
// interpretations for matching elements.

GENERIC const long MAP::scanForKey(const Key &anObject) const
{
  long start= ((unsigned long)anObject) * 2 % _capacity;

  // search from start to the end of the array
  for (long index= start; index < _capacity; ++index)
    if (_array[index] == _undefined || _array[index].key == anObject)
      return index;

  // search from 0 to where we started
  for (long index= 0; index < start - 1; ++index)
    if (_array[index] == _undefined || _array[index].key == anObject)
      return index;

  return -1;	// no match AND no empty slot
}


GENERIC const long MAP::scanFor(const ASSOC &assoc) const
{
  return scanForKey(assoc.key);
}


GENERIC const long MAP::findKeyOrNil(const Key &key) const
{
  long index;
  index= scanForKey(key);
  // index==-1 => Bad scene.  Neither have we found a matching element
  // nor even an empty slot.  No hashed set is ever supposed to get
  // completely full.
  pandora_assert(index != -1);
  return index;
}

GENERIC Key &MAP::iterNextKey(set_iterator_t &iter)
{
  return iterNextElement(iter).key;
}

GENERIC Val &MAP::iterNextValue(set_iterator_t &iter)
{
  return iterNextElement(iter).value;  
}


#define keysDo(MAP, KEY) \
  (KEY)= (MAP)._undefined.key; \
  for(long _1= 0; _1 < ((MAP)._capacity); ++_1) \
    if (((MAP)._array[_1] != (MAP)._undefined) && \
	(((KEY)= (((((MAP)._array)[_1])).key)), \
	 1))

#define valuesDo(MAP, VAL) \
  (VAL)= (MAP)._undefined.value; \
  for(long _2= 0; _2 < ((MAP)._capacity); ++_2) \
    if (((MAP)._array[_2] != (MAP)._undefined) && \
	(((VAL)= ((((MAP)._array)[_2]).value)), \
	 1))

#define keysValuesDo(MAP, KEY, VAL) \
  (KEY)= (MAP)._undefined.key; \
  (VAL)= (MAP)._undefined.value; \
  for(long _3= 0; _3 < ((MAP)._capacity); ++_3) \
    if (((MAP)._array[_3] != (MAP)._undefined) && \
	(((KEY)= ((((MAP)._array)[_3]).key)), \
	 ((VAL)= ((((MAP)._array)[_3]).value)), \
	 1))


#undef GENERIC
#undef MAP
#undef ASSOC


#endif /* MAP_H */
