/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#define GLOBAL_RESOURCE_MANAGER_DEBUG 0

#include <libpandora/globalresourcemanager.h>
#include <libpandora/resourcemanager.h>

GlobalResourceManager *grm = NULL;

static bool grm_update(const char *str, void *d)
{
  if (d == NULL) return false;
  GlobalResourceManager *g = static_cast<GlobalResourceManager *>(d);
  return g->init(str);
}

long GlobalResourceManager::grm_id = string_hash("resources");

GlobalResourceManager::GlobalResourceManager(void)
{
  if (grm != NULL) {
    pandora_error("only one instance of GlobalresourceManager may exist");
  } else {
    grm = this;
  }  
  registerManager(grm_id, &grm_update, this);
}

GlobalResourceManager::~GlobalResourceManager(void)
{
  ResourceManager *rm = NULL;
  valuesDo(managers, rm) __DELETE(rm);
  managers.clear();
  grm = NULL;
}

void GlobalResourceManager::registerManager(long id, action_t a, void *d)
{
#if GLOBAL_RESOURCE_MANAGER_DEBUG
  pandora_debug("registering resource manager for id: " << id);
#endif
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) {
    rm = new ResourceManager(a, d);
    managers.atPut(id, rm);
  } else {
    rm->setAction(a,d);
  }
}

bool GlobalResourceManager::addResource(long id, const text &uri, int pri)
{
#if GLOBAL_RESOURCE_MANAGER_DEBUG
  pandora_debug("adding resource for id: " << id);
#endif
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) return false;
  return rm->addResource(uri, pri);
}

bool GlobalResourceManager::setPriority(long id, const text &uri, int pri)
{
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) return false;
  return rm->setPriority(uri, pri);
}

bool GlobalResourceManager::deleteResource(long id, const text &uri)
{
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) return false;
  return rm->deleteResource(uri);
}

int GlobalResourceManager::listResources(long id, resource_entry_t *re, 
					 int max_res)
{
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) return false;
  return rm->listResources(re, max_res);
}

bool GlobalResourceManager::update(long id)
{
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) return false;
  return rm->update();
}

bool GlobalResourceManager::check(long id, const text &uri)
{
  ResourceManager *rm = managers.atOrNil(id);
  if (rm == NULL) return false;
  return rm->deleteResource(uri);
}

bool GlobalResourceManager::init(const char *str)
{
  char buf[2048];

  const char *start = str;
  char *ptr = NULL;
  const char *delims = " \t\r\n";

  while ((ptr = strchr(start, '\n')) != NULL) {
    int i = 0;
    int lsz = ptr - start;
    int csz = pandora_min(lsz, (int)(sizeof(buf)-1));
    memcpy(buf, start, csz);
    start += (lsz+1);
    buf[csz] = '\0';

    if (buf[0] == '#') continue;
    char *tag = strtok(buf, delims);
    if (tag == NULL) continue;

#if GLOBAL_RESOURCE_MANAGER_DEBUG
    pandora_debug("adding resource for: " << tag);
#endif
    char *uri = strtok(NULL, delims);
    if (uri == NULL) continue;
    char *pri_str = strtok(NULL, delims);
    if (pri_str == NULL) pri_str = "1000";

    addResource(string_hash(tag), uri, atoi(pri_str));
  }

  return true;
}
