/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <libpandora/global.h>

#include <libpandora/dynsymbol.h>
#include <libpandora/dynlibrary.h>
#include <libpandora/error.h>

#define DYN_SYMBOL_DEBUG 0

DynSymbol::DynSymbol(const text &name, DynLibrary *l) 
  : symbol(name), lib(l), count(0), old(false)
{ 
  pandora_assert(lib != NULL);
  lib->used();
#if DYN_SYMBOL_DEBUG
  pandora_debug("create dynsym: " << symbol << " [" << lib->lib.name << "]");
#endif
}

DynSymbol::~DynSymbol(void) 
{ 
  pandora_assert(lib != NULL);
  lib->unused();
#if DYN_SYMBOL_DEBUG
  pandora_debug("delete dynsym: " << symbol << " [" << lib->lib.name << "]");
#endif
  pandora_assert(count == 0);
  if (lib->useless()) {
    __DELETE(lib);
  } else {
    lib = NULL; 
  }
}

void *DynSymbol::resolve(void)
{
  pandora_assert(lib != NULL);

  if ((count == 0) && (!lib->open())) return NULL;
  ++count;

#if DYN_SYMBOL_DEBUG
  pandora_debug("resolve dynsym: " << symbol << " [" << lib->lib.name << "]");
#endif

  return lib->sym(symbol.data());
}

bool DynSymbol::use(void)
{
  pandora_assert(lib != NULL);
  ++count;

#if DYN_SYMBOL_DEBUG
  pandora_debug("use dynsym:     " << symbol << " [" << lib->lib.name << "]");
#endif

  return (count > 0);
}

bool DynSymbol::release(void)
{
  pandora_assert(lib != NULL);

  --count;

#if DYN_SYMBOL_DEBUG
  pandora_debug("release dynsym: " << symbol << " [" << lib->lib.name << "]");
#endif

  if (count == 0) return lib->close();

  return true;
}

bool DynSymbol::flush(void)
{
  pandora_assert(lib != NULL);

  count = 0;

#if DYN_SYMBOL_DEBUG
  pandora_debug("flush   dynsym: " << symbol << " [" << lib->lib.name << "]");
#endif

  return lib->close();
}

void DynSymbol::print(ostream *f)
{
  if (f == NULL) return;
  if (symbol.isNull()) return;
  if (lib == NULL) return;

  char type[64];
  char *sym = symbol.data(), *tmp = NULL;
  const char *prefix = "__pandora_";
  int plen = strlen(prefix);

  if ((strncmp(sym, prefix, plen) == 0)
      && ((tmp = strchr(sym+plen+1, '_')) != NULL)){
    int l = pandora_min((int)(sizeof(type)-1), tmp-sym-plen);
    strncpy(type, sym+plen, l);
    type[l] = '\0';
    sym = tmp+1;
  } else {
    strcpy(type, "??");
  }

  *f << "[" << type << "] " << sym << " " << lib->getID() << "\n";
}

ostream& operator<<(ostream& f, DynSymbol &dsym)
{
  dsym.print(&f);
  return f;
}
