/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef DYN_LOADER_H
#define DYN_LOADER_H

#include <libpandora/global.h>

#include <libpandora/dynbindings.h>
#include <libpandora/packetfactory.h>
#include <libpandora/thread.h>
#include <libpandora/map.h>
#include <libpandora/text.h>

class PacketFactory;

#define pandora_make_sym(type, func) __pandora_##type##_##func

#define EXPORT_SYMBOL(type, func)					 \
  static const bool __##func##_tmp__ =					 \
    DynLoader::registerFunc((symbol_id_t)string_hash(#func),		 \
			    (void *)&pandora_make_sym(type, func))  


class DynLoader {
public:
  static long grm_id;

private:
  static Map<symbol_id_t, void *> 	static_ctors;
  static Mutex 			mx;
  DynBindings 			bindings;
  PacketFactory 		pf;
  Map<symbol_id_t, text>	symbols;
  bool 				dynamic;

public:
  DynLoader(void);
  ~DynLoader(void);

  bool init(const char *);
  bool update(void);

  void *load(symbol_id_t id);
  void  unload(void *sym);
  void *refresh(symbol_id_t id, void *sym);

  Packet *createPacket(symbol_id_t ptype) { return pf.createPacket(ptype); }

  static bool registerFunc(symbol_id_t, void *);

  bool registerSymbol(const text &name, const text &lib, const text &prefix);
  bool registerLibrary(const text &id, long version, const text &loc);
  bool registerDependencies(const text &id, const text *deps, int nbdeps);

  bool lookup(text &xname, const text &sname);
  
  bool getSymbol(const text &name, text &lib);
  int  getLibrary(const text &id, int &version, text &loc,
		  text *deps, int max_deps);

  int listSymbols(text *syms, int max_syms);
  int listLibraries(text *libs, int max_libs);

  bool dump(ostream *f = &cout);

  inline int getNbSymbols(void) const 	{ return bindings.getNbSymbols(); }
  inline int getNbLibraries(void) const { return bindings.getNbLibraries(); }

  static inline symbol_id_t make_id(const char *p);
  static inline symbol_id_t make_id(const text &p);

private:
  void scanSymbols(const char **toks, int ntoks, const text &prefix);
  void scanLibraries(const char **tok, int ntoks);
  void scanDependencies(const char **tok, int ntoks);
};

extern DynLoader *dynloader;

symbol_id_t DynLoader::make_id(const char *pname) 
{ 
  return (symbol_id_t)(string_hash(pname)); 
}

symbol_id_t DynLoader::make_id(const text &pname) 
{ 
  return (symbol_id_t)(pname); 
}

#endif /* DYN_LOADER_H */
