/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#define DYN_LOADER_DEBUG 0

extern "C" {
#include <libpandora/conf/snprintf.h>
}

#include <iostream>
  
#include <libpandora/dynloader.h>
#include <libpandora/dynsymbol.h>
#include <libpandora/dynlibrary.h>
#include <libpandora/stackdesc.h>
#include <libpandora/globalresourcemanager.h>
#include <libpandora/packetfactory.h>
#include <libpandora/util.h>

#define DEBUG_MALLOC 0

DynLoader *dynloader = NULL;

#if DEBUG_MALLOC
#include <mcheck.h>
static int mtrace_tmp = (mtrace(), fprintf(stderr, "malloc debugging\n"), 0);
#endif

Map<symbol_id_t, void*> DynLoader::static_ctors(0, 31);
Mutex DynLoader::mx;

static bool dyn_update(const char *str, void *d)
{
  if (d == NULL) return false;
  DynLoader *dl = static_cast<DynLoader *>(d);
  //pandora_debug("dynloader init: " << str);
  return dl->init(str);
}

long DynLoader::grm_id = string_hash("libraries");

DynLoader::DynLoader(void)
{
  if (dynloader != NULL) {
    pandora_error("only one instance of DynLoader may exist");
  } else {
    dynloader = this;
  }

  if (grm != NULL) {
    grm->registerManager(grm_id, &dyn_update, this);
  }

#ifdef PIC
  dynamic = true;
#else
  dynamic = (static_ctors.size() < 16);
#endif

#if 0
  if (dynamic) {
    pandora_debug("using dynamic linkage");
  } else {
    pandora_debug("using static  linkage (" << static_ctors.size() << ")");
  }
#endif
}

DynLoader::~DynLoader(void)
{
  pf.clean();
  dynloader = NULL;
}

void *DynLoader::load(symbol_id_t id)
{
  void *func = NULL;

  if (dynamic) {
    text name = symbols.atOrNil(id);
    if (name.isNull()) {
      pandora_warning("undefined symbol with id: #" << id);
      return NULL;
    }
    mx.lock();
    func = bindings.useSymbol(name);
    mx.unlock();

    if (func == NULL) {
      pandora_warning("cannot resolve symbol: " << name);
    }
    
  } else {
    func = static_ctors.atOrNil(id);
  }


  return func;
}

void DynLoader::unload(void *sym)
{
  if (dynamic) {
    mx.lock();
    bindings.releaseSymbol(sym, false);
    mx.unlock();
  }
}

void *DynLoader::refresh(symbol_id_t id, void *sym)
{
  unload(sym);
  return load(id);
}

bool DynLoader::registerFunc(symbol_id_t id, void *func)
{
#if DYN_LOADER_DEBUG
  pandora_debug("resgistering func for id: " << id);
#endif
  static_ctors.atPut(id, func);
  return true;
}

bool DynLoader::registerSymbol(const text &name, 
			       const text &lib, 
			       const text &prefix)
{
  char buf[512]; 

  if (!prefix.isNull()) {
    snprintf(buf, sizeof(buf), "__pandora_%s_%s", prefix.data(), name.data());
  } else {
    strncpy(buf, name.data(), sizeof(buf));
  }

  text ctor(buf);

  mx.lock();
  bool ret = bindings.registerSymbol(ctor, lib);

  if (ret) {
    symbols.atPut((symbol_id_t) name, ctor);
#if DYN_LOADER_DEBUG
    pandora_debug("registering symbol: " << ctor << " [" << name << "]");
#endif
  }
  mx.unlock();

  return ret;
}

bool DynLoader::registerLibrary(const text &id, long version, const text &loc)
{
#if DYN_LOADER_DEBUG
  pandora_debug("registering library:  " << id);
#endif

  mx.lock();
  bool ret = bindings.registerLibrary(id, version, loc);
  mx.unlock();

  return ret;
}

bool DynLoader::registerDependencies(const text &id, 
				     const text *deps, int nbdeps)
{
#if DYN_LOADER_DEBUG
  pandora_debug("registering dependencies for:  " << id);
#endif

  mx.lock();
  bool ret = bindings.registerDependencies(id, deps, nbdeps);
  mx.unlock();

  return ret;
}

bool DynLoader::init(const char *str) 
{
#define MAX_TOKS 64
  char buf[2048];
  char *tag;
  char *tok[MAX_TOKS];

  const char *start = str;
  char *ptr = NULL;
  const char *delims = " \t\r\n";
  
  while ((ptr = strchr(start, '\n')) != NULL) {
    int i = 0;
    int lsz = ptr - start;
    int csz = pandora_min(lsz, (int)(sizeof(buf)-1));
    memcpy(buf, start, csz);
    start += (lsz+1);
    buf[csz] = '\0';

    if (buf[0] == '#') continue;
    tag = strtok(buf, delims);
    if (tag == NULL) continue;
    for (i = 0; i < MAX_TOKS; ++i) {
      tok[i] = strtok(NULL, delims);
      if (tok[i] == NULL) break;
    }

    if (strcasecmp(tag, "[library]") == 0) {
      scanLibraries((const char **)tok, i);
    } else if (strcasecmp(tag, "[dependency]") == 0) {
      scanDependencies((const char **)tok, i);
    } else { // symbol
      int len = strlen(tag);
      while ((len > 0) && tag[--len] != ']') /* empty */ ;
      if (len == 0) continue;
      tag[len] = '\0';
      scanSymbols((const char **)tok, i, tag+1);
    }
  }
#undef MAX_TOKS
  return true;
}

void DynLoader::scanLibraries(const char **tok, int ntoks)
{
  if (ntoks < 3) return;

  long version = strtol(tok[1], NULL, 0);

  if (!registerLibrary(tok[0], version, tok[2])) {
    pandora_warning("failed to register library: " << tok[0]);
  }
}

void DynLoader::scanDependencies(const char **tok, int ntoks)
{
  if (ntoks < 2) return;

#define MAX_LD_DEPS 16
  text lib_deps[MAX_LD_DEPS];
  int nbdeps = pandora_min(MAX_LD_DEPS, ntoks-1);

  for (int i = 0; i < nbdeps; ++i)
    lib_deps[i].init(tok[i+1]);

  if (!registerDependencies(tok[0], lib_deps, nbdeps)) {
    pandora_warning("failed to register dependencies for: " << tok[0]);
  }
#undef MAX_LD_DEPS
}

void DynLoader::scanSymbols(const char **tok, int ntoks,
			    const text &prefix)
{
  if (ntoks < 2) return;
    
  if (!registerSymbol(tok[0], tok[1], prefix)) {
    pandora_warning("failed to register symbol: " << tok[0]);
  }
}

bool DynLoader::lookup(text &xname, const text &sname)
{
  int plen = strlen(Component::getPrefix());
  text ctor, tmp;
  int nbmatch = 0;

  do {
    valuesDo(symbols, tmp) {
      if (tmp.length()-plen < sname.length()) continue;
      if (strncasecmp(tmp.data()+plen, sname.data(), sname.length()) == 0) {
	++nbmatch;
	if (nbmatch > 1) {
	  pandora_warning("ambiguous component name: " << sname);
	  goto failed;
	}
	ctor = tmp;
      }
    }
  } while (ctor.isNull() && update());

  if (ctor.isNull()) {
    pandora_warning("unknown component name: " << sname);
    goto failed;
  }

  xname.init((ctor.data())+plen);
  return true;

 failed:
  return false;
}

bool DynLoader::getSymbol(const text &name, text &lib)
{
  if (name.isNull()) return false;
  text real;
  if (!lookup(real, name)) return false;
  text sym = symbols.atOrNil((symbol_id_t)real);
  if (sym.isNull()) return false;
  DynSymbol *dsym = bindings.getSymbol(sym);
  if (dsym == NULL) return false;
  lib = dsym->getLib()->getID();
  return true;
}

int DynLoader::listSymbols(text *syms, int max_syms)
{
  return bindings.listSymbols(syms, max_syms);
}

int DynLoader::getLibrary(const text &id, int &version, text &loc,
			  text *deps, int max_deps)
{
  DynLibrary *dlib = bindings.getLibrary(id);
  if (dlib == NULL) return -1;
  version = dlib->getVersion();
  loc = dlib->getLocalisation();
  return  dlib->getDeps(deps, max_deps);
}

int DynLoader::listLibraries(text *libs, int max_libs)
{
  return bindings.listLibraries(libs, max_libs);
}

bool DynLoader::update(void)
{
  if (grm == NULL) return false;
  return grm->update(grm_id);
}

bool DynLoader::dump(ostream *f)
{
  update();
  return bindings.dump(f); 
}
