/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#include <libpandora/global.h>

#include <libpandora/dynbindings.h>
#include <libpandora/dynloader.h>
#include <libpandora/dynlibrary.h>
#include <libpandora/resource.h>
#include <libpandora/error.h>

#define DYN_LIBRARY_DEBUG 	0
#define DLREP_DEBUG 		0

DynLibrary::DynLibrary(const text &i, DynBindings *b)
  : id(i), bindings(b), version(-1), dep_names(NULL), deps(NULL),
    ndeps(0), use(0), count(0), old(false), lock(false)
{
#if DYN_LIBRARY_DEBUG > 1
  pandora_debug("create dynlib: " << id);
#endif
}

bool DynLibrary::setVersion(long ver)
{
  if ((version >= 0) && (version != ver)) {
    pandora_warning("cannot change version number for: " << id);
    return false;
  }
  version = ver;
  return true;
}

bool DynLibrary::setLocalisation(const text &loc)
{
  if (!resource_t::fetch(loc, lib.name)) {
    pandora_warning("invalid resource: " << loc);
    return false;
  }
  return true;
}



bool DynLibrary::setDeps(const text *d, int n)
{
  if (n > 0 && d == NULL) return false;
  ndeps = n;
  if (n > 0) {
    dep_names = new text[n];
    deps = new DynLibrary*[n];
    for (int i = 0; i < ndeps; ++i) dep_names[i] = d[i];
  }
#if DYN_LIBRARY_DEBUG > 1
  pandora_debug("set dependencies for: " << id << " (#" << ndeps << ")");
#endif
  return true;
}

DynLibrary::~DynLibrary(void) 
{
#if DYN_LIBRARY_DEBUG > 1
  pandora_debug("delete dynlib: " << lib.name);
#endif

  __DELETE_ARRAY(deps);
  __DELETE_ARRAY(dep_names);
  pandora_assert((use == 0) && (count == 0)); 
}

bool DynLibrary::open(void) 
{
  bool ret = true;
  lock = true;
  if (count == 0) {
    if ((dynloader != NULL) & (bindings != NULL)) {

      int i = 0;

      for (i = ndeps - 1; i >= 0; --i) {
	deps[i] = bindings->getLibrary(dep_names[i]);
	if (deps[i] == NULL) {
	  pandora_warning(lib.name << ": cannot find dependency library: " 
			  << dep_names[i]);
	  ret = false;
	  break;
	} else {
	  if ((deps[i])->lock) {
	    pandora_warning(lib.name << " -> " << dep_names[i] 
			    << ": dependency cycle, skipping");
	    deps[i] = NULL;
	    continue;
	  }
	  if (!(deps[i])->open()) {
	    ret = false;
	    break;
	  }
	}
      }
      
      if (ret) ret = lib.open();
      
      if (!ret) {
	for (i = i+1; i < ndeps; ++i)  {
	  if (deps[i] != NULL) deps[i]->close();
	}
      }

    } else {
      if (ndeps > 0) 
	pandora_warning(lib.name << ": cannot load dependency libraries");
      ret = false;
    }
  }

  lock = false;

  if (ret) ++count;
  
#if DYN_LIBRARY_DEBUG
  pandora_debug("open " << lib.name << " (" << count << ")");
#endif
  return ret;
}

bool DynLibrary::close(void) 
{
  bool ret = true;
  --count;
#if DYN_LIBRARY_DEBUG
  pandora_debug("close " << lib.name << " (" << count << ")");
#endif
  if (count == 0) {
    ret = lib.close() & ret;
    for (int i = 0; i < ndeps; ++i)  {
      ret = (deps[i] == NULL || (deps[i])->close()) & ret;
    }
  }
  return ret;
}

void *DynLibrary::sym(const char *name) 
{
  pandora_assert(count > 0);
  return lib.sym(name);
}

int DynLibrary::getDeps(text *d, int max_deps)
{
  if (d == NULL) return 0;
  int m = pandora_min(ndeps, max_deps);
  for (int i = 0; i < m; ++i) d[i] = dep_names[i];
  return m;
}

bool DynLibrary::dlrep_t::open(void) 
{
  if (name.isNull()) return false;
  handle = lt_dlopen(name.data());
  if (handle == NULL) {
    pandora_warning(name <<": " << lt_dlerror());
    return false;
  }

#if DLREP_DEBUG
  pandora_debug(">>> dlopen  '" << name << "'");
#endif
  return true;
}

bool DynLibrary::dlrep_t::close(void) 
{
  if (handle == NULL) return false;

#if DLREP_DEBUG
  pandora_debug("<<< dlclose '" << name << "'");
#endif

#if 1
  if (lt_dlclose(handle) != 0) {
    pandora_warning(name <<": " << lt_dlerror());
    return false;
  }
#endif

  handle = NULL;
  return true;
}

lt_ptr DynLibrary::dlrep_t::sym(const char *symbol) 
{
  if (handle == NULL) return NULL;
#if DLREP_DEBUG > 1
  pandora_debug("=== dlsym   '" << symbol << "'");
#endif
  lt_ptr func = lt_dlsym(handle, symbol);
  if (func == NULL) pandora_warning(lt_dlerror());
  return func;
}

void DynLibrary::print(ostream *f)
{
  if (f == NULL) return;
  *f << "[library] " << id << " " << version << " " << lib.name << "\n";
  if (ndeps > 0) {
    *f << "[dependency] " << id << " ";
    for (int i = 0; i < ndeps; ++i) *f << dep_names[i] << " ";
    *f << "\n";
  }
}
