/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#define DYN_BINDINGS_DEBUG 	0
#define DYN_BINDINGS_SYM_DEBUG 	0

#include <libpandora/dynbindings.h>
#include <libpandora/dynsymbol.h>
#include <libpandora/dynlibrary.h>
#include <libpandora/storage.h>

Mutex DynBindings::mx;

DynBindings::~DynBindings(void)
{
#if 1
  DynSymbol *dsym;
  valuesDo(symUsed, dsym) {
    CERR(LOG_DEBUG) << "remaining symbol: " << *dsym;
    dsym->flush();
  }
  symUsed.clear();
#else
  pandora_assert(symUsed.size() == 0);
#endif

#if 1
  symbols.cleanup();
  libraries.cleanup();

  pandora_assert(symbols.size() == 0);
  pandora_assert(libraries.size() == 0);
#else
  DynSymbol *sym = NULL;
  valuesDo(symbols, sym) __DELETE(sym);
  symbols.clear();

  DynLibrary *lib = NULL;
  valuesDo(libraries, lib) __DELETE(lib);
  libraries.clear();
#endif
}

void *DynBindings::useSymbol(const text &id)
{
  DynSymbol *symbol = symbols.atOrNil(id);
  if (symbol == NULL) {
    pandora_warning("undefined symbol: " << id);
    return NULL;
  }
  
  void *sym = symbol->resolve();

  if ((sym != NULL)
      && (!symUsed.includesKey(sym)))
    symUsed.atPut(sym, symbol);

#if DYN_BINDINGS_SYM_DEBUG
  pandora_debug("using symbol         @" << sym 
	       << " (" << id << ")");
#endif

  return sym;
}

void DynBindings::releaseSymbol(void *sym, bool force)
{
  DynSymbol *symbol = symUsed.atOrNil(sym);
  if (symbol == NULL) {
    pandora_warning("releasing undefined symbol @" << sym);
    return;
  }

#if DYN_BINDINGS_SYM_DEBUG
  pandora_debug("releasing symbol @" << sym 
	       << " (" << symbol->getName() << ")");
#endif


  if (force)	symbol->flush();
  else		symbol->release();

  if (!symbol->inUse()) {
#if DYN_BINDINGS_SYM_DEBUG
    pandora_debug("unused symbol    @" << sym 
		  << " (" << symbol->getName() << ")");
#endif
    symUsed.removeKey(sym);
  }

  if (symbol->useless()) {
#if DYN_BINDINGS_SYM_DEBUG
    pandora_debug("useless symbol   @" << sym 
		  << " (" << symbol->getName() << ")");
#endif
    __DELETE(symbol);
  }
}

bool DynBindings::registerLibrary(const text &id, long version,
				  const text &loc)
{
  DynLibrary *lib = libraries.atOrNil(id);
  if (lib != NULL) {
    int lvers = lib->getVersion();
    if (lvers >= 0) {
      if (lvers >= version) return true;
      lib->obsolete();
      if (lib->useless()) __DELETE(lib);
      libraries.removeKey(id);
    } 
  } else {
    lib = newDynLib(id);
  }

  bool status = (lib->setVersion(version) && lib->setLocalisation(loc));

#if DYN_BINDINGS_DEBUG
  pandora_debug("registered lib: " << id);
#endif
  storage->save(Storage::library, id, lib);
  return status;
}

bool DynBindings::registerDependencies(const text &id, const text *d, int n)
{
  DynLibrary *lib = libraries.atOrNil(id);
  if (lib == NULL) lib = newDynLib(id);

  bool status = lib->setDeps(d, n);
#if DYN_BINDINGS_DEBUG
  pandora_debug("registered dependencies for: " << id);
#endif
  storage->save(Storage::library, id, lib);
  return status;
}

bool DynBindings::registerSymbol(const text &id, const text &l)
{
  DynLibrary *lib = libraries.atOrNil(l);
  if (lib == NULL) {
    pandora_warning("uknown library: " << l);
    return false;
  }

  DynSymbol *sym = symbols.atOrNil(id);
  if (sym != NULL) {
    sym->obsolete();
    if (sym->useless()) __DELETE(sym);
  }

  DynSymbol *new_sym = new DynSymbol(id, lib);
  symbols.atPut(id, new_sym);

#if DYN_BINDINGS_DEBUG
  pandora_debug("registered symbol: " << id << "(" << l << ")" );
#endif  
  storage->save(Storage::symbol, id, new_sym);
  return true;
}

int DynBindings::listSymbols(text *syms, int max_syms)
{
  if (syms == NULL) return -1;
  if (max_syms <= 0) return 0;
  int i = 0;
  text *tmp = syms;


  keysDo(symbols, *tmp) {
    if (i > max_syms) break;
    ++i;
    ++tmp;
  }

  return i;
}

int DynBindings::listLibraries(text *libs, int max_libs)
{
  if (libs == NULL) return -1;
  if (max_libs <= 0) return 0;
  int i = 0;
  text *tmp = libs;


  keysDo(libraries, *tmp) {
    if (i > max_libs) break;
    ++i;
    ++tmp;
  }

  return i;
}

bool DynBindings::dump(ostream *f)
{
  DynLibrary *lib = NULL;
  valuesDo(libraries, lib) lib->print(f);

  DynSymbol *sym = NULL;
  valuesDo(symbols, sym)   sym->print(f);
  return true;
}

DynLibrary *DynBindings::newDynLib(const text &id)
{
  pandora_assert(!libraries.includesKey(id));
  DynLibrary *lib = new DynLibrary(id, this);
  libraries.atPut(id, lib);
  return lib;
}
