%{
#include <libpandora/global.h>

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <libpandora/util.h>
#include <libpandora/stackentry.h>
#include <libpandora/stackfactory.h>
#include <libpandora/pandoraentry.h>
#include <libpandora/multivalue.h>

#define YYPARSE_PARAM	parm
#define YYLEX_PARAM	parm

#define pe		((PandoraEntry *)parm)
#define YYDEBUG 1

#define YYERROR_VERBOSE
static void yyerror(char *);

#define YY_DECL int yylex(union yystype *yylval, struct yyltype *yylloc, \
			  void *parm)
YY_DECL;

static char *err_type;
static int err_line = 0;
static void parse_error(PandoraEntry *, int);
static void new_stack(PandoraEntry *, char *);
static void new_component(PandoraEntry *, char *, char *);
static void new_macro(PandoraEntry *, char *);
static void new_option(PandoraEntry *, char *, char *, MultiValue *);

static void add_component(PandoraEntry *, int);
static void add_option(PandoraEntry *);

static void add_param(PandoraEntry *, char *);
static void add_value(PandoraEntry *, MultiValue *);

%}
%pure_parser

%union yystype {
  int 		 i;
  bool 		 b;
  float 	 f;
  char 		*s;
  MultiValue    *mv;
}


%type <i>	ctors
%type <s>	alias alias_def
%type <mv>	value value_def

%token <s> TOK LINE OPTION COMP STACK MACRO ALIAS
%token <i> NUM
%token <b> BOOL
%token <f> FLOAT

%%

file:		/* empty */ 	{ YYACCEPT; }
	|	entry		{ YYACCEPT; }
	;

entry:		stack		
	|	macro	
	|	component	
	|	option		
	|	error		{ parse_error(pe, @1.first_line); yyerrok; }
	;

param_list:	/* empty */
	|	'(' param_list0 ')'
	;

param_list0:	param			
	|	param_list0 ',' param
	;

value_list:	/* empty */
	|	'[' value_list0 ']'
	;

value_list0:	value_def 			{ add_value(pe, $1); }
	|	value_list0 ',' value_def 	{ add_value(pe, $3); }
	;

comp_list:	'{' comp_list0 '}'
	;

comp_list0:	/* empty */		
	|	comp_list0 com ctors	{ add_component(pe, $3); }
	;

option_list:	/* empty */			
	|	'[' option_list0 ']'		
	;

option_list0:	option			{ add_option(pe); }
	|	option_list0 ',' option	{ add_option(pe); }
	;

stack:		stack_def				
	|	stack_def param_list comp_list	
	;

stack_def:	STACK			{ new_stack(pe, $1); }
	;

macro:		macro_def value_list
	;

macro_def:	MACRO			{ new_macro(pe, $1); }
	;

component:	component_def option_list
	;

component_def:	COMP alias		{ new_component(pe, $1, $2); }
	;

option:		option_def
	;

option_def:	OPTION alias value	{ new_option(pe, $1, $2, $3); }
	;

alias:		/* empty */	{ $$ = NULL; }
	|	alias_def	{ $$ = $1; }
	;

alias_def:	ALIAS		{ $$ = $1; }
	;

value:		/* empty */	{ $$ = NULL; }
	|	'=' value_def	{ $$ = $2; }
	;

value_def:	NUM 		{ $$ = new MultiValue($1); }
	|	BOOL 		{ $$ = new MultiValue($1); }
	|	FLOAT 		{ $$ = new MultiValue($1); }
	|	TOK 		{ $$ = new MultiValue($1); __FREE($1); }
	|	LINE 		{ $$ = new MultiValue($1); __FREE($1); }
	;

ctors:		/* empty */ 	{$$ = 0; }
	|	ctors '<' 	{$$ |= COMP_DEMUX;  INC_COMP_DEMUX_FACT($$); }
	|	ctors '>' 	{$$ |= COMP_PREMUX; INC_COMP_DEMUX_FACT($$); }
	|	ctors '(' 	{$$ |= COMP_SWITCH; INC_COMP_SWITCH_FACT($$); }
	|	ctors ')' 	{$$ |= COMP_LAST;   INC_COMP_SWITCH_FACT($$); }
	|	ctors '|' 	{$$ |= COMP_INTER;  INC_COMP_SWITCH_FACT($$); }
	|	ctors '-' 	{$$ |= COMP_SIMPLE; }
	;

com:		component
	|	macro
	;

param:		param_def
	;

param_def:	TOK		{ add_param(pe, $1); }
	;
%%

void new_stack(PandoraEntry *pentry, char *name)
{
  if (name == NULL) return;
  StackEntry *se = new StackEntry(name);
  if (*name == '*') se->noRun();  
  pentry->setStack(se); 
  __FREE(name); 
}

void new_component(PandoraEntry *pentry, char *name, char *alias)
{
  CompEntry *ce = NULL;
  bool param;
  if (name == NULL) goto out;
  param = (*name == '!');
  ce = new CompEntry(CompEntry::component, (param ? name+1 : name));
  if (param) ce->setParam();
  ce->type = 0;
  pentry->setComponent(ce);
  if (alias != NULL) ce->setAlias(alias);

  __FREE(name); 
 out:
  __FREE(alias);
}

void new_macro(PandoraEntry *pentry, char *name)
{
  if (name == NULL) return;
  bool param = (*name == '!');
  CompEntry *ce  = new CompEntry(CompEntry::macro, (param ? name+1 : name));
  if (param) ce->setParam();
  ce->type = COMP_MACRO;
  pentry->setComponent(ce);

  __FREE(name); 
}

void new_option(PandoraEntry *pentry, char *name, 
		char *alias, MultiValue *mv)
{
  OptionEntry *oe = NULL;
  bool param;
  char *id;
  if (name == NULL) goto out;
  param = (*name == '!');
  id = (param ? name+1 : name);
  oe = new OptionEntry(id);
  if (param) oe->setParam();
  oe->setAlias((alias != NULL) ? alias : id);
  if (mv != NULL) oe->set(*mv);
  pentry->setOption(oe); 
  
  __FREE(name); 
 out:
  __FREE(alias); 
  __DELETE(mv); 
}

void add_component(PandoraEntry *pentry, int type)
{
  StackEntry *se = pentry->stack();
  CompEntry *ce  = pentry->takeComponent();
  if (ce == NULL) return;
  if (se == NULL) goto out;

  ce->type |= type;
  se->pushComponent(*ce); 

 out:
  __DELETE(ce);
}

void add_option(PandoraEntry *pentry)
{
  StackEntry *se = pentry->stack();
  CompEntry *ce = pentry->component();
  OptionEntry *oe  = pentry->takeOption();
  if (oe == NULL) return;
  if (ce == NULL) goto out;

  //if (se != NULL) pandora_debug(se->id << "." << ce->id << "." << oe->id);
  ce->pushOption(*oe);
  if (oe->isParam()) ce->setOpParam();

 out:
  __DELETE(oe);
}

void add_param(PandoraEntry *pentry, char *name)
{
  StackEntry *se = pentry->stack();
  if (se == NULL) goto out;
  if (name != NULL) se->pushParam(name);
  
 out:
  __FREE(name);

}

void add_value(PandoraEntry *pentry, MultiValue *mv)
{
  CompEntry *ce = pentry->component();
  OptionEntry oe;
  if (ce == NULL) goto out;
  if (mv == NULL) return;
  oe.set(*mv);
  ce->pushOption(oe);
  
 out:
  __DELETE(mv);
}

static void yyerror(char *msg)
{
  // pandora_warning(msg);
}

static void parse_error(PandoraEntry *pentry, int line)
{
  pandora_notice("parse error at line " << line); 
  if (pentry->option() != NULL) {
    pandora_notice("near option: " << pentry->option()->id);
  } else if (pentry->component() != NULL) {
    pandora_notice("near component: " << pentry->component()->id);
  } else if (pentry->stack() != NULL) {
    pandora_notice("near stack: " << pentry->stack()->id);
  } else {
    pandora_notice("at top level");
  }
  pentry->reset();  
}
