/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/string.h>
	   }

#include <iomanip>
#include <libpandora/dispatcher.h>
#include <libpandora/pandora.h>
#include <libpandora/component.h>
#include <libpandora/inputcomponent.h>
#include <libpandora/packet.h>

#define DISP_DEBUG 1

#if DISP_DEBUG
#define disp_status(str)						\
  pandora_info(getID() << "[" << handle << "]: " << str)
#else
#define disp_status(str)
#endif

Dispatcher::Dispatcher(void) 
  : handle(NIL_STACK_HANDLE), 
    compStack(NULL), firstComponent(NULL), startComponent(NULL), 
    suspended(false), force(false), deleted(false), root(true),
    refCount(0)
{
  pandora_assert(pandora != NULL);
}

Dispatcher::Dispatcher(stack_handle_t h) 
  : handle(h),
    compStack(NULL), firstComponent(NULL), startComponent(NULL), 
    suspended(false), force(false), deleted(false), root(true),
    refCount(0)
{
  pandora_assert(pandora != NULL);
}

Dispatcher::~Dispatcher(void) 
{
  deleted = true;

  cleanup();
  startComponent = NULL;
  firstComponent = NULL;
}


bool Dispatcher::init(StackEntry *se, MultiValue *vals, int nbvals) 
{
  pandora_assert(se != NULL);

  if (compStack != NULL) cleanup();

  compStack = new StackDesc();
  if (!compStack->init(handle, se, vals, nbvals)) {
    __DELETE(compStack);
    return false;
  }

  if (!compStack->isRunable()) return comp_init();

  return true;
}


bool Dispatcher::start(bool threaded)
{
  pandora_assert(compStack != NULL);
  if (!compStack->isRunable()) return true;

  bool ret = false;
  mx.lock();
  if (!isRunning()) {
    if ((startComponent == NULL) 
	&& !comp_init())
      goto bad;
    pandora_assert(startComponent != NULL);
    
    int fd = startComponent->init();
    if (fd == ERROR_FILENO) goto bad;

    force = false;
    disp_status("started (fd=" << fd << ")" );

    if (threaded || fd == THREADED_FILENO) {
      run();
    } else {
      comp_start();
      finished();
    }
    ret = true;
  }

 bad:
  mx.unlock();
  
  return ret;
}

bool Dispatcher::stop(void)
{
  bool ret = false;
  mx.lock();
  cv.lock();
  if (suspended) {
    suspended = false;
    force = true;    
    cv.broadcast();
  }
  cv.unlock();

  if (isRunning()) {
    pandora_assert(startComponent != NULL);
    disp_status("stopped");
    startComponent->stop(true);
    join();
    ret = true;
  }

  ret = flush();
  mx.unlock();

  return ret;
}

bool Dispatcher::stop_async(void)
{
  if (startComponent == NULL) return false;
  disp_status("signaled");
  startComponent->stop(true);

  return true;
}

bool Dispatcher::suspend(void)
{
  bool ret = false;

  mx.lock();
  cv.lock();
  if (isRunning() && !suspended) {
    suspended = true;
    disp_status("suspended");
    pandora_assert(startComponent != NULL);
    startComponent->stop(false);
    cv.wait();
    ret = true;
  }

  cv.unlock();
  mx.unlock();

  return ret;
}

bool Dispatcher::resume(void)
{
  bool ret = false;

  mx.lock();
  cv.lock();
  if (isRunning() && suspended) {
    disp_status("resumed");
    suspended = false;
    cv.broadcast();
    ret = true;
  }

  cv.unlock();
  mx.unlock();

  return ret;
}

static void printComp(Component *comp, void *data)
{
  int *n = ((int *)data);
  ++(*n);
  cout << *n << ":\t[" << comp << "] #" << comp->ind 
       << " (" << comp->nbBranches() << ")" << endl;
}

bool Dispatcher::flush(void) 
{
  if (firstComponent != NULL) {

    if (extUse() && !deleted) {
      disp_status("NOT flushed (" << firstComponent->nb_refs() << ")");
      return false;
    }
    
    disp_status("flushed");

    compStack->setCacheFlush();
    Component::clean(firstComponent);

    if (startComponent != NULL) {
      pandora_warning("zombie stack: " << getID() 
      		      << ", trying to collect");
      startComponent->refCount = 1;
      Component::clean(startComponent);
    }
    firstComponent = NULL;
  }
  return true;
}

bool Dispatcher::cleanup(void) 
{
  if (compStack != NULL) {
    if (!stop()) return false;
    disp_status("cleaned up");
    __DELETE(compStack);    
    //pandora_debug("[clean up completed]");    
  }
  handle = NIL_STACK_HANDLE;
  return true;
}

bool Dispatcher::refresh(StackEntry *se, MultiValue *vals, int nbvals) 
{
  disp_status("refreshed");

  suspend();

  StackDesc *old_stack = compStack;

  compStack = new StackDesc();
  if (!compStack->init(handle, se, vals, nbvals)) {
    pandora_warning("failed init of new stack");
    __DELETE(compStack);
    compStack = old_stack;
    resume();
    return false;
  }

  if (!compStack->refresh(old_stack)) {
    pandora_warning("failed refresh");
    firstComponent = NULL;
    startComponent = NULL;
  }

  __DELETE(old_stack);

  if (compStack->isRunable()) {
    resume();
  } else {
    stop();
  }

  return true;
}

void Dispatcher::main(void)
{
  comp_start();
  finished();
}


bool Dispatcher::setOption(const text &comp, const text &op,
			   const MultiValue &mv) 
{
  suspend();
  bool status = compStack->setOption(comp, op, mv);
  resume();
  return status;
}

void Dispatcher::finished(void)
{
  pandora_assert(startComponent != NULL);
  startComponent->finish();
  disp_status("finished");
  pandora_assert(pandora != NULL);
  pandora->finished(handle);
}

bool Dispatcher::comp_init(void) 
{
  //pandora_debug(getID() << "[" << handle << "]: init0 (" << refCount << ")");
  if (firstComponent == NULL) {
    compStack->setFirstComponent(&firstComponent);
    if (firstComponent == NULL) return false;
  }

  ++refCount;
  //pandora_debug(getID() << "[" << handle << "]: init1 (" << refCount << ")");

  if (!compStack->isRunable()) {
    return true;
  }
  
  if (!firstComponent->input) {
    release();
    return false;
  }
  
  startComponent = (InputComponent *)firstComponent;
  firstComponent->referrer = (Component **)&startComponent;

  return true;
}
  
bool Dispatcher::use(void)
{ 
  //pandora_debug(getID() << "[" << handle << "]: use (" << refCount << ")");
  return comp_init();
}

bool Dispatcher::release(void)
{
  --refCount;
  //pandora_debug(getID() <<"[" << handle <<"]: release (" <<refCount <<")");
  pandora_assert(refCount >= 0);
  if (refCount == 0) {
    root = true;
    return cleanup();
  } else {
    return false;
  }
}

bool Dispatcher::push(Packet *pkt)
{
#if 1
  if (firstComponent == NULL) {
    pandora_error(getID() << "[" << handle << "]: "
		    << "push failed: first component not set");
    cleanPacket(pkt);
    return false;
  }
#endif
  mx.lock();
  firstComponent->add(pkt);
  mx.unlock();

  return true;
}

bool Dispatcher::comp_start(void)
{
  if (startComponent == NULL) {
    pandora_warning("start failed: startComponent not set");
    return false;
  }

  cv.lock();
  for (;;) {
    cv.unlock();
    startComponent->start();
    cv.lock();
    startComponent->reset();

    if (!suspended) {
      if (root) break;
      else continue;
    }
    
    cv.broadcast();
    cv.wait();

    if (force) break;
  } 
  force = false;
  cv.unlock();
  
  
  return true;
}
