/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef DEMAP_H
#define DEMAP_H

#define DEMAP_DEBUG 0

#include <libpandora/global.h>

#include <libpandora/map.h>

#define GENERIC	template<class Key, class Val> inline
#define DEMAP	DEMap<Key, Val>

#if DEMAP_DEBUG
#define SIZE_CHECK()	pandora_assert(kvmap.size() == vkmap.size())
#else
#define SIZE_CHECK()
#endif

template<class Key, class Val>
class DEMap
{
public:
  Map<Key, Val> kvmap;
  Map<Val, Key> vkmap;

public:
  DEMap(void) : kvmap(Key()), vkmap(Val()) { }
  virtual ~DEMap(void) { }

public:
  Val	&valAtOrNil(const Key &key) const;
  Val	&valAt(const Key &key) const;
  Key	&keyAtOrNil(const Val &val) const;
  Key	&keyAt(const Val &val) const;
  void	atPut(const Key &key, const Val &val);
  bool	includesKey(const Key &key) const;
  bool	includesVal(const Val &val) const;
  Val	removeKey(const Key &key);
  Key	removeVal(const Val &val);
  long  size(void) const			{ return kvmap.size(); }
};


//
// INLINE IMPLEMENTATION
//

GENERIC Val &DEMAP::valAtOrNil(const Key &key) const
{
  SIZE_CHECK();
  return kvmap.atOrNil(key);
}

GENERIC Val &DEMAP::valAt(const Key &key) const
{
  SIZE_CHECK();
  return kvmap.at(key);
}

GENERIC Key &DEMAP::keyAtOrNil(const Val &val) const
{
  SIZE_CHECK();
  return vkmap.atOrNil(val);
}

GENERIC Key &DEMAP::keyAt(const Val &val) const
{
  SIZE_CHECK();
  return vkmap.at(val);
}

GENERIC void DEMAP::atPut(const Key &key, const Val &val)
{
  SIZE_CHECK();
  vkmap.atPut(val, key);
  kvmap.atPut(key, val);
}

GENERIC bool DEMAP::includesKey(const Key &key) const
{
  SIZE_CHECK();
  return kvmap.includesKey(key);
}

GENERIC bool DEMAP::includesVal(const Val &val) const
{
  SIZE_CHECK();
  return vkmap.includesKey(val);
}

GENERIC Val  DEMAP::removeKey(const Key &key)
{
  SIZE_CHECK();
  Val val = kvmap.removeKey(key);
  (void) vkmap.removeKey(val);
  return val;
}

GENERIC Key  DEMAP::removeVal(const Val &val)
{
  SIZE_CHECK();
  Key key = vkmap.removeKey(val);
  (void) kvmap.removeKey(key);
  return key;
}

#undef GENERIC
#undef DEMAP

//#define deKeysDo(DEMAP, KEY)   keysDo((DEMAP).kvmap, KEY)

//#define deValuesDo(DEMAP, VAL) keysDo((DEMAP).vkmap, VAL)

//#define deKeysValuesDo(DEMAP, KEY, VAL) keysValuesDo((DEMAP).kvmap, KEY, VAL)

//#define deValuesKeysDo(DEMAP, VAL, KEY) keysValuesDo((DEMAP).vkmap, VAL, KEY)

#endif /* DEMAP_H */
