/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#ifndef COMPONENT_H
#define COMPONENT_H

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/time.h>
	   }

#include <libpandora/packet.h>
#include <libpandora/option_desc.h>
#include <libpandora/optionentry.h>
#include <libpandora/map.h>
#include <libpandora/util.h>
#include <libpandora/error.h>
#include <libpandora/dynloader.h>

class Pandora;
class Packet;
class PandoraKey;
class Component;
class StackDesc;
class TimerSupervisor;
class sched_task_t;

#define UNLINK_FORWARD		(1 << 0)
#define UNLINK_BACKWARD		(1 << 1)
#define UNLINK_BOTH		(UNLINK_FORWARD | UNLINK_BACKWARD)

#define COMP_SYMBOL_PREFIX	__pandora_component_

#define component_init(cl, vers)					\
  static symbol_id_t component_id;					\
  virtual symbol_id_t getID(void) { return this->component_id; }	\
  virtual const char *getName(void) { return #cl; }			\
  const static int __version = vers

#define component_export(cl, in, out)					\
  extern "C" Component *pandora_make_sym(component, cl)(void)		\
    { return new cl(); }						\
  symbol_id_t cl::component_id = DynLoader::make_id(#cl);		\
  static const bool __##cl##_pkt_types =				\
    ((add_pkt_map(&Component::pkt_in, cl::component_id, #in)),		\
     (add_pkt_map(&Component::pkt_out, cl::component_id, #out)),	\
     true);								\
  static const bool __##cl##_name__ =					\
    ((Component::names.atPut(cl::component_id, #cl)),			\
     true);								\
  EXPORT_SYMBOL(component, cl)

#define exportHandler(cl, pkt)				\
   bool __##pkt##handler__(cl *comp, Packet *p) {	\
    return comp->add(static_cast<pkt *>(p));		\
  }

#define registerHandler(cl, pkt)				\
  extern bool __##pkt##handler__(cl *comp, Packet *p);		\
  __registerHandler(pkt::packet_id, (void *)&__##pkt##handler__)

#define unregisterHandler(pkt)				\
  __unregisterHandler(pkt::packet_id)

#define registerComplexOption(type, name, var)	\
  registerOption(name, var, &type::func, false)

#define registerStaticOption(type, name, var) 	\
  registerOption(name, var, &type::func, true)

#define return_clean(pkt)	do { cleanPacket(pkt); return false; } while(0)
#define return_push(pkt)	do { push(pkt); return false; } while(0)
#define return_clean_fini(pkt)	do { cleanPacket(pkt); return true; } while(0)
#define return_push_fini(pkt)	do { push(pkt); return true; } while(0)

extern void add_pkt_map(Set<symbol_id_t> *, const char *);

typedef	void (comp_func)(Component *, void *);

class Component {
public:
  static Map<symbol_id_t, text> names;
  static MultiMap<symbol_id_t, symbol_id_t> pkt_in, pkt_out;

public:
  int 			 		ind;
  int			 		branch;
  time_t 		 		lastTime;
  int 			 		rank;
  bool 			 		input;
  bool 			 		flushed;
#ifndef NDEBUG
  static int 		 		created;
  static int		 		alive;
#endif

protected:
  int 			 		timeout;
  int			 		maxpkt;
  bool 			 		resched;
  bool 			 		beat;
  bool 			 		cacheable;
  bool					demuxed;

  Component 		 		*nextComponent;
  Component				*prevComponent;
  Component				*prevDemux;
  Component				*prevCalling;
  Component 				**referrer;
  Component				**refdComponent;
  Map<int, Component **> 		nextComponents;
  Map<symbol_id_t, void *> 		handlers;

  StackDesc 				*compStack;
  TimerSupervisor 			*timerSupervisor;
  sched_task_t				*timeout_sched;
  
private:
  Map<text, option_desc_t *> 		options;
  bool					opclean;
  int 			 		refCount;
  int 			 		npkt;

public:  
  inline Component(void);
  virtual ~Component(void);

private:
  Component(const Component&) 			{ abort(); }
  Component& operator= (const Component&)	{ abort(); return *this; }

public:  
  virtual bool add(Packet *pkt);
  inline void __registerHandler(symbol_id_t type, void *handler);
  inline void __unregisterHandler(symbol_id_t type);

  virtual bool prepare(void) 			{ return true; }
  virtual bool notify(Component *) 		{ return false; }
  inline  void _cleanup(void);
  virtual void cleanup(void) 			{ }

  virtual bool push(Packet *pkt);
  inline bool push(Packet *pkt, int b);
  bool push(Packet *, stack_handle_t);
  void setNextComponent(Component **ptr, int);

  virtual bool pull(Packet *pkt) 		{ return false; }
  void reply(Packet *pkt);

  inline TimerSupervisor *getTimerSupervisor(void) { return timerSupervisor; }
  void touch(void);
  void schedule(void);
  void reschedule(int);
  void remove(void);
  static void expired(Component *, time_t);  
  
  static void clean(Component *);
  inline Component *setUpBranch(int b);
  inline void closeBranch(int b);
  inline int nbBranches(void) 		      { return nextComponents.size(); }
  inline void cleanBranches(void);
  
  inline bool inUse(void) 			{ return (refCount > 0); }
  inline int  nb_refs(void) 			{ return refCount; }
  inline void refer(void) 			{ ++refCount; }
  inline void release(void) 			{ --refCount; }


  int getNbOptions(void) 			{ return options.size(); }
  int listOptions(text *ops, int max_ops);
  bool getOption(const text &op, MultiValue *mv);
  bool setOption(const text &op, const MultiValue &mv);
  inline void cleanOptions(void);
  static bool getOptionDefault(const text &comp, const text &op, 
			       MultiValue *mv);
  static int listOptions(const text &, text *, int);
  virtual bool query(const text &, MultiValue *);
  virtual void setup(void) 			{ }

  const text getStackID(void);
  stack_handle_t getStackHandle(void);

  void unlink(char);
  static void apply(Component *, comp_func *, void *);
  static Component *locateComponent(Component *start, const text &id);
  static void print(Component *, void *);
  
  void quit(void);

  static const char *getPrefix(void) { return cpp_quote(COMP_SYMBOL_PREFIX); }

protected:
  template <class _t>
  inline void registerOption(const char *, _t *var);
  template <class _t>  
  inline void registerOption(const char *, _t *var, 
			     fpcov *ctor, bool _static);
  bool push(Packet *pkt, Component *comp);

  const static symbol_id_t component_id = 0;
  virtual symbol_id_t getID(void) { return 0; }
  virtual const char *getName(void) { return "Component"; }
  const static int __version = 1;

private:
  static void _apply(Component *, comp_func *, void *);

  inline bool setOptions(const OptionEntry *opts, int cnt);
  bool setOption(const OptionEntry *opt);
  bool setStaticOption(OptionEntry *opt);

  friend class Dispatcher;
  friend class StackDesc;
  friend class CompCache;
  friend class CompRefStore;
};


Component::Component(void) : 
  ind(-1), branch(0),
  lastTime(0), rank(-1), input(false), flushed(false),
  timeout(0), maxpkt(-1),
  resched(true), beat(true), cacheable(true), demuxed(false),
  nextComponent(NULL), prevComponent(NULL), prevDemux(NULL), prevCalling(NULL),
  referrer(NULL), refdComponent(NULL),
  compStack(NULL), timerSupervisor(NULL), timeout_sched(NULL),
  opclean(false), refCount(0), npkt(0)
{
  registerOption("timeout", &timeout);
  registerOption("maxpkt", &maxpkt);
  registerOption("resched", &resched);
  registerOption("cache", &cacheable);
#ifndef NDEBUG
  rank = created;
  ++created; ++alive;
#endif
}

void Component::_cleanup(void)
{
  remove();
  cleanOptions();
  cleanup();
  cleanBranches();
}

void Component::__registerHandler(symbol_id_t type, void *handler)
{
  handlers.atPut(type, handler);
}

void Component::__unregisterHandler(symbol_id_t type)
{
  if (handlers.includesKey(type))
    handlers.removeKey(type);
}

bool Component::setOptions(const OptionEntry *opt, int cnt) 
{
  if (opt == NULL) return false;
  bool status = true;
  for (int i = 0; i < cnt; ++i) status = setOption(&opt[i]) & status;
  //setup();
  return status;
}

void Component::cleanOptions(void)
{
  if (!opclean) return;
  option_desc_t *opdesc;
  valuesDo(options, opdesc) opdesc->cleanup();
  opclean = false;
}

template <class _t>
void Component::registerOption(const char *name, _t *var) 
{
  options.atPut(text(name), new option_desc_t(var));
}

template <class _t>
void Component::registerOption(const char *name, _t *var, 
			       fpcov *ctor, bool _static) 
{
  option_desc_t *opdesc = new option_desc_t(var);
  opdesc->init(ctor, this);
  if (_static) opdesc->setStatic();
  options.atPut(text(name), opdesc);
}

bool Component::push(Packet *pkt, int b)
{
  //pandora_debug("push -> #" << b);

  if (b == 0) {
    return push(pkt);
  } else {
    Component *comp = setUpBranch(b);
    return push(pkt, comp);
  }
}

Component *Component::setUpBranch(int branch) 
{
  Component **cc = nextComponents.atOrNil(branch);
  if (cc == NULL) {
    cc = new Component*;
    *cc = NULL;
    nextComponents.atPut(branch, cc);
#if 0
    ++created_branches;
    ++alive_branches;
#endif
  }
  if (*cc == NULL) {
    setNextComponent(cc, branch);
    if (*cc ==  NULL) {
      closeBranch(branch);
      return NULL;
    }
  }
  return *cc;
}

void Component::closeBranch(int b) 
{
  pandora_assert(b != 0);
  pandora_assert(nextComponents.includesKey(b));
  Component **cc = nextComponents.removeKey(b);
  __DELETE(cc);
#if 0
  --alive_branches;
#endif
}

void Component::cleanBranches(void) 
{
  int b = 0;
  Component **cc = NULL;
  while(nbBranches() > 0) {
    keysValuesDo(nextComponents, b, cc) {
      pandora_assert(cc != NULL);
      if (*cc != NULL) {
	clean(*cc);
      } else {
	nextComponents.removeKey(b);
	__DELETE(cc);
      }
    }
  }
#if 0
  if (alive_branches != 0) {
    pandora_warning("branches have not been collected: comp [" << this << "] #"
		    << rank << ": " << alive_branches 
		    << " / " << created_branches);
  }
#endif
}

#endif /* COMPONENT_H */
