/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

#include <libpandora/component.h>
#include <libpandora/stackdesc.h>
#include <libpandora/timersupervisor.h>
#include <libpandora/pandora.h>
#include <libpandora/packet.h>
#include <libpandora/sched_task.h>

#define COMP_TDEBUG	0

Map<symbol_id_t, text> Component::names;
MultiMap<symbol_id_t, symbol_id_t> Component::pkt_in, Component::pkt_out;

#ifndef NDEBUG
int Component::created = 0;
int Component::alive = 0;
#endif

Component::~Component(void) 
{ 
  option_desc_t *opdesc;
  valuesDo(options, opdesc) {
    opdesc->cleanup();
    __DELETE(opdesc);
  }
  options.clear();

  pandora_assert(nextComponents.size() == 0);
  __DELETE(timeout_sched);
  pandora_assert(refCount == 0);

#ifndef NDEBUG
  --alive; 
#endif
}

bool Component::push(Packet *pkt, stack_handle_t h) 
{
  if (h == NIL_STACK_HANDLE) {
    pandora_warning("linked stack died");
    cleanPacket(pkt);
    return false;
  }
  return pandora->push(h, pkt);
}

bool Component::push(Packet *pkt, Component *comp) 
{
  if (comp == NULL) {
    pandora_error("invalid stack configuration");
    cleanPacket(pkt);
    return false;
  }

#if 0
  pandora_debug(pkt->getName() << ": " 
		<< this->getName() << " -> " << comp->getName());
#endif

  time_t current_time = timerSupervisor->getTime();
  pkt->curTime = current_time;
  if (comp->timeout
      && (current_time > comp->lastTime)) {
    comp->lastTime = current_time;
    if (comp->beat && (comp->timeout > 0)) comp->schedule();
  }

  comp->prevCalling = this;
  if (comp->add(pkt)) clean(comp);
  
  return true;
}

typedef bool (pktHandler)(Component *, Packet *);

bool Component::add(Packet *pkt)
{
  void *handler = handlers.atOrNil(pkt->getID());
  if (handler != NULL) {
    return ((pktHandler *)handler)(this, pkt);
  } else {
    pandora_warning("component #" << rank << " cannot handle packets of type #"
		 << pkt->getID());
    cleanPacket(pkt);
    return false;
  }
}

void Component::setNextComponent(Component **ptr, int branch)
{
  pandora_assert(compStack != NULL);
  compStack->setNextComponent(this, ptr, branch);
}


void Component::reply(Packet *pkt)
{
  bool handled = false;

  for (Component *comp = prevCalling; 
       comp != NULL; 
       comp = comp->prevCalling) {
    handled = comp->pull(pkt);
    if (handled) break;
  }

  if (!handled) cleanPacket(pkt); 
}

void Component::clean(Component *comp)
{
  if (comp == NULL) return;

  pandora_assert( comp->compStack != NULL );

  if (!comp->prepare()) {
    comp->touch();
    if (comp->timeout > 0) {
      comp->schedule();
    } else {
      comp->reschedule(30);
    }
    return;
  }

  Component *pcomp = comp->prevComponent;

  int b = comp->branch;
  bool doClose = ((comp->demuxed) & (pcomp != NULL));

  if (pcomp != NULL && pcomp->notify(comp)) 
    return;

  comp->compStack->clean(comp);

  if (doClose) pcomp->closeBranch(b);
}

bool Component::push(Packet *pkt) 
{
  if (*refdComponent == NULL)
    setNextComponent(refdComponent, 0);
  return push(pkt, *refdComponent);
}

bool Component::setStaticOption(OptionEntry *opt) 
{
  option_desc_t *opdesc = options.atOrNil(opt->id);
  if (opdesc == NULL 
      || opdesc->var == NULL 
      || !opdesc->isStatic()
      || opdesc->isComplex()
      || opt->mv.type != MultiValue::textual) return false;

  void *ref = opdesc->getReference(opt->mv.value.s);
  *((void **)opdesc->var) = ref;
  opdesc->type = option_desc_t::pointer;

  __FREE(opt->mv.value.s);
  opt->mv.type = MultiValue::pointer;
  opt->mv.value.v = ref;  
  
  return true;
}

bool Component::setOption(const OptionEntry *opt) 
{
  pandora_assert(opt != NULL);
  if ((opt->mv).type == MultiValue::undefined) return true;

  option_desc_t *tmp = options.atOrNil(opt->id);
  if (tmp == NULL)      return false;
  if (tmp->var == NULL) return false;

  if (tmp->isComplex()
      && opt->mv.type == MultiValue::textual) {
    *((void **)tmp->var) = tmp->getReference(opt->mv.value.s);
    opclean = true;
    return true;
  }
  
  switch(tmp->type) {
  case option_desc_t::boolean:  *((bool *)tmp->var)  = opt->mv.value.b; 
    return true;
  case option_desc_t::integer:  *((int *)tmp->var)   = opt->mv.value.d; 
    return true;
  case option_desc_t::floating: *((float *)tmp->var) = opt->mv.value.f; 
    return true;
  case option_desc_t::textual:  *((char **)tmp->var) = opt->mv.value.s; 
    return true;
  case option_desc_t::pointer:  *((void **)tmp->var) = opt->mv.value.v; 
    return true;
  case option_desc_t::undefined:					 
    break;
  }
  
  return false;
}

bool Component::query(const text &, MultiValue *)
{
  return false;
}

void Component::unlink(char dir)
{
  //pandora_debug("<<< unlink comp [" << this << "]");
  if (dir & UNLINK_FORWARD) {
    nextComponent = NULL;
  }
  if (dir & UNLINK_BACKWARD) {
    if (referrer != NULL) *referrer = NULL;
    referrer = NULL;
    prevComponent = NULL;
    prevDemux = NULL;
    branch = 0;
  }
}

void Component::apply(Component *firstComponent, comp_func *func, void *udata)
{
  pandora_assert(func != NULL);
  _apply(firstComponent, func, udata);
}


void Component::_apply(Component *comp, comp_func *func, void *udata)
{
  (*func)(comp, udata);

  if (comp->nbBranches() > 0) {
    set_iterator_t iter;
    comp->nextComponents.iterInit(iter);
    Component **bcomp = NULL;
    while((bcomp = comp->nextComponents.iterNextValue(iter)) != NULL)
      _apply(*bcomp, func, udata);
  }

  Component *next = comp->nextComponent;

  if (next != NULL) _apply(next, func, udata);
}

void Component::print(Component *comp, void *)
{
  cout << "[" << comp << "] #" << comp->ind //<< "/" << comp->branch
       << " (" << comp->nbBranches() << ") ";
  if (comp->prevComponent != NULL) {
    cout << "\t<- [" << comp->prevComponent << "] #" 
	 << comp->prevComponent->ind;
  } else {
    cout << "\t\t\t";
  }
  if (comp->nextComponent != NULL) {
    cout << "\t-> [" << comp->nextComponent << "] #" 
	 << comp->nextComponent->ind;
  }
  cout << endl;
}

void Component::expired(Component *comp, time_t t)
{
  pandora_assert(comp != NULL && comp->timeout_sched != NULL);

  comp->timeout_sched->reset();

  if ((t < (comp->lastTime + comp->timeout)) && comp->resched) {
#if COMP_TDEBUG
    pandora_debug("reschd'ing comp [" << comp << "] #" << comp->rank 
		 << " @" << t 
		 << " (" << comp->lastTime << ")" );
#endif
    comp->schedule();
  }
  else {
#if COMP_TDEBUG
    pandora_debug("expiring   comp [" << comp << "] #" << comp->rank 
		 << " @" << t);
#endif
    comp->beat = true;
    clean(comp);
  }
}

void Component::schedule(void)
{
  if (timeout_sched == NULL) {
    timeout_sched = new sched_task_t((sched_task_func_t *)&expired, 
				     (void *)this);
    timeout_sched->setScheduler(timerSupervisor->getScheduler(0));
  }

  pandora_assert(timeout < 3600);

  timeout_sched->schedule(lastTime + timeout);
  beat = false;

#if COMP_TDEBUG
  pandora_debug("scheduling comp [" << this << "] #" << rank 
	       << " @" << lastTime + timeout);
#endif
}

void Component::touch(void)
{
  lastTime = timerSupervisor->getTime();
}

void Component::reschedule(int ntimeout) {
  remove();
  timeout = ntimeout;
  schedule();
}

void Component::remove(void) {
  if (timeout_sched != NULL) {
    timeout_sched->remove();
  }
  beat = true;
}

void Component::quit(void) {
  pandora_assert(pandora != NULL);
  pandora->finished(compStack->getID());
}

Component *Component::locateComponent(Component *start, const text &id)
{
  for (Component *comp = start; comp != NULL; comp = comp->nextComponent) {
    if (comp->compStack == NULL) break;
    if (comp->compStack->getCompId(comp->ind) == id) return comp;
  }
  return NULL;
}

int Component::listOptions(text *ops, int max_ops)
{
  if (ops == NULL) return -1;
  if (max_ops <= 0) return 0;
  int i = 0;
  text *tmp = ops;

  keysDo(options, *tmp) {
    if (i > max_ops) break;
    ++i;
    ++tmp;
  }

  return i;
}

bool Component::getOptionDefault(const text &comp, const text &op, 
				 MultiValue *mv)
{
  if (dynloader == NULL) return -1;
  text cid;
  if (!dynloader->lookup(cid, comp)) return false;

  Component *c = NULL;
  bool ret = false;

  void *func = dynloader->load((symbol_id_t)cid);
  if (func == NULL) goto failed;

  c = (*((Component *(*)(void))func))();
  ret = c->getOption(op, mv);

 failed:
  __DELETE(c);
  dynloader->unload(func);
  return ret;
}

bool Component::getOption(const text &op, MultiValue *mv)
{
  if (mv == NULL) return false;
  option_desc_t *tmp = options.atOrNil(op);
  if (tmp == NULL) return false;

  if (tmp->var == NULL) return false;
  if (tmp->isStatic()) return false;
  switch(tmp->type) {
  case option_desc_t::boolean:  mv->set(*((bool *)tmp->var)) ; break;
  case option_desc_t::integer:  mv->set(*((int *)tmp->var))  ; break;
  case option_desc_t::floating: mv->set(*((float *)tmp->var)); break;
  case option_desc_t::textual:  mv->set(*((char **)tmp->var)); break;
  case option_desc_t::pointer:  mv->set(*((void **)tmp->var)); break;
  case option_desc_t::undefined:			return false;
  }

  return true;
}

bool Component::setOption(const text &op, const MultiValue &mv)
{
  OptionEntry oe(op);
  oe.mv = mv;
  return setOption(&oe);
}

int Component::listOptions(const text &comp, text *ops, 
			   int max_ops)
{
  if (dynloader == NULL) return -1;
  
  text cid;
  if (!dynloader->lookup(cid, comp)) return -1;

  Component *c = NULL;
  int n = -1;

  void *func = dynloader->load((symbol_id_t)cid);
  if (func == NULL) return n;

  c = (*((Component *(*)(void))func))();
  n = c->listOptions(ops, max_ops);

  __DELETE(c);
  dynloader->unload(func);
  return n;
}

const text Component::getStackID(void)
{
  return (compStack != NULL ? compStack->getID() : text());
}

stack_handle_t Component::getStackHandle(void)
{
  return (compStack != NULL ? compStack->getHandle() : NIL_STACK_HANDLE);
}
