/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef COMMON_OPTIONS_H
#define COMMON_OPTIONS_H

#include <libpandora/global.h>

extern "C" {
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>
}

#include <iostream>
#include <fstream>

#include <libpandora/complex_option.h>
#include <libpandora/set.h>
#include <libpandora/array.h>
#include <libpandora/text.h>

class Component;
class Dispatcher;

class file_option : public complex_option {
protected:
  FILE *f;

public:
  file_option(void) : f(NULL)	 	{ }
  virtual ~file_option(void) 		{ }
  virtual void cleanup(void);
  virtual bool set(const char*) 	= 0;
  void *getReference(void) 		{ return (void *)f; }

};

class write_file_option : public file_option {
public:
  export_option(write_file_option);
  bool set(const char*);
};

class read_file_option : public file_option {
public:
  export_option(read_file_option);
  bool set(const char*);
};

class ofstream_option : public complex_option {
protected:
  fstream *s;

public:
  export_option(ofstream_option);
  ofstream_option(void) : s (NULL) 	{ }
  virtual ~ofstream_option(void) 	{ }
  virtual void cleanup(void) 		{ __DELETE(s); }
  virtual bool set(const char*); 
  void *getReference(void) 		{ return (void *)s; }
};

class stack_option : public complex_option {
protected:
  stack_handle_t h;
  Dispatcher *disp;

public:
  stack_option(void);
  virtual ~stack_option(void) 		{ }
  virtual void cleanup(void);
  virtual bool set(const char*) 	= 0;
  void *getReference(void) 		{ return (void *)disp; }
};

class fresh_stack_option : public stack_option {
public:
  export_option(fresh_stack_option);
  fresh_stack_option(void) 		{ }
  virtual ~fresh_stack_option(void) 	{ }
  virtual bool set(const char*); 
};

class running_stack_option : public stack_option {
public:
  export_option(running_stack_option);
  running_stack_option(void) 		{ }
  virtual ~running_stack_option(void) 	{ }
  virtual bool set(const char*); 
};

class inet_addr_option : public complex_option {
private:
  struct in_addr addr;

public:
  export_option(inet_addr_option);
  inet_addr_option(void) 		{ addr.s_addr = (in_addr_t)0; }
  virtual ~inet_addr_option(void) 	{ }
  virtual bool set(const char*);
  void *getReference(void) 		{ return (void *)(addr.s_addr); }
};

class symbol_option : public complex_option {
private:
  void *sym;;

public:
  export_option(symbol_option);
  symbol_option(void) : sym(NULL)	{ }
  virtual ~symbol_option(void) 		{ }
  virtual void cleanup(void);
  virtual bool set(const char*);
  void *getReference(void) 		{ return sym; }
};

class text_set_option : public complex_option {
private:
  Set<text> tset;

public:
  export_option(text_set_option);
  text_set_option(void) 		{ }
  virtual ~text_set_option(void) 	{ }
  virtual bool set(const char*);
  void *getReference(void) 		{ return (void *)&tset; }
};

class addr_set_option : public complex_option {
private:
  Set<in_addr_t> aset;

public:
  export_option(addr_set_option);
  addr_set_option(void) 		{ }
  virtual ~addr_set_option(void) 	{ }
  virtual bool set(const char*);
  void *getReference(void) 		{ return (void *)&aset; }
};

class int_set_option : public complex_option {
private:
  Set<int> iset;

public:
  export_option(int_set_option);
  int_set_option(void) 			{ }
  virtual ~int_set_option(void) 	{ }
  virtual bool set(const char*);
  void *getReference(void) 		{ return (void *)&iset; }
};

class int_array_option : public complex_option {
private:
  const static int max_size = 16;
  Array<int> iarray;

public:
  export_option(int_array_option);
  int_array_option(void) 		{ }
  virtual ~int_array_option(void) 	{ }
  virtual bool set(const char*);
  void *getReference(void) 		{ return (void *)&iarray; }
};

template<class T>
class template_option : public complex_option {
private:
  T *t;

public:
  export_option(template_option<T>);
  template_option(void) : t(NULL) 	{ }
  virtual ~template_option(void) 	{ }
  virtual void cleanup(void) 		{ __DELETE(t); }
  virtual bool set(const char *str) 	{ t = new T(str); return true; }
  void *getReference(void) 		{ return (void *)t; }
};

#endif /* COMMON_OPTIONS_H */
