/* Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include <libpandora/global.h>

extern "C" {
#include <libpandora/conf/inet_aton.h>
}

#include <libpandora/common_options.h>
#include <libpandora/component.h>
#include <libpandora/dispatcher.h>
#include <libpandora/pandora.h>
#include <libpandora/error.h>

void file_option::cleanup(void)
{
  if (f != NULL) {
    if ((f != stdin) & (f != stdout) & (f != stderr)) {
      fclose(f);
    } else {
      fflush(f);
    }
    f = NULL;
  }
}

bool write_file_option::set(const char *name)
{
  if (f != NULL) return true;
  f = ((name == NULL) ? stdout : fopen(name, "w"));
  return (f != NULL);
}

bool read_file_option::set(const char *name)
{
  if (f != NULL) return true;
  f = ((name == NULL) ? stdout : fopen(name, "r"));
  return (f != NULL);
}

bool ofstream_option::set(const char *str)
{
  if (s != NULL) return true;
  s = new fstream(str, ios::out);
  return !s->bad();
}

stack_option::stack_option(void) 
  : h(NIL_STACK_HANDLE)
{ 
}

void stack_option::cleanup(void)
{
  pandora->release(h);
  h = NIL_STACK_HANDLE;
}

bool fresh_stack_option::set(const char *stk)
{
  if (h != NIL_STACK_HANDLE) pandora->release(h);
  //pandora_debug("[linking to " << stk << " (fresh)]");
  h = pandora->start(text(stk), true, false);
  disp = pandora->dispatcher(h);
  return (h != NIL_STACK_HANDLE);
}

bool running_stack_option::set(const char *stk)
{
  if (h != NIL_STACK_HANDLE) pandora->release(h);
  //pandora_debug("[linking to " << stk << " (running)]");
  h = pandora->use(text(stk));
  disp = pandora->dispatcher(h);
  return (h != NIL_STACK_HANDLE);
}

bool inet_addr_option::set(const char *str)
{
  if (str == NULL) return false;
  if (inet_aton(str, &addr) == 0) {
    pandora_warning(str << ": invalid IP address");
    return false;
  }

  return true;
}

bool symbol_option::set(const char *str)
{
  if (sym != NULL) return true;
  if (str == NULL) return false;

  sym = dynloader->load((symbol_id_t) string_hash(str));

  if (sym == NULL) {
    pandora_warning("cannot find specified symbol: " << str);
    return false;
  }

  return true;
}

void symbol_option::cleanup(void)
{
  if (sym != NULL) {
    dynloader->unload(sym);
    sym = NULL;
  }
}

bool text_set_option::set(const char *str)
{
  if (tset.size() > 0) return true;
  if (str == NULL) return false;

  const char *beg, *end;
  for (beg = str, end = str; *end != '\0'; ++end) {
    if (*end == ';') {
      if (beg < end) tset.add(text(beg, end - beg));
      beg = end + 1;
    }
  }
  if (beg < end) tset.add(text(beg, end - beg));

  return true;
}


bool addr_set_option::set(const char *_str)
{
  if (aset.size() > 0) return true;
  if (_str == NULL) return false;

  struct in_addr addr;
  char *str = xstrdup(_str);
  for (char *tmp = strtok(str, "; \t\r\n"); 
       tmp != NULL; 
       tmp = strtok(NULL, "; \t\r\n")) {
    if (inet_aton(tmp, &addr) == 0) continue;
    //pandora_debug("parsed address: " << addr);
    aset.add(addr.s_addr);
  }

  __FREE(str);

  return true;
}

bool int_set_option::set(const char *str)
{
  if (iset.size() > 0) return true;
  if (str == NULL) return false;

  const char *beg = str;
  char *end;
  do {
    int val = strtol(beg, &end, 0);
    if (end == NULL) break;
    if (end == beg) {
      ++beg;
      continue;
    }
    iset.add(val);
  } while((end != NULL) && (*end != '\0'));

  return true;
}

bool int_array_option::set(const char *str)
{
  if (iarray.size() > 0) return true;
  if (str == NULL) return false;

  iarray.init(max_size);
  int i = 0;
  const char *beg = str;
  char *end;
  do {
    int val = strtol(beg, &end, 0);
    if (end == NULL) break;
    if (end == beg) {
      ++beg;
      continue;
    }
    //pandora_debug("storing " << val << " at pos " << i);
    iarray[i] = val;
    if ((++i) > max_size) break;
    beg = end;
  } while((end != NULL) && (*end != '\0'));

  return true;
}
