/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef BATCH_H
#define BATCH_H

#include <libpandora/global.h>

#include <list>
#include <libpandora/thread.h>

template<class elt_t>
class Batch 
{
private:
  typedef list<elt_t> list_t;
  typedef Batch<elt_t> batch_t;

private:
  Condition cond;
  list_t l;
  int factor;
  int timeout;
  int nelts;
  bool finished;

public:
  Batch(void) : factor(1), timeout(1000), nelts(0), finished(false) { }
  ~Batch(void) { }

public:
  void put(elt_t data) {
    cond.lock();
    l.push_back(data);
    ++nelts;
    if (nelts >= factor) cond.signal();
    cond.unlock();
  }

  elt_t get(void) {
    cond.lock();
    while (nelts == 0) {
      if (timeout == 0) {
	cond.wait();
      } else {
	cond.timed_wait(timeout);
      }
    }

    elt_t elt = l.front();
    l.pop_front();
    --nelts;
    cond.unlock();
    return elt;
  }

  int get(elt_t *tab, int siz) {
    cond.lock();
    if (nelts < siz) {
      if (timeout == 0) {
	cond.wait();
      } else {
	cond.timed_wait(timeout);
      }
    }

    int i = 0;
    for (i = 0; (i < siz) && (nelts > 0); ++i) {
      tab[i] = l.front();
      l.pop_front();
      --nelts;
    }

    cond.unlock();
    return i;
  }

  size_t size(void) {
    return nelts;
  }

  bool isEmpty(void) {
    return l.empty();
  }

  bool isFinished(void) {
    return finished;
  }

  void setFactor(int f) {
    if (f > 0) factor = f;
  }

  void setTimeout(int t) {
    if (t >= 0) timeout = t;
  }

  void flush(void) {
    timeout = 1;
    factor = 1;
  }
};

#endif /* BATCH_H */
