/* ---*-C++-*---------------------------------------------------------------
Copyright (C) 1999, 2000, 2001 Simon Patarin, INRIA

This file is part of Pandora, the Flexible Monitoring Platform.

Pandora is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Pandora is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Pandora; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */


#ifndef ASSOCIATION_H
#define ASSOCIATION_H

#include <libpandora/global.h>

/* Association{}s are used to store bindings between keys and values
in Map{}s.  This allows Map{} to inherit much of its implementation
from Set{}.  Association{}s have a particular property: they compare
(and hash) on their key only.  If you're upset about this, you
shouldn't be. */

template<class Key, class Val> class Association			//@
/* The application must specify the types of the key and value for an
   Association{}.  See the documentation of Map{} for details of how
   this is hidden from the application in most circumstances. */
{
public:
  Key key;
  Val value;
public:
  //=Constructors
  Association(Key k, Val v)						//!
    : key(k), value(v) {}
  /* Both a key and a value are normally required. */

  Association()							//!
    : key(Key()), value(Val()) {}
  /* The default constructor creates an Association{} suitable for use
     as the ``undefined element'' in a Map{}.  See the documentation
     of Map{} for details of how subclasses should be implemented to
     create a suitable default Association{}. */

  Association(const Association<Key, Val> &other) {
    key= other.key;
    value= other.value;
  }

  Association &operator=(const Association<Key, Val> &other) {
    key= other.key;
    value= other.value;
    return *this;
  }

  //=Comparing

  // Associations are lookup keys: they compare and hash the same as their key
  operator long(void) const						//!
    { return (long)key; }
  /* Returns a hash for the Association{}, which is the same as the
     hash for the Association{}'s key. */

  bool operator==(const Association<Key, Val> &other) const		//!
    { return key == other.key; }
  bool operator!=(const Association<Key, Val> &other) const		//!
    { return !operator==(other); }
  /* Equality tests are also based on the equality of the key. */

};


#endif /* ASSOCIATION_H */
