/*
   pam_bioapi - Provides a PAM-compliant interface to use in biometrically 
   authenticating local users.

   Copyright (C) 2005 Michael R. Crusoe <michael at qrivy dot net>
   Copyright (C) 2006 Josef Hajas <josef at hajas dot net>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


*/

#include "pam_bioapi.h"

#ifdef DEBUG
int debug = 1;
#else
int debug = 0;
#endif

int
processMsg(const char *whatToProcess, struct pam_conv *conv,
	   FILE *streamFromChild, FILE * streamToChild,  out_params_t *out)
{
    const char *line;
    char *payload = NULL;
    char *msg = NULL;
    char *endOfLinePtr;
    char msgFromChild[MAXMSG];
    char *listStr = NULL;
    char action = '\0';
    int doIt;
    int pamRet;
    int end = 0;
    int type = 0;
    int success = 0;
    int msgType = PAM_TEXT_INFO;
    struct pam_response *response = NULL;
    int rc = -1;
    int i;

    //if (debug == 1) syslog(LOG_DEBUG, "processMsg(): whatToProcess=[%s]", whatToProcess);

    while (whatToProcess[0] != '\0') {
	line = whatToProcess;
	endOfLinePtr = strchr(whatToProcess, '\n');
	if (endOfLinePtr != NULL) {
	    whatToProcess = endOfLinePtr + 1;
	    endOfLinePtr[0] = '\0';
	}

	switch (line[0]) {
	case 'u':
	    /* it's authenticated username */
	    if ((line[1] == ':') && (line[2] == ' ')) {
		out->user = (char *) strdup (line + 3);
	    } else {
	      syslog(LOG_ERR, "%s", line);
	    }
	    break;
	case 'p':
	    if ((line[0] == 'p') && (line[1] == ':') && (line[2] == ' ')) {
	      out->payload = strdup ((const char *)line+3);
	    } else 
	    if ((line[0] == 'p') && (line[1] == '?')) {
	      payload = getPayloadFromUser(conv);
	      fprintf(streamToChild, "p: %s\n", payload);
	      fflush(streamToChild);
	      for (i = 0; i < strlen(payload); i++) payload[i] = 'x';
	      //free (payload);
	    } else {
	      syslog(LOG_ERR, "%s", line);
	    }
	    break;
	case 'o':
	    if ((line[1] == ':') && (line[2] == ' ')) {
	      out->operation = strdup((char *) line + 3);
	    } else {
	      syslog(LOG_ERR, "%s", line);
	    }
	    break;
	case 'g':
	    /* some pam_conv() msg */
	    rc = sscanf(line, "g: %d ", &msgType);
	    if (rc == 1) {
	      msg = (char *) line + 5; // TODO: rewrite to make usable with multidigit pam msg type
	      if ((out->operation != NULL) && (msgType == PAM_ERROR_MSG)) {
		asprintf (&msg, _("Operation %s failed: %s"), _(out->operation), _(msg));
		free(out->operation);
		out->operation = NULL;
	      } 
		pamRet = pam_info(conv, _(msg), msgType, &response); 
		if (pamRet != PAM_SUCCESS) {
                    kill (child_pid, SIGINT);
		}
		if ((msgType == PAM_PROMPT_ECHO_OFF) ||
		    (msgType == PAM_PROMPT_ECHO_ON)) {
		    if (response != NULL) {
			fprintf(streamToChild, "r: %s\n", response->resp);
			fflush(streamToChild);
			//free(response->resp);
			free(response);
		    }
		}
	    } else {
		syslog(LOG_ERR, "%s", line);
	    }
	    break;
	case 'l':
	    if (line[1] == ':') {
	      line = line + 2;
	      if (out->arrayOfEnrolled) {
		free (out->arrayOfEnrolled);
	      }
	      out->arrayOfEnrolled = earray_get(line);
	      if (! out->arrayOfEnrolled) {
		fprintf(streamToChild, "a: x\n");
		fflush(streamToChild);
		return (1);
	      }
	    } else {
		syslog(LOG_ERR, "%s", line);
	    }
	    break;
	case 'a':
	    if (line[1] == '?') {
	      do {
		doIt = 1;
	        action = getActionFromUser(conv);
		switch (action) {
		  case 'e':
		    listStr = listAllRecords(out->arrayOfEnrolled, KNOWN_TYPES_COUNT);
		    type = getTypeFromUser(conv, listStr, KNOWN_TYPES_COUNT, NULL);
		    if (type == -1) {
		      doIt = 0;
		    } else if (earray_isInArray(out->arrayOfEnrolled, type)) {
		      doIt = askIfDelete(conv,type,2);
		      if (doIt) {
			fprintf(streamToChild, "a: d\n");
			fflush(streamToChild);
			fprintf(streamToChild, "t: %d\n", type);
			fflush(streamToChild);
		        fgets(msgFromChild, sizeof(msgFromChild),streamFromChild);
		        fgets(msgFromChild, sizeof(msgFromChild),streamFromChild);
		      }
		    }
		    if (doIt) {
		      fprintf(streamToChild, "a: e\n");
		      fflush(streamToChild);
		      fprintf(streamToChild, "t: %d\n", type);
		      fflush(streamToChild);
		    }
		    break;
		  case 'd':
		    listStr = listOnlyEnrolledRecords (out->arrayOfEnrolled);
		    type = getTypeFromUser(conv, listStr, KNOWN_TYPES_COUNT, out->arrayOfEnrolled);
		    if (type == -1) {
		      doIt = 0;
		    } else {
		      doIt = askIfDelete(conv,type,0);
		      if (doIt) {
			fprintf(streamToChild, "a: d\n");
			fflush(streamToChild);
			fprintf(streamToChild, "t: %d\n", type);
			fflush(streamToChild);
		      }
		    }
		    break;
		  case 'l': 
		    listStr = listOnlyEnrolledRecords (out->arrayOfEnrolled);
		    pam_info (conv, listStr, PAM_TEXT_INFO, &response);
		    doIt = 0;
		    break;
		  case 'x': 
		    doIt = 1;
		    fprintf(streamToChild, "a: x\n");
		    fflush(streamToChild);
		  break;
		}
	      } while (doIt == 0);
	    } else {
	      syslog(LOG_ERR, "%s", line);
	    }
	    break;
	case 'd':
	    success = sscanf (line, "d? %d", &type);
	    if (success) {
	      doIt = askIfDelete(conv,type,1);
	      if (doIt) {
		fprintf(streamToChild, "r: y\n");
		fflush(streamToChild);
	      } else {
		fprintf(streamToChild, "r: n\n");
		fflush(streamToChild);
	      }

	    } else {
	      syslog(LOG_ERR, "%s", line);
	    }
	    break;
	default:
	    /*it is something BSP send in it's error output -> print it to log */
	    syslog(LOG_ERR, "%s", line);
	}
    }

    return end;
}

int
pam_info(const struct pam_conv *conv, const char *msg,
	 int style, struct pam_response **response)
{
    int retval;
    const void *void_conv;
    struct pam_message message;
    const struct pam_message *pmsg;

    message.msg_style = style;
    message.msg = _(msg);
    pmsg = &message;

    retval = conv->conv(1, &pmsg, response, conv->appdata_ptr);
    if (retval != PAM_SUCCESS) {
	syslog(LOG_ERR, "converse failed");
    }
    return retval;
}

char getActionFromUser (struct pam_conv *conv) {
  int pamRet = PAM_AUTH_ERR;
  int correct = 0;
  char *response = NULL;
  struct pam_message msg[2];
  const struct pam_message *pmsg[2];
  struct pam_response *presponse = NULL;
  char action = 'e';
  const char *unknownActionMsg =
	_("Unknown action. Please try it again.");
  const char *actionMsg =
	_("What would you like to do?\ne ... enroll new finger.\nd ... delete finger\nl ... list enrolled fingers\nx ... exit (next pam module please)\nYour choice [e]: ");

  while (correct == 0) {
      pmsg[0] = &msg[0];
      msg[0].msg_style = PAM_PROMPT_ECHO_ON;
      msg[0].msg = actionMsg;
      presponse = NULL;
      /* Prompt the user with the challenge, and get their response */
      pamRet = conv->conv(1,
			  (const struct pam_message **) &pmsg,
			  &presponse, conv->appdata_ptr);
      if (pamRet != PAM_SUCCESS) {
	  exit(pamRet);
      }

      response = presponse[0].resp;
      switch (response[0]) {
      case '\0':
      case 'e':
	  action = 'e';
	  correct = 1;
	  break;
      case 'd':
	  action = 'd';
	  correct = 1;
	  break;
      case 'l':
	  action = 'l';
	  correct = 1;
	  break;
      case 'x':
	  action = 'x';
	  correct = 1;
	  break;
      default:
	  pmsg[0] = &msg[0];
	  msg[0].msg = unknownActionMsg;
	  msg[0].msg_style = PAM_TEXT_INFO;
	  pamRet = conv->conv(1,
			      (const struct pam_message **) &pmsg,
			      &presponse, conv->appdata_ptr);
	  if (pamRet != PAM_SUCCESS) {
	      exit(PAM_CONV_ERR);
	  }
      }
  }
  return action;
}

int askIfDelete (struct pam_conv *conv, int type, int reason) {
  char *question;
  char *reasonStr;
  int cont = 0;
  int pamRet = PAM_AUTH_ERR;
  int i = 0;
  int correct = 0;
  char *response = NULL;
  struct pam_message msg[2];
  const struct pam_message *pmsg[2];
  struct pam_response *presponse = NULL;

  switch (reason) {
    case 0: 
        reasonStr = "";
	break;
    case 1:
    	reasonStr = _("Record already stored as different finger. ");
	break;
    case 2:
    	reasonStr = _("Record already enrolled. ");
	break;
  }

  asprintf (&question, _("%sAre you sure you want to delete %s? [N/y] "), 
      reasonStr, _(TypeDefinition[type-1]));

  while (correct == 0) {
      pmsg[i] = &msg[i];
      msg[i].msg_style = PAM_PROMPT_ECHO_ON;
      msg[i].msg = question;
      i++;
      /* Prompt the user with the challenge, and get their response */
      pamRet = conv->conv(i,
			  (const struct pam_message **) &pmsg,
			  &presponse, conv->appdata_ptr);

      if (pamRet == PAM_SUCCESS) {
	  i--;
	  response = presponse[i].resp;
	  if ((response[0] == 'y') || (response[0] == 'Y')) {
	      correct = 1;
	      cont = 1;
	  } else if ((response[0] == '\0') || (response[0] == 'n') || (response[0] == 'N')) {
	      correct = 1;
	      cont = 0;
	  } else {
	      pmsg[0] = &msg[0];
	      msg[0].msg =
		  _("Wrong answer! Please answer either Y or N.");
	      msg[0].msg_style = PAM_TEXT_INFO;
	      i = 1;
	  }
      }
  }
  return cont;
}

/* compose string with types description and return it.
 * also in maxTypes return number of types printed
 */
char *listOnlyEnrolledRecords (earray_t earray) {
  int i, type;
  char *rec = NULL;
  char *msg = "";

  for (i=0; earray[i] != EARRAY_END; i++) {
     type = earray[i];
     if (type <= KNOWN_TYPES_COUNT) {
       asprintf (&msg, "%s%d ... %s\n", msg, type, _(TypeDefinition[type-1]));
     } else  {
       asprintf (&rec, "type %d", type);
       asprintf (&msg, "%s%d ... %s\n", msg, type, _(rec));
     }
  }
  if (strlen(msg) == 0) {
    msg = _("No records found.");
  }

  return msg;
}

char *listAllRecords (earray_t earray, int maxTypes) {
  int i, type;
  char *rec = NULL;
  char *msg = "";
  char *enrolledMsg = _("(enrolled)");
  char *empty = "";
  char *enrolled = "";

  for (i=0; i < maxTypes; i++) {
     type = i+1;
     if (earray_isInArray(earray, type)) {
       enrolled = enrolledMsg;
     } else {
       enrolled = empty;
     }
     if (type <= KNOWN_TYPES_COUNT) {
       asprintf (&msg, "%s%d ... %s %s\n", msg, i+1, _(TypeDefinition[i]), enrolled);
     } else  {
       asprintf (&rec, "type %d", type);
       asprintf (&msg, "%s%d ... %s %s\n", msg,i+1, _(rec), enrolled);
     }
  }

  return msg;
}

char *getPayloadFromUser (struct pam_conv *conv) {
  int pamRet = PAM_AUTH_ERR;
  int correct = 0;
  int type = 0, i = 0;
  char *response = NULL;
  char *payload = NULL;
  struct passwd *pw;
  struct pam_message msg[2];
  const struct pam_message *pmsg[2];
  struct pam_response *presponse = NULL;
  const char *recordTypeMsg = NULL;
  const char *payloadMsg = _("Password to save at finger: ");

  pmsg[i] = &msg[i];
  msg[i].msg_style = PAM_PROMPT_ECHO_OFF;
  msg[i].msg = payloadMsg;
  /* Prompt the user with the challenge, and get their response */
  pamRet = conv->conv(1, (const struct pam_message **) &pmsg,
		      &presponse, conv->appdata_ptr);

  if (pamRet == PAM_SUCCESS) {
	payload = presponse[i].resp;
  } else {
	syslog(LOG_DEBUG, "Error in getPayloadFromUser()");
  }

  payload = (char *) memfrob (payload, sizeof(payload));

  return payload;
}

/* Ask user for finger and send it to helpder binary. Also check if intered 
 * type is correct. If earray != NULL, then check entered ID agains it. If it is
 * NULL then check if ID between 1 and maxType */
int getTypeFromUser (struct pam_conv *conv, char *listOfTypes, int maxTypes, earray_t earray) {
  int pamRet = PAM_AUTH_ERR;
  int correct = 0;
  int type = 0, i = 0;
  char *response = NULL;
  struct pam_message msg[2];
  const struct pam_message *pmsg[2];
  struct pam_response *presponse = NULL;
  const char *recordTypeMsg = NULL;
  asprintf (&recordTypeMsg, "---\n%s---\n%s\n%s ", listOfTypes,
        _("x ... cancel"),
	_("Finger ID (number please):"));

  while (correct == 0) {
      pmsg[i] = &msg[i];
      msg[i].msg_style = PAM_PROMPT_ECHO_ON;
      msg[i].msg = recordTypeMsg;
      i++;
      /* Prompt the user with the challenge, and get their response */
      pamRet = conv->conv(i,
			  (const struct pam_message **) &pmsg,
			  &presponse, conv->appdata_ptr);

      if (pamRet == PAM_SUCCESS) {
	  i--;
	  if (presponse[i].resp[0] == 'x') {
	    type = -1;
	    correct = 1;
	  } else {
	    type = atoi(presponse[i].resp);
	    if (debug == 1) 
		syslog(LOG_DEBUG, "Got finger ID: '%d'", type);
	    if ((earray == NULL) && ((type < 1) || (type > maxTypes))
		|| ((earray != NULL) && (! earray_isInArray(earray,type)))) {
		pmsg[0] = &msg[0];
		msg[0].msg =
		    _("Wrong finger specified! Please specify finger as number.");
		msg[0].msg_style = PAM_TEXT_INFO;
		i = 1;
	    } else {
		correct = 1;
	    }
	  }
      }
  }
  return type;
}

int
askUserIfContinue (struct pam_conv *conv) {
  int pamRet = PAM_AUTH_ERR;
  int i = 0;
  int cont = 1;
  int correct = 0;
  char *response = NULL;
  struct pam_message msg[2];
  const struct pam_message *pmsg[2];
  struct pam_response *presponse = NULL;
  const char *startMsg =
	_("Would you like to manipulate with biometrics records? [n/Y]");

  while (correct == 0) {
      pmsg[i] = &msg[i];
      msg[i].msg_style = PAM_PROMPT_ECHO_ON;
      msg[i].msg = startMsg;
      i++;
      /* Prompt the user with the challenge, and get their response */
      pamRet = conv->conv(i,
			  (const struct pam_message **) &pmsg,
			  &presponse, conv->appdata_ptr);

      if (pamRet == PAM_SUCCESS) {
	  i--;
	  response = presponse[i].resp;
	  if ((response[0] == '\0') || (response[0] == 'y') || (response[0] == 'Y')) {
	      correct = 1;
	      cont = 1;
	  } else if ((response[0] == '\0') || (response[0] == 'n') || (response[0] == 'N')) {
	      correct = 1;
	      cont = 0;
	  } else {
	      pmsg[0] = &msg[0];
	      msg[0].msg =
		  "Wrong answer! Please answer either Y or N.";
	      msg[0].msg_style = PAM_TEXT_INFO;
	      i = 1;
	  }
      }
  }
  return cont;

}

static void setup_int_signal(void)
{
    struct sigaction action;

    (void) memset((void *) &action, 0, sizeof(action));
    action.sa_handler = su_sighandler;
    action.sa_flags = 0;
    (void) sigaction(SIGINT, &action, NULL); //for cancelation of biometric operation
    (void) sigaction(SIGUSR1, &action, NULL); //for cancelation of biometric operation
    action.sa_handler = child_sighandler;
    (void) sigaction(SIGCHLD, &action, NULL);
}

static void child_sighandler(int sig)
{
    child_died = 1;
    setup_int_signal();
}

static void su_sighandler(int sig)
{
    //if (sig == SIGINT) {
        if (child_pid != -1) {
	    user_signaled = 1;
            kill (child_pid, SIGINT);
	}
	setup_int_signal();
    //}
}


 /* Authentication functions */
PAM_EXTERN int
pam_sm_authenticate(pam_handle_t * pamh, int flags,
		    int argc, const char **argv)
{
    extern int optind;
    int opt;
    const char *user = NULL;
    char *payload = NULL;
    int askUsername = 0;
    int powersave = 0;
    int pamRet = PAM_AUTH_ERR;
    int pamRetTmp = PAM_AUTH_ERR;
    struct pam_conv *conv;
    struct pam_response *response = NULL;
    char *tmpArgs[argc+1];
    char *newargs[argc + 4];
    char *package_name = "tfmessbsp";
    char *bsp = NULL;
    int i = 0, j = 0;

    int child, pipeToChild[2], pipeFromChild[2];
    void (*sighandler) (int) = NULL;

    openlog("pam_bioapi", LOG_PID | LOG_CONS, LOG_AUTHPRIV);

    //pamRet = pam_fail_delay(pamh, 100);

    /* copy argument for binary execution */
    tmpArgs[0] = "pam_bioapi";
    for (i = 0; i < argc; i++) {
	tmpArgs[i+1] = (char *) argv[i];
    }

    optind = 0;

    while ((opt = getopt_long (argc+1, (char **) tmpArgs, 
	    "b:B:d:t:T:ugpswD", long_options, NULL)) != -1) {
	switch (opt) {
	    /* BSP UUID */
	case 'b':
	    bsp = (char *) tmpArgs[optind - 1];
	    break;
	/* Translation */
	case 'T':
	    package_name = (char *) tmpArgs[optind - 1];
	    break;
	case 'u':
	    askUsername = 1;
	    break;
	/* Power save regime - ask user question before start bioapi operation */
	case 'w':
	    powersave = 1;
	    break;
	case 'D':
	    debug = 1;
	    break;
	}
    }

    if (bsp == NULL) {
	syslog(LOG_ALERT,
	       "BSP is mandatory parameter. Please specify it!");
	return PAM_AUTH_ERR;
    }

    syslog(LOG_ALERT,
	       "%s",PACKAGE_LOCALE_DIR);
    setlocale(LC_MESSAGES, "");
    bindtextdomain(package_name, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (package_name, "UTF-8");
    textdomain(package_name);

    /* create a pipes */
    if (pipe(pipeFromChild) != 0) {
	return PAM_AUTH_ERR;
    }
    if (pipe(pipeToChild) != 0) {
	return PAM_AUTH_ERR;
    }
    //if (debug == 1) syslog(LOG_DEBUG,"pipes created.");

    /* TODO
     * if (off(UNIX_NOREAP, ctrl)) {
     *
     * This code arranges that the demise of the child does not cause
     * the application to receive a signal it is not expecting - which
     * may kill the application or worse.
     *
     * The "noreap" module argument is provided so that the admin can
     * override this behavior.
     *
     * sighandler = signal(SIGCHLD, SIG_DFL);
     * }
     */
 
    pamRet = pam_get_item(pamh, PAM_CONV, (const void **) &conv);
    if ((pamRet != PAM_SUCCESS) || (conv == NULL)) {
	syslog(LOG_ALERT, "fail pam_get_item PAM_CONV: %s",
	       pam_strerror(pamh, pamRet));
	return pamRet;
    }
   
    if (askUsername == 1) {
	pamRet = pam_get_user(pamh, &user, NULL);
	if (pamRet != PAM_SUCCESS) {
	    if (pamRet == PAM_CONV_AGAIN) {
		pamRet = PAM_INCOMPLETE;
	    } else {
		syslog(LOG_ALERT, "Error retrieving user name: %s",
		       pam_strerror(pamh, pamRet));
		pamRet = PAM_USER_UNKNOWN;
	    }
	    closelog();
	    return pamRet;
	}
    }
    if (powersave == 1) {
        pam_info(conv, _("Start biometrics authentication? [n/Y]"), 
	    PAM_PROMPT_ECHO_ON, &response);
	if ((response == NULL) || 
	    (response->resp[0] == 'n') ||
	    (response->resp[0] == 'N')) {
	    closelog();
	    pamRet = PAM_AUTH_ERR;
	    return pamRet;
	}
    }

    /* fork */
    child = fork();
    if (child == 0) {
	struct rlimit rlim;
	static char *envp[] = { NULL };

	close(0);
	close(2);
	/* reopen stderr as pipe */
	close(pipeFromChild[0]);
	dup2(pipeFromChild[1], STDERR_FILENO);

	close(pipeToChild[1]);
	dup2(pipeToChild[0], STDIN_FILENO);

	if (getrlimit(RLIMIT_NOFILE, &rlim) == 0) {
	    for (i = 3; i < (int) rlim.rlim_max; i++) {
		//if (pipeToChild[0] != i)
		close(i);
	    }
	}

	if (geteuid() == 0) {
	    /* must set the real uid to 0 so the helper will not error
	       out if pam is called from setuid binary (su, sudo...) */
	    //setuid(0);
	}

	/* exec binary helper */
	newargs[0] = strdup(CHBIRD_HELPER);
	newargs[1] = strdup("auth");

	/* copy argument for binary execution */
	j = 2;
	for (i = 0; i < argc; i++) {
	    newargs[j] = (char *) argv[i];
	    if ((argv[i][0] == '-') && (argv[i][1] == 'u')) {
		j++;
		newargs[j] = strdup(user);
	    }
	    j++;
	}
	newargs[j] = NULL;

	execve(CHBIRD_HELPER, newargs, envp);

	/* should not get here: exit with error */
	if (debug == 1) {
	    syslog(LOG_ERR,
		   "Error running helper binary. Error code: %d. Please see errno-base.h and man execve",
		   errno);
	} else {
	    syslog(LOG_ERR, "Error running helper binary.");
	}
	exit(PAM_AUTHINFO_UNAVAIL);
    } else if (child > 0) {
	int rc = 0;
	int endReceived = 0;
	char msgFromChild[MAXMSG + 1];
	char *str = NULL;
	int status;
	FILE *streamFromChild = fdopen(pipeFromChild[0], "r");
	FILE *streamToChild = fdopen(pipeToChild[1], "w");
	out_params_t out;

        struct sigaction old_action_int;
        struct sigaction old_action_usr;
        struct sigaction old_action_chld;
	child_pid = child;
	sigaction (SIGINT, NULL, &old_action_int);
	sigaction (SIGUSR1, NULL, &old_action_usr);
	sigaction (SIGCHLD, NULL, &old_action_chld);
	/*if (old_action.sa_handler != SIG_IGN)*/
	    setup_int_signal();

	out.user = (char *) user;
	out.payload = NULL;
	out.operation = NULL;
	out.arrayOfEnrolled = NULL;

	close(pipeFromChild[1]);
	close(pipeToChild[0]);

	while (!feof(streamFromChild)
	       && !ferror(streamFromChild)
	       && fgets(msgFromChild, MAXMSG, streamFromChild) != NULL ) {
	    msgFromChild[MAXMSG] = '\0';	/* NULL terminate */
	    endReceived = processMsg(msgFromChild, conv, 
		streamFromChild, streamToChild, &out);
	}
        user = out.user;
        payload = out.payload;
	if (out.arrayOfEnrolled) {
	  free(out.arrayOfEnrolled);
	}

	close(pipeFromChild[0]);

        rc=waitpid(child, &status, 0); 
	
	if ((rc<0) || (! WIFEXITED(status))) {
	  pamRet = PAM_AUTH_ERR;
        } else {
	  pamRet = WEXITSTATUS(status);
	}
	//if (debug) syslog (LOG_DEBUG, "helper exit status %d",pamRet);

        sigaction (SIGINT, &old_action_int, NULL);
        sigaction (SIGUSR1, &old_action_usr, NULL);
        sigaction (SIGCHLD, &old_action_chld, NULL);

    } else {
	if (debug == 1)
	    syslog(LOG_DEBUG, "fork failed");
	close(pipeFromChild[1]);
	pamRet = PAM_AUTH_ERR;
    }
    close(pipeFromChild[0]);
    
    if ((askUsername == 0) && (pamRet == PAM_SUCCESS) && (user != NULL)) {
	if (debug == 1)
	    syslog(LOG_DEBUG, "Setting username to %s", user);
	pamRetTmp = pam_set_item(pamh, PAM_USER, user);
	if (pamRetTmp != PAM_SUCCESS) {
	    syslog(LOG_ALERT, "Can't set username %s as PAM item!\n",
		   user);
	}
    }
    if (payload != NULL) {
        payload = (char *) memfrob (payload, sizeof(payload));
	pamRetTmp = pam_set_item(pamh, PAM_AUTHTOK, payload);
	for (i = 0; i < strlen(payload); i++) payload[i] = 'x';
	//free (payload);
	if (pamRetTmp != PAM_SUCCESS) {
	    syslog(LOG_ALERT, "Can't set payload as PAM item!\n");
	}
    }

    if (pamRet == PAM_SUCCESS) {
	syslog(LOG_INFO,
	       "User %s successfully biometrically authenticated.", user);
    } else {
	syslog(LOG_ALERT, "User %s failed to biometrically authenticate.",
	       user);
    }

    closelog();
    return pamRet;
}

/* "Password" functions (in our case, enrollment of the biometric) */
PAM_EXTERN int
pam_sm_chauthtok(pam_handle_t * pamh, int flags, int argc,
		 const char **argv)
{
    extern int optind;
    int opt;
    const char *user = NULL;
    int askUsername = 0;
    int pamRet;
    char *tmpArgs[argc+1];
    char *newargs[argc + 5];
    char *package_name = "tfmessbsp";
    int i = 0, j = 0;

    struct pam_conv *conv;
    char *bsp = NULL;

    int child, pipeToChild[2], pipeFromChild[2];
    void (*sighandler) (int) = NULL;

    openlog("pam_bioapi", LOG_PID | LOG_CONS, LOG_AUTHPRIV);

    //pamRet = pam_fail_delay(pamh, 100);

    if (debug == 1)
	syslog(LOG_DEBUG, "Starting pam_bioapi process");

    /* copy argument for binary execution */
    tmpArgs[0] = "pam_bioapi";
    for (i = 0; i < argc; i++) {
	tmpArgs[i+1] = (char *) argv[i];
    }
    optind = 0;

    while ((opt = getopt_long (argc+1, (char **) tmpArgs,
	    "b:B:d:t:T:gpswD", long_options, NULL)) != -1) {
	switch (opt) {
	    /* BSP UUID */
	case 'b':
	    bsp = (char *) tmpArgs[optind - 1];
	    break;
	/* Translation */
	case 'T':
	    package_name = (char *) tmpArgs[optind - 1];
	    break;
	case 'D':
	    debug = 1;
	    break;
	}
    }

    if (bsp == NULL) {
	syslog(LOG_ALERT,
	       "BSP is mandatory parameter. Please specify it!");
	return PAM_PERM_DENIED;
    }

    setlocale(LC_MESSAGES, "");
    bindtextdomain(package_name, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset (package_name, "UTF-8");
    textdomain(package_name);

    /* TODO
     * if (off(UNIX_NOREAP, ctrl)) {
     *
     * This code arranges that the demise of the child does not cause
     * the application to receive a signal it is not expecting - which
     * may kill the application or worse.
     *
     * The "noreap" module argument is provided so that the admin can
     * override this behavior.
     *
     sighandler = signal(SIGCHLD, SIG_DFL);
     }*/

    pamRet = pam_get_user(pamh, &user, NULL);
    if (pamRet != PAM_SUCCESS) {
	if (pamRet == PAM_CONV_AGAIN) {
	    pamRet = PAM_INCOMPLETE;
	} else {
	    syslog(LOG_ALERT, "Error retrieving user name: %s",
		   pam_strerror(pamh, pamRet));
	    pamRet = PAM_USER_UNKNOWN;
	}
	closelog();
	return pamRet;
    }

    if (flags & PAM_PRELIM_CHECK) {
        // first call
        // TODO: check if helper binary exist
	closelog();
	return PAM_SUCCESS;
    } else if (! (flags & PAM_UPDATE_AUTHTOK)) {
        // this is not first either second call - what's this?
	return PAM_ABORT;
    }

    pamRet = pam_get_item(pamh, PAM_CONV, (const void **) &conv);
    if ((pamRet != PAM_SUCCESS) || (conv == NULL)) {
	syslog(LOG_ALERT, "fail pam_get_item PAM_CONV: %s",
	       pam_strerror(pamh, pamRet));
	return pamRet;
    }

    if (askUserIfContinue (conv)) {
      /* create a pipes */
      if (pipe(pipeFromChild) != 0) {
	  syslog(LOG_ALERT, "Could not make pipe from helper binary");
	  return PAM_AUTH_ERR;
      }
      if (pipe(pipeToChild) != 0) {
	  syslog(LOG_ALERT, "Could not make pipe to helper binary");
	  return PAM_AUTH_ERR;
      }

      if (debug == 1)
	  syslog(LOG_DEBUG, "pipes created.");


      if (debug == 1)
	  syslog(LOG_ALERT, "forking");
      /* fork */
      child = fork();
      if (child == 0) {
	  struct rlimit rlim;
	  static char *envp[] = { NULL };

	  if (debug == 1)
	      syslog(LOG_ALERT, "This is child.");
	  close(0);
	  close(2);
	  /* reopen stderr as pipe */
	  close(pipeFromChild[0]);
	  close(pipeToChild[1]);
	  dup2(pipeFromChild[1], STDERR_FILENO);
	  dup2(pipeToChild[0], STDIN_FILENO);
	  if (debug == 1)
	      syslog(LOG_ALERT, "Output/input redirected to/from pipes");

	  if (getrlimit(RLIMIT_NOFILE, &rlim) == 0) {
	      for (i = 3; i < (int) rlim.rlim_max; i++) {
		  //if (pipeToChild[0] != i)
		  close(i);
	      }
	  }

	  if (geteuid() == 0) {
	      /* must set the real uid to 0 so the helper will not error
		 out if pam is called from setuid binary (su, sudo...) */
	      //setuid(0);
	  }

	  /* exec binary helper */
	  newargs[0] = strdup(CHBIRD_HELPER);
	  newargs[1] = strdup("password");
	  newargs[2] = strdup("-u");
	  newargs[3] = strdup(user);

	  /* copy argument for binary execution */
	  j = 4;
	  for (i = 0; i < argc; i++) {
	      if (!((argv[i][0] == '-') && (argv[i][1] == 'u'))) {
		  newargs[j] = (char *) argv[i];
		  j++;
	      }
	  }
	  newargs[j] = NULL;

	  execve(CHBIRD_HELPER, newargs, envp);

	  /* should not get here: exit with error */
	  if (debug == 1) {
	      syslog(LOG_ERR,
		     "Error running helper binary. Error code: %d. Please see errno-base.h and man execve",
		     errno);
	  } else {
	      syslog(LOG_ERR, "Error running helper binary.");
	  }
	  exit(PAM_AUTHINFO_UNAVAIL);
      } else if (child > 0) {
	  /* wait for child */
	  /* if the stored password is NULL */
	  int rc = 0;
	  int type = 0;
	  char msgFromChild[MAXMSG + 1];
	  char *str = NULL;
	  char action;
	  char *operation = NULL;
	  FILE *streamFromChild;
	  FILE *streamToChild;
	  out_params_t out;
	  
	  struct sigaction old_action_int;
	  struct sigaction old_action_usr;
	  struct sigaction old_action_chld;
	  child_pid = child;
    	  user_signaled = 0;
    	  child_died = 0;
	  sigaction (SIGINT, NULL, &old_action_int);
	  sigaction (SIGUSR1, NULL, &old_action_usr);
	  sigaction (SIGCHLD, NULL, &old_action_chld);
	  //if (old_action.sa_handler != SIG_IGN)
	      setup_int_signal();

	  out.user = (char *) user;
	  out.payload = NULL;
	  out.operation = NULL;
	  out.arrayOfEnrolled = NULL;

	  close(pipeFromChild[1]);
	  close(pipeToChild[0]);
	  streamFromChild = fdopen(pipeFromChild[0], "r");
	  streamToChild = fdopen(pipeToChild[1], "w");

	  while ((rc != 1) 
	         && !feof(streamFromChild)
		 //&& !ferror(streamFromChild)
	         /*&& fgets(msgFromChild, sizeof(msgFromChild),
			  streamFromChild) != NULL*/)
	  {
              str = fgets(msgFromChild, sizeof(msgFromChild),
			  streamFromChild);
	      /*if (ferror(streamFromChild)) {
		      syslog (LOG_DEBUG,
			  "str = 0x%Xh, "
			  "errno = %d,"
			  "child_died = %d, ", 
			  str, errno, child_died);
	      }*/
	      if (str == NULL) {
		  continue;
	      }
	      msgFromChild[MAXMSG] = '\0';	/* NULL terminate */
	      rc = processMsg(msgFromChild, conv, 
		  streamFromChild, streamToChild, &out);
	  }

	  rc = waitpid(child, &pamRet, 0);
	  sigaction (SIGINT, &old_action_int, NULL);
	  sigaction (SIGUSR1, &old_action_usr, NULL);
	  sigaction (SIGCHLD, &old_action_chld, NULL);

	  if ((! WIFEXITED (pamRet)) || (rc < 0)) {
	      /* syslog(LOG_ERR,
		     "bioapi_chbird waitpid returned %d, errno = %d", rc,
		     errno); */
	      pamRet = PAM_AUTH_ERR;
	  } else {
	      pamRet = WEXITSTATUS(pamRet);
	  }
      } else {
	  close(pipeFromChild[1]);
	  pamRet = PAM_AUTH_ERR;
      }
    }
    //syslog(LOG_ERR, "chauthok returned %d", pamRet);

    closelog();
    return pamRet;
}

PAM_EXTERN int
pam_sm_setcred(pam_handle_t * pamh, int flags, int argc, const char **argv)
{
    return PAM_SUCCESS;
}

PAM_EXTERN int
pam_sm_acct_mgmt(pam_handle_t * pamh, int flags, int argc,
		 const char **argv)
{
    return PAM_SUCCESS;
}
