/*-
 * Copyright (c) 2006 Fredrik Lindberg. <fli at shapeshifter dot se>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: birdb.h,v 1.1 2006/11/29 20:21:54 nax Exp $
 */
#ifndef _BIRDB_H_
#define _BIRDB_H_

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/queue.h>
#include <time.h>
#include <string.h>
#include <dlfcn.h>
#include <bioapi.h>

#ifndef TAILQ_EMPTY
 #define	TAILQ_EMPTY(head)	((head)->tqh_first == NULL)
#endif

#ifndef TAILQ_FIRST
 #define	TAILQ_FIRST(head)	((head)->tqh_first)
#endif

#ifndef TAILQ_NEXT
 #define	TAILQ_NEXT(elm, field) ((elm)->field.tqe_next)
#endif

#ifndef TAILQ_FOREACH_SAFE
 #define TAILQ_FOREACH_SAFE(var, head, field, tvar)                      \
         for ((var) = TAILQ_FIRST((head));                               \
             (var) && ((tvar) = TAILQ_NEXT((var), field), 1);            \
             (var) = (tvar))
#endif
/*
 * Type of other field in backend database record
 */
typedef struct {
  uint32 Length; /* in bytes */
  uint8 *Data;
} br_other_t;

/*
 * Backend database record
 */
struct birdb_rec
{
  time_t br_ctime;
  char *br_key;
  int br_type;
  br_other_t *br_other;
  BioAPI_BIR *br_bir;
};

/*
 * Backend driver module params
 */
struct birdb_backend
{
  char *be_name;
  char *be_desc;
  void *(*be_open) (const char *, int, char *[]);
  void (*be_close) (void *);
  struct birdb_rec **(*be_get) (void *, struct birdb_rec *);
  void (*be_freegetres) (void *, struct birdb_rec **);
  int (*be_ins) (void *, struct birdb_rec *);
  int (*be_del) (void *, struct birdb_rec *);
  struct birdb_rec *(*be_seqgetfirst) (void *, struct birdb_rec *);
  struct birdb_rec *(*be_seqgetnext) (void *, struct birdb_rec *);
  void (*be_seqfree) (void *, struct birdb_rec *);
};

#define BIRDB_BACKEND_OBJ	"__modcallbacks"
#define BIRDB_BACKEND_PARAMS struct birdb_backend __modcallbacks

#define BIRDB_MAJOR 1
#define BIRDB_MINOR 0

/*
 * Module entry
 */
struct birdb_mod
{
  TAILQ_ENTRY (birdb_mod) entries;
  void *bm_dlh;
  int bm_argc;
  char **bm_argv;
  struct birdb_backend *bm_be;
};

/*
 * Opaque type
 */
typedef struct birdb
{
  TAILQ_HEAD (modhead, birdb_mod) modhead;
} birdb;


birdb *birdb_init (void);
void birdb_close (birdb *);
struct birdb_mod *birdb_addmod (birdb *, const char *);
void birdb_delmod (birdb *, struct birdb_mod *);
struct birdb_mod *birdb_findmod (birdb *, const char *);
struct birdb_mod **birdb_getmodlist (birdb *, int *);
void birdb_freemodlist (struct birdb_mod **);
int birdb_cfgparse (birdb *, const char *);
void birdb_freerec (struct birdb_rec *);
void birdb_free_br_other (br_other_t *);

const char *birdb_backend_getname (struct birdb_mod *);
const char *birdb_backend_getdesc (struct birdb_mod *);
void *birdb_backend_open (struct birdb_mod *, const char *, int, char *[]);
void birdb_backend_close (struct birdb_mod *, void *);
struct birdb_rec **birdb_backend_get
  (struct birdb_mod *, void *, struct birdb_rec *);
void birdb_backend_freegetres
  (struct birdb_mod *, void *, struct birdb_rec **);
int birdb_backend_ins (struct birdb_mod *, void *, struct birdb_rec *);
int birdb_backend_del (struct birdb_mod *, void *, struct birdb_rec *);
struct birdb_rec *birdb_backend_seqgetfirst (struct birdb_mod *, void *, struct birdb_rec *);
struct birdb_rec *birdb_backend_seqgetnext
  (struct birdb_mod *, void *, struct birdb_rec *);
void birdb_backend_seqfree (struct birdb_mod *, void *, struct birdb_rec *);

#endif /* _BIRDB_H_ */
