/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * PAM related function for Biometric Manager
 * Copyright (C) Josef Hajas 2006 <josef@hajas.net>
 * 
 * biom-pam.c is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * callbacks.c is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with callbacks.c.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include "biom.h"

static pam_handle_t *pam_handle = NULL;
static int pipe_from_pam = -1;
static int pipe_to_pam = -1;

static BmMessageStyle
pam_style_to_bm_style (int pam_style)
{
        BmMessageStyle style;

        switch (pam_style) {
        case PAM_PROMPT_ECHO_ON:
                style = BM_AUTH_MESSAGE_PROMPT_ECHO_ON;
                break;
        case PAM_PROMPT_ECHO_OFF:
                style = BM_AUTH_MESSAGE_PROMPT_ECHO_OFF;
                break;
        case PAM_ERROR_MSG:
                style = BM_AUTH_MESSAGE_ERROR_MSG;
                break;
        case PAM_TEXT_INFO:
                style = BM_AUTH_MESSAGE_TEXT_INFO;
                break;
        default:
                g_assert_not_reached ();
                break;
        }

        return style;
}

static void
whack_child_fds (void)
{
        if (pipe_to_pam > 0)
                close (pipe_to_pam);
        pipe_to_pam = -1;
        if (pipe_from_pam > 0)
                close (pipe_from_pam);
        pipe_from_pam = -1;
}

void 
bm_fdprintf (int fd, const gchar *format, ...)
{
        va_list args;
        gchar *s;
        int written, len;

        va_start (args, format);
        s = g_strdup_vprintf (format, args);
        va_end (args);

        len = strlen (s);

        if (len == 0) {
                g_free (s);
                return;
        }

        written = 0;
        while (written < len) {
                int w;
                w = write (fd, &s[written], len - written);
                if (w < 0)
                        /* evil! */
                        break;
                written += w;
        }

        g_free (s);
}

static gboolean
child_ctrl_handler (GIOChannel *source,
                      GIOCondition cond,
                      gpointer data)
{
    //g_message ("child_ctrl_handler");

    gchar buf[PIPE_SIZE];
    //gchar *p;
    gsize len;
    gboolean res = TRUE;
    struct pam_closure *c = data;
    char *response = NULL;

    /* If this is not incoming i/o then return */
    if (cond != G_IO_IN)
      return TRUE;
    /* Read random garbage from i/o channel until first STX is found */
    do {
      g_io_channel_read_chars (source, buf, 1, &len, NULL);

      if (len != 1)
        return TRUE;
    } while (buf[0] && buf[0] != STX);

    memset (buf, '\0', sizeof (buf));
    if (g_io_channel_read_chars (source, buf, sizeof (buf) - 1, &len, NULL) !=
        G_IO_STATUS_NORMAL)
      return TRUE;

    /*p = memchr (buf, STX, len);
    if (p != NULL) {
      len = p - buf;
      g_io_channel_seek_position (source, -((sizeof (buf) - 1) - len), G_SEEK_CUR, NULL);
      memset (buf + len, '\0', (sizeof (buf) - 1) - len);
    }*/
    buf[len - 1] = '\0';

    res = c->cb_func ((BmMessageStyle) buf[0], buf + 1, &response, c->cb_data);

    if ((response != NULL) && (pipe_to_pam != -1)) {
	bm_fdprintf (pipe_to_pam, "%c%s\n", STX, response);
    } 

    return res;
}

static gboolean
get_response_from_parent (
                  struct pam_response      *resp)
{
	gchar    response[PIPE_SIZE];
	gboolean ret = FALSE;
	gsize    length;

	if (fgets(response, sizeof(response), stdin) != NULL) {
	    if (response[0] == STX) {
		response[PIPE_SIZE] = '\0';	/* NULL terminate */
		length = strlen(response);
		response[length - 1] = '\0';	// without end of line
		resp->resp = strdup (response + 1);
		resp->resp_retcode = 0;
		ret = TRUE;
	    } else {
		g_warning ("Wrong pam_bioapi protocol! Got: %s",
		       response);
		ret = FALSE;
	    }
	}
	return ret;
}


static int
pam_conversation (int                        nmsgs,
                  const struct pam_message **msg,
                  struct pam_response      **resp,
                  void                      *closure)
{
        int                  i = 0;
        gboolean             res;
        int                  ret;
	BmMessageStyle       style;

        res = TRUE;
        ret = PAM_SUCCESS;

        for (i = 0; i < nmsgs; i++) {

	      style = pam_style_to_bm_style (msg[i]->msg_style);
	      printf ("%c%c%s\n", STX, style, msg[i]->msg);
	      fflush (stdout);

	      resp[i] = (struct pam_response *)
		    malloc(sizeof(struct pam_response));
	      res = get_response_from_parent (resp[i]);

	      /* If the handler returns FALSE - interrupt the PAM stack */
	      if (res) {
		      resp [i]->resp_retcode = PAM_SUCCESS;
	      } else {
		      resp [i]->resp_retcode = PAM_INCOMPLETE;
	      }

        }

        return ret;
}

static gboolean
close_pam_handle (int status)
{

        if (pam_handle != NULL) {
                int status2;

                status2 = pam_end (pam_handle, status);
                pam_handle = NULL;

        }

        return TRUE;
}

static gboolean
create_pam_handle (const char      *username,
                   int             *status_code)
{
        int         status;
        const char *service = PAM_SERVICE_NAME;
        gboolean    ret;
        struct pam_conv    conv;

        conv.conv = &pam_conversation;
        conv.appdata_ptr = NULL;

	if (pam_handle != NULL) {
		g_warning ("create_pam_handle: Stale pam handle around, cleaning up");
                close_pam_handle (PAM_SUCCESS);
	}

	/* init things */
	pam_handle = NULL;
        status = -1;
        ret = TRUE;

	/* Initialize a PAM session for the user */
	if ((status = pam_start (service, username, &conv, &pam_handle)) != PAM_SUCCESS) {
		pam_handle = NULL;

                if (status_code != NULL) {
                        *status_code = status;
                }

                ret = FALSE;
                goto out;
	}

        ret = TRUE;

 out:
        if (status_code != NULL) {
                *status_code = status;
        }

        return ret;
}

gboolean
bm_pam_enroll (const char       *username,
                     BmMessageFunc func,
                     gpointer          data)
{
  	gint pipe1[2], pipe2[2];
	gint pid;
	gint status = -1;
	struct pam_closure *c = malloc (sizeof (struct pam_closure));
	gboolean res;
	GIOChannel *ctrlch;
	
	/* Open a pipe for communications */
	if G_UNLIKELY (pipe (pipe1) < 0) {
	    g_warning (_("Can't init pipes"));
	    _exit(1);
	}

	if G_UNLIKELY (pipe (pipe2) < 0) {
	    close (pipe1[0]);
	    close (pipe1[1]);
	    g_warning (_("Can't init pipes"));
	    _exit(1);
	}
	pid = fork ();
	switch (pid) {
	  /* child */
	  case 0:
	    close (pipe1[1]);
	    close (pipe2[0]);
	    dup2 (pipe1[0], STDIN_FILENO);
	    dup2 (pipe2[1], STDOUT_FILENO);

	    res = create_pam_handle (username, &status); 
	    if (! res) {
	      g_warning ("pam_start returned error!");
	      return res;
	    }
	    res = pam_chauthtok (pam_handle, 0);
	    /*if (! res) {
	      g_warning ("pam_chauthok returned error!");
	    }*/

	    close_pam_handle (status); 
	    _exit (res);

	    break;
	  case -1:
	    g_warning (_("Can't run child performing pam operations."));
	    _exit (1);
	    break;
	  /* parent */
	  default:

	    close (pipe1[0]);
	    close (pipe2[1]);

	    whack_child_fds ();

	    gui_t *gui = data;
	    gui->pipe_to_pam = pipe_to_pam = pipe1[1];
	    pipe_from_pam = pipe2[0];

	    gui->pam_child_pid = pid;

	    c->username = username;
	    c->cb_data = data;
	    c->cb_func = func;

	    ctrlch = g_io_channel_unix_new (pipe_from_pam);
	    g_io_channel_set_encoding (ctrlch, NULL, NULL);
	    g_io_channel_set_buffered (ctrlch, TRUE);
	    g_io_channel_set_flags (ctrlch,
				    g_io_channel_get_flags (ctrlch) | G_IO_FLAG_NONBLOCK,
				    NULL);
	    g_io_add_watch (ctrlch, 
			    /*G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP | G_IO_NVAL,*/
			    G_IO_IN,
			    (GIOFunc) child_ctrl_handler,
			    c);
	    g_io_channel_unref (ctrlch);

	    res = TRUE;

	    break;
	}
	return res;
}

