#!/usr/bin/env python
# -*- coding: iso-8859-1 -*-

##  This file is part of orm, The Object Relational Membrane Version 2.
##
##  Copyright 2002-2006 by Diedrich Vorberg <diedrich@tux4web.de>
##
##  All Rights Reserved
##
##  For more Information on orm see the README file.
##
##  This program is free software; you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation; either version 2 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program; if not, write to the Free Software
##  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
##
##  I have added a copy of the GPL in the file gpl.txt.

#
# $Log: datatypes.py,v $
# Revision 1.1  2006/04/28 08:45:50  diedrich
# Initial commit.
#
#

"""
This module defines a number of datatypes (dbattributes) for
particular purposes.
"""

from orm2.datatypes import Unicode
from orm2.validators import *
from orm2.sql import unicode_literal

class idna_literal(unicode_literal):
    """
    SQL literal class for <b>valid</b> Unicode (idna) domain names and
    email addresses.
    """
    def __sql__(self, ds):
        if "@" in self.content: # e-Mail address
            local, remote = split(self.content, "@")
            local = local.encode("ascii")
            remote = remote.encode("idna")

            s = "%s@%s" % ( local, remote, )
            s = ds.escape_string(s)
            sql = ds.string_quotes(s)
        else:
            s = self.content.encode("idna")
            s = ds.escape_string(s)
            sql = ds.string_quotes(s)

        return sql
            
            

class PDomain(Unicode):
    """
    Just like orm2.datatypes.Unicode, except that it doesn't use the
    backend's charset to convert the Unicode string, but Python's idna
    (Internationalized Domain Names in Applications) codec which takes
    care of lowercasing and punicode representation and so on.
    """
    sql_literal_class = idna_literal
    
    def __init__(self, column=None, title=None, validators=(),
                 widget_specs=(), has_default=False):
        validators = list(validators)
        validators.append(idna_fqdn_validator())
        
        Unicode.__init__(self, column, title, validators,
                         widget_specs, has_default)

    def __convert__(self, value):
        if type(value) is not UnicodeType:
            raise TypeError(
                "You must set a PDomain property to a unicode value!")
        else:
            return value


class PEMail(Unicode):
    """
    Like PDomain above, but for e-Mail addresses. The local part will be 
    checked against a regular expression, the remote part will be treated
    like a domain name by the PDomain class above.
    """
    sql_literal_class = idna_literal
    
    def __init__(self, column=None, title=None, validators=(),
                 widget_specs=(), has_default=False):
        validators = list(validators)
        validators.append(idna_email_validator)
        
        Unicode.__init__(self, column, title, validators,
                         widget_specs, has_default)


    def __convert__(self, value):
        if type(value) is not UnicodeType:
            raise TypeError(
                "You must set a PEMail property to a unicode value!")
        else:
            return value
        

# Local variables:
# mode: python
# ispell-local-dictionary: "english"
# End:

