#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

import logging
import logging.handlers
import os
import signal
import sys

from database import database
from ..common import version
from webservice.webservice import Webservice
from webservice.engineservice import OpenCloneEngineServiceImpl
from config import config

silent_exit = False

def main():
    global silent_exit
    
    # Register cleanup function
    signal.signal(signal.SIGTERM, cleanup)
    
    # Read config file
    config.read('/etc/openclone/webservice.conf')
    
    # Setup logging
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.NOTSET)
    handler = logging.handlers.TimedRotatingFileHandler(config.get('logging', 'logfile'), 'midnight', 1, int(config.get('logging', 'keep_days')))
    formatter = logging.Formatter('%(asctime)s %(name)s %(levelname)s %(message)s')
    handler.setFormatter(formatter)
    root_logger.addHandler(handler)
    
    logger = logging.getLogger('main')
    
    try:
        logger.info('OpenClone Webservice ' + version.version_long_str)
        
        # Set configured loglevel
        loglevels = {'debug': logging.DEBUG,
                     'info': logging.INFO,
                     'warning': logging.WARNING,
                     'error': logging.ERROR,
                     'critical': logging.CRITICAL}
        
        root_logger.setLevel(loglevels.get(config.get('logging', 'loglevel'), logging.NOTSET))
        
        # Initialize database
        database.initialize(config.get('database', 'db_url'))
        logger.info('Database initialized')
        
        # Setup service container
        webserv = Webservice(config.get('webservice', 'listen_addr'),
                             config.getint('webservice', 'listen_port'))
        
        webserv.addService(OpenCloneEngineServiceImpl())
        
        logger.info('Forking into background...')
        
        # Fork daemon process
        pid = os.fork()
        if pid > 0:
            silent_exit = True
            sys.exit(0)
        
        # Setup environment
        os.chdir('/')
        os.umask(022)
        os.setsid()
        
        # Second fork
        pid = os.fork()
        if pid > 0:
            silent_exit = True
            sys.exit(0)
        
        # Write pid file
        pidfile = config.get('common', 'pidfile')
        if pidfile != "":
            f = open(pidfile, 'w')
            f.write('%d\n' % os.getpid())
            f.close()
        
        # Setup standard input, output, error
        null_read = open('/dev/null', 'r')
        null_write = open('/dev/null', 'w')
        os.dup2(null_read.fileno(), sys.stdin.fileno())
        os.dup2(null_write.fileno(), sys.stdout.fileno())
        os.dup2(null_write.fileno(), sys.stderr.fileno())
        
        logger.info('Starting service container...')
        webserv.run()
    
    except Exception, e:
        logger.exception(e)
        raise
    
    finally:
        if not silent_exit:
            logger.error('Unexpectedly reached end of code. Exiting...')
    
    return 0

def cleanup(signum, frame):
    global silent_exit
    
    if signum == signal.SIGTERM:
        logging.getLogger().setLevel(logging.NOTSET)
        logging.getLogger('main').info('Signal SIGTERM received. Exiting...')
        
        # Remove pid file
        pidfile = config.get('common', 'pidfile')
        if os.path.exists(pidfile):
            os.remove(pidfile)
    
    else:
        logging.getLogger('main').error('Unexpected signal received. Exiting...')
    
    # TODO: Kill remaining udpcast processes
    
    silent_exit = True
    sys.exit(0)

if __name__ == "__main__":
    main()
