#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

from __future__ import with_statement
import hashlib
import logging
import threading
import os

logger = logging.getLogger('hash')

class Hash(threading.Thread):
    hash = None
    hash_lock = None
    input_r = None
    input_w = None
    output_r = None
    output_w = None
    status = None
    status_lock = None
    
    def __init__(self, algorithm, **kw):
        threading.Thread.__init__(self)
        
        if 'input' in kw:
            self.input_r = kw['input']
        
        else:
            r, w = os.pipe()
            self.input_r = os.fdopen(r, 'r')
            self.input_w = os.fdopen(w, 'a')
        
        if 'output' in kw:
            self.output_w = kw['output']
        
        else:
            r, w = os.pipe()
            self.output_r = os.fdopen(r, 'r')
            self.output_w = os.fdopen(w, 'a')
        
        self.hash = hashlib.new(algorithm)
        self.hash_lock = threading.RLock()
        
        self.status = {}
        self.status_lock = threading.RLock()
        
        self.start()
    
    def run(self):
        while True:
            #os.read(self.input_r, 1024)
            data = self.input_r.read(1024)
            if data == '':
                break
            
            with self.hash_lock:
                self.hash.update(data)
            
            self.output_w.write(data)
        
        with self.hash_lock:
            logger.debug('Stream end. Computed hash %s' % self.hash.hexdigest())
        
        self.output_w.close()
        
        with self.status_lock:
            self.status['success'] = True
        
        logger.debug('success: True')
    
    def getStdin(self):
        return self.input_w
    
    def getStdout(self):
        return self.output_r
    
    def getStatus(self):
        with self.status_lock:
            return dict(self.status)
    
    def digest(self):
        with self.hash_lock:
            return self.hash.digest()
    
    def hexdigest(self):
        with self.hash_lock:
            return self.hash.hexdigest()
    
    def wait(self, timeout=None):
        self.join(timeout)
