/*
 * OPALE is a scientific library under LGPL. Its main goal is to
 * develop mathematical tools for any scientist.
 *
 * Copyright (C) 2002 Opale Group
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * You can visit the web site http://opale.tuxfamily.org to obtain more
 * informations about this program and/or to contact the authors by mail
 * developers@opale.tuxfamily.org.
 */



package opale.mathtools.parser;

import java.lang.reflect.*;
import java.util.*;
import java.io.*;

/**
* This class can parse a string definition of a function and  return its value. It's possible to choose the independent variable that must be recognized (for instance 'x', 'y', 'x1' etc...). 
* <P> The recognised operations are : +, -, *, /, (, ), ^
* <P> The recognised functions are : 
* <UL>
* <LI> sin, cos, tan
* <LI> asin, acos, atan
* <LI> log, log10, exp
* <LI> sqrt, abs, rnd
* <LI> deg, rad
* </UL>
* Moreover, to add functions is possible (<code>void addFunc(UnaryFunction f)</code>). 
* <P> The recognised constants are : Pi, E
* <P> This parser is not case sensitive: "e" and "E" will be interpreted as the string "e".
 
* @author O.C.
*/

public class Parser 
{
private final static boolean DEBUG=opale.tools.Debug.On;

private Hashtable les_func;
private Hashtable les_var;
private Hashtable les_cst;
private int state = 0;
private int flag_deg_rad = 1; // 0 = deg, 1 = rad 

private static final int ADD          =  '+';
private static final int SUBTRACT     =  '-';
private static final int DIVIDE       =  '/';
private static final int MULTIPLY     =  '*';
private static final int POWER        =  '^';
private static final int GRP        =  '(';
private static final int ENDGRP        =  ')';

private static final  Operator OP_ADD = new Operator(Parser.precedence(ADD),ADD);
private static final  Operator OP_MUL = new Operator(Parser.precedence(MULTIPLY),MULTIPLY);
private static final  Operator OP_DIV = new Operator(Parser.precedence(DIVIDE),DIVIDE);
private static final  Operator OP_SUB = new Operator(Parser.precedence(SUBTRACT),SUBTRACT);
private static final  Operator OP_GRP = new Operator(0,GRP);
private static final  Operator OP_ENDGRP = new Operator(0,ENDGRP);
private static final  Operator OP_POW = new Operator(Parser.precedence(POWER),POWER);

private  PNode root;
private String str;

/**
* Instantiate an object of this  class
*/
public Parser()
	{
	root = null;
	
	les_func = new Hashtable(20);
	init_func();
	
	les_var = new Hashtable(5);
	
	les_cst = new Hashtable(5);
	init_cst();
	
	str = "0";
	}

/**
* Instantiate an object of this  class with specifying a string to be parsed
* @param s, the string that will be parsed.
*/
public Parser(String s)
	{
	root = null;
	
	les_func = new Hashtable(20);
	init_func();
	
	les_var = new Hashtable(5);
	
	les_cst = new Hashtable(5);
	init_cst();

	if (!s.equals("")) str = new String(s);
	else str ="0";
	}

/**
* Parse a string
* @param s, the string that will be parsed.
*/
public void parseString(String s)
	{
	if (!s.equals("")) str = new String(s);
	else str ="0";
	parseString();
	}
	
/**
* Parse the  string defined in this class.
*/
public void parseString()
	{
	Stack st_ob = new Stack();
	Stack st_op = new Stack();
	
	StringReader sr = new StringReader(str);
	StreamTokenizer tokenizer = new StreamTokenizer(sr);

	tokenizer.parseNumbers();
	tokenizer.lowerCaseMode(true);
	tokenizer.ordinaryChar(ADD);
	tokenizer.ordinaryChar(SUBTRACT);
	tokenizer.ordinaryChar(MULTIPLY);
	tokenizer.ordinaryChar(DIVIDE);
	tokenizer.ordinaryChar(POWER);

	try
	{
	state = 0;

	int type;
	while ( (type=tokenizer.nextToken()) != StreamTokenizer.TT_EOF)
		{
		switch (state)
		{
		case 0:
			parserState0(tokenizer,st_ob,st_op);
			break;
		case 1:
			parserState1(tokenizer,st_ob,st_op);
			break;
		case 2:
			parserState2(tokenizer,st_ob,st_op);
			break;
		case 3:
			break;
		default :
			break;
		}

	if (DEBUG) System.out.println("op = "+st_op);
	if (DEBUG) for (int i=0;i<st_ob.size();i++) ((PNode) st_ob.get(i)).print(0);

		}
	parserState2(tokenizer,st_ob,st_op);

	if (DEBUG) System.out.println("op = "+st_op);
	if (DEBUG) ((PNode) st_ob.get(0)).print(0);

	root = (PNode) st_ob.get(0);
	}
	catch (IOException evt)
	{
	System.err.println(evt);
	}
	}
	

/**
* Evaluate the function represented by the string.
* @return double, the value
*/
public double eval()
	{
	return eval(this.root);
	}


/**
* Add a function for the string analyser.
* @param UnaryFunction f, a function that must be recognized in a string
*/
public void addFunc(UnaryFunction f)
	{
	f.setParser(this);
	les_func.put(f.keyword().toLowerCase(),f);
	}

/**
* Add a variable.
* @param String, the keyword of the variable that must be parsed (x, y, x2, a etc...).
*/
public void addVar(String name)
	{
	les_var.put(name.toLowerCase(),new Variable());
	}

/**
* Set the value of a variable.
* @param String, the keyword of the variable to set.
* @param double, the value of this variable.
*/
public void setVar(String name, double value)
	{
	Variable x = (Variable) les_var.get(name.toLowerCase());
	if (x == null)	throw new ParseFunctionException("method setVar : Unknown variable "+name+" !!!");
	x.setValue(value);
	}

/**
* Set the value of a variable.
* @param String, the keyword of the variable to set.
* @param String, a string representation of the value.
*/
public void setVar(String name, String value)
	{
	Variable x = (Variable) les_var.get(name.toLowerCase());
	if (x == null)	throw new ParseFunctionException("method setVar : Unknown variable "+name+" !!!");
	x.setValue(value);
	}


/**
* Get the value of a variable.
* @param String, the keyword of a variable.
* @return double, the value of this variable.
*/
public double getVar(String name)
	{
	Variable x = (Variable) les_var.get(name.toLowerCase());
	if (x == null)	throw new ParseFunctionException("method getVar : Unknown variable "+name+" !!!");
	return x.getValue();
	}

/**
* Add a constant.
* @param String, the keyword of the constant.
* @param double, the value of this constant.
*/
public void addCst(String name, double value)
	{
	les_cst.put(name.toLowerCase(),new Cst(value));
	}

	
//mthodes prives :
	

private static Operator getOp(int op)
	{
	switch (op)
		{
		case ADD:
		return OP_ADD;
		case SUBTRACT:
		return OP_SUB;
		case MULTIPLY:
		return OP_MUL;
		case DIVIDE:
		return OP_DIV;
		case POWER:
		return OP_POW;
		case GRP:
		return OP_GRP;
		case ENDGRP:
		return OP_ENDGRP;
		}
	return null;
	}
		
private double eval(PNode node)
	{
        double value = 0.;
	
	if (node == null) throw new  ParseFunctionException("method eval : null node !!!");
	
        switch (node.type) 
		{
		case PNode.OP:   
			value = evalOp(node);
			break;
		case PNode.VALUE:
			value = node.value;
			break;
		case PNode.FUNCTION:
			value = evalFunc(node);
			break;
		case PNode.VAR:
			value = evalVar(node);
			break;
		default:
			throw new ParseFunctionException("method eval : Unknown type for this node !!!");
		}
		
	
	return value;
	}

private double evalOp(PNode node)
	{
	double value = 0.;
	
	switch (node.op)
		{
		case ADD:
			if(node.left != null) value = eval(node.left);
			value += eval(node.right);
        	 	break;
		case SUBTRACT:
			if(node.right != null) value = eval(node.right);
			value = eval(node.left) - value;
        	 	break;
		case DIVIDE:
			value = eval(node.left);
			value /= eval(node.right);
			break;
		case MULTIPLY:
			value = eval(node.left);
			value *= eval(node.right);
			break;
		case POWER:
			value = Math.pow(eval(node.left),eval(node.right));
			break;
		default:
			throw new ParseFunctionException("method evalOp : Unknown operator !!!");
		}
	return value;
	}

private double evalFunc(PNode node)
	{
	double value = 0.;

	UnaryFunction f = (UnaryFunction) les_func.get(node.svalue.toLowerCase());
	if (f == null)	throw new ParseFunctionException("method evalFunc : Unknown function !!!");
	value = f.eval(eval(node.left));
	return value;
	}

private double evalVar(PNode node)
	{
	double value = 0.;
	//System.out.println("var = "+node.svalue);
	Variable x = (Variable) les_var.get(node.svalue);
	if (x == null)	throw new ParseFunctionException("method evalVar : Unknown variable "+node.svalue+" !!!");
	value = x.getValue();
	return value;
	}


private static int precedence(int op)
	{
	switch (op)
		{
		case ADD:
			return 1;
		case SUBTRACT:
			return 2;
		case MULTIPLY:
			return 3;
		case DIVIDE:
			return 3;
		case POWER:
			return 5;
		}
	return -1;
	}
			

private void parserState0(StreamTokenizer tokenizer, Stack ob, Stack op) throws IOException
	{
	try {
		switch (tokenizer.ttype)
		{
		case StreamTokenizer.TT_WORD :
			//if (DEBUG) System.out.println("s = "+tokenizer.sval);
			if (DEBUG) System.out.println(tokenizer.sval);
			Variable x = (Variable) les_var.get(tokenizer.sval);
			if (x != null)
				{
				PNode node = new PNode();
				node.type = PNode.VAR;
				node.svalue = tokenizer.sval;
				ob.push(node);
				state = 2;
				}
			else 
				{
				UnaryFunction f = (UnaryFunction) les_func.get(tokenizer.sval);
				if (f!=null)
					{
					op.push(f);
					state = 0;
					}
				else
					{
					Cst c = (Cst) les_cst.get(tokenizer.sval);
					if (c!=null)
						{
						PNode node = new PNode();
						node.type = PNode.VALUE;
						node.value = c.getValue();
						ob.push(node);
						state = 2;
						}
					else throw new ParseFunctionException("Unknown keyword "+tokenizer.sval+" !!");
					
					}
				}

			break;
		case StreamTokenizer.TT_NUMBER :
			//System.out.println("nb = "+tokenizer.nval);
			if (DEBUG) System.out.println(tokenizer.nval);
			PNode node = new PNode();
			node.type = PNode.VALUE;
			node.value = tokenizer.nval;
			ob.push(node);
			state = 2;
			break;
		case   ADD :
			if (DEBUG) System.out.println("plus unaire");
			op.push(getOp(ADD));
			node = new PNode();
			node.type = PNode.VALUE;
			node.value = 0.;
			ob.push(node);
			state = 1;
			break;
		case   SUBTRACT :
			if (DEBUG) System.out.println("moins unaire");
			op.push(getOp(SUBTRACT));
			node = new PNode();
			node.type = PNode.VALUE;
			node.value = 0.;
			ob.push(node);
			state = 1;
			break;
		case   GRP :
			if (DEBUG) System.out.println("(");
			op.push(getOp(GRP));
			state = 0;
			break;
		default:
			if (DEBUG) System.out.println(tokenizer.ttype);
			throw new ParseFunctionException("parsing this string fails !!");
		}
	}
	catch (java.util.EmptyStackException evt)
	{
	throw new  ParseFunctionException("Missing arguments !!!");
	}

	}
	
private void parserState1(StreamTokenizer tokenizer, Stack ob, Stack op) throws IOException
	{
	try {	
		switch (tokenizer.ttype)
		{
		case StreamTokenizer.TT_WORD :
			//System.out.println("s = "+tokenizer.sval);
			if (DEBUG) System.out.println(tokenizer.sval);
			Variable x = (Variable) les_var.get(tokenizer.sval);
			if (x != null)
				{
				PNode node = new PNode();
				node.type = PNode.VAR;
				node.svalue = tokenizer.sval;
				ob.push(node);
				state = 2;
				}
			else 
				{
				UnaryFunction f = (UnaryFunction) les_func.get(tokenizer.sval);
				if (f!=null)
					{
					op.push(f);
					state = 0;
					}
				else
					{
					Cst c = (Cst) les_cst.get(tokenizer.sval);
					if (c!=null)
						{
						PNode node = new PNode();
						node.type = PNode.VALUE;
						node.value = c.getValue();
						ob.push(node);
						state = 2;
						}
					else throw new ParseFunctionException("Unknown keyword "+tokenizer.sval+" !!");
					
					}
				}
			break;
		case StreamTokenizer.TT_NUMBER :
			//System.out.println("nb = "+tokenizer.nval);
			if (DEBUG) System.out.println(tokenizer.nval);
			PNode node = new PNode();
			node.type = PNode.VALUE;
			node.value = tokenizer.nval;
			ob.push(node);
			state = 2;
			break;
		case   GRP :
			if (DEBUG) System.out.println("(");
			op.push(getOp(GRP));
			state = 0;
			break;
		default:
			if (DEBUG) System.out.println(tokenizer.ttype);
			throw new ParseFunctionException("parsing this string fails !!");
		}
	}
	catch (java.util.EmptyStackException evt)
	{
	throw new  ParseFunctionException("Missing arguments !!!");
	}
	}
	
private void parserState2(StreamTokenizer tokenizer, Stack ob, Stack op) throws IOException
	{
	try {
	Operator op_read;
		switch (tokenizer.ttype)
		{
		case   ADD: case SUBTRACT : case MULTIPLY: case DIVIDE: case POWER:  
			if (DEBUG) System.out.println((char) tokenizer.ttype);
			op_read = getOp(tokenizer.ttype);
			if (!op.empty()) 
				{
				TypeOp tmpi = ((TypeOp) op.peek());
				while (tmpi.precedence() >= op_read.precedence() )
					{
					if (tmpi instanceof Operator)
						{
						Operator tmp = ((Operator) op.pop());
						if (DEBUG) System.out.println("depile = "+tmp);
						PNode deux = (PNode) ob.pop();
						PNode un = (PNode) ob.pop();
						PNode node = new PNode();
						node.type=PNode.OP;
						node.op=tmp.op;
						node.left = un;
						node.right=deux;
					
						ob.push(node);
						}
					else if (tmpi instanceof UnaryFunction)
						{
						UnaryFunction tmp = ((UnaryFunction) op.pop());
						if (DEBUG) System.out.println("depile = "+tmp);
						PNode un = (PNode) ob.pop();
						PNode node = new PNode();
						node.type=PNode.FUNCTION;
						node.svalue=tmp.keyword();
						node.left = un;
						node.right=null;
					
						ob.push(node);
						}
					
					if (!op.empty()) tmpi = ((TypeOp) op.peek());
					else break ;
					}
				}
			op.push(op_read);
			state =1;
			break;
		case   ENDGRP :
			{
			if (DEBUG) System.out.println(")");
			op_read = getOp(tokenizer.ttype);
			if (!op.empty())
				{
				TypeOp tmp = ((TypeOp) op.pop());
				while (!tmp.keyword().equals("("))
					{
					if (tmp instanceof Operator)
						{
						Operator tmpi = (Operator) tmp;
						if (DEBUG) System.out.println("depile = "+tmpi);
						PNode deux = (PNode) ob.pop();
						PNode un = (PNode) ob.pop();
						PNode node = new PNode();
						node.type=PNode.OP;
						node.op = tmpi.op;
						node.left = un;
						node.right=deux;
					
						ob.push(node);
						}
					else if (tmp instanceof UnaryFunction)
						{
						UnaryFunction tmpi = (UnaryFunction) tmp;
						if (DEBUG) System.out.println("depile = "+tmpi);
						PNode un = (PNode) ob.pop();
						PNode node = new PNode();
						node.type=PNode.FUNCTION;
						node.svalue = tmpi.keyword();
						node.left = un;
						node.right=null;
					
						ob.push(node);
						}
					
					if (!op.empty()) tmp = ((TypeOp) op.pop());
					else break ;
					}
				}
			state = 2;
			break;
			}
		case StreamTokenizer.TT_EOF :
			if (DEBUG) System.out.println("EOF");
				while (!op.empty())
					{
					TypeOp tmp = (TypeOp) op.pop();
					if (tmp instanceof Operator)
						{
						Operator tmpi = ((Operator) tmp);
						if (DEBUG) System.out.println("depile = "+tmpi);
						PNode deux = (PNode) ob.pop();
						PNode un = (PNode) ob.pop();
						PNode node = new PNode();
						node.type=PNode.OP;
						node.op=tmpi.op;
						node.left = un;
						node.right=deux;
					
						ob.push(node);
						}
					else if (tmp instanceof UnaryFunction)
						{
						UnaryFunction tmpi = ((UnaryFunction) tmp);
						PNode un = (PNode) ob.pop();
						PNode node = new PNode();
						node.type=PNode.FUNCTION;
						node.svalue=tmpi.keyword();
						node.left = un;
						node.right=null;
					
						ob.push(node);
						}
						
					}
			state = 0;
			break;
		default:
			if (DEBUG) System.out.println(tokenizer.ttype);
			if (DEBUG) System.out.println(tokenizer.sval);
			throw new ParseFunctionException("parsing this string fails !!");
		}
	}
	catch (java.util.EmptyStackException evt)
	{
	throw new  ParseFunctionException("Missing arguments !!!");
	}
	
	
	}
	
private void init_cst()
	{
	addCst("pi",Math.PI);
	addCst("e",Math.E);
	}
	
	
private void init_func()
	{
	addFunc(new Sin());
	addFunc(new Cos());
	addFunc(new Tan());
	addFunc(new Log());
	addFunc(new Log10());
	addFunc(new Exp());
	addFunc(new Sqrt());
	addFunc(new Abs());
	addFunc(new Acos());
	addFunc(new Asin());
	addFunc(new Atan());
	addFunc(new Rnd());
	addFunc(new Rad());
	addFunc(new Deg());
	}
	
public void setRad() { flag_deg_rad = 1; }
public void setDeg() { flag_deg_rad = 0; }
public boolean isDeg() { if (flag_deg_rad == 0 ) return true; else return false; }

public static void main(String[] arg)
	{
	//Parser parser = new Parser("4-2*sin(x-3.14)*2");
	//Parser parser = new Parser("1-2*3*3-2*2+3-2*3+2*2*2/8-1*1*2/1-(2*3+1)");
	//Parser parser = new Parser("-1-2*x-(-2)-(-3*(-4))*2");
	//Parser parser = new Parser("-e^(-2+cos(4))");
	//Parser parser = new Parser("-1.3*(-3)-2*1");
	//Parser parser = new Parser("cos(y)");
	Parser parser = new Parser("cos(myFunc(x)^2-2*myFunc(2))");
	parser.addVar("X");
	parser.addVar("y");
	
	parser.setVar("x",1);
	parser.setVar("y","acos(0.5)");
	
	
	SUnaryFunction myFunc = new SUnaryFunction("MYfunc","x","3*x");
	parser.addFunc(myFunc);
	
	parser.parseString();
	System.out.println(parser.eval());
	}

}


class Operator extends TypeOp
	{
	
	int p;
	int op;
	
	Operator(int p, int o) { this.p = p; op = o; } 
	public String toString() { return ((char) op) + " " +precedence();}
	public String keyword() { return ""+((char) op);}
	public int precedence() { return this.p;}
	}
