package opale.matrix;
/*
Copyright (C) 2000  Groupe Opale (http://www.opale.ovh.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


 

import opale.tools.Debug;
import opale.mathtools.DVect;

/**
* This class implements a symmetrical matrix with real coefficients.
* @since Opale-Matrix 0.1
* @author J.M., O.C.
* @date 03/2001
*/
public final class  SymMatrix implements  Cloneable
{
// storage in one dimensional array :
// 0 
// 1 2
// 3 4 5
// ...
private double[] a;
private int n;	// dimension
private int size; // size of array a

/** 
* Construct an n X n matrix of zeros. 
* @param int n, row dimension.
*/

public SymMatrix (int n) 
	{
	this.n = n;
	size = n*(n+1)/2;
	a = new double[size];
	}

/**
* Get the dimension.
* @return int, the  dimension.
*/
public int getDim() { return n; }



/** 
* Get a coefficient of the matrix.
* @param int i, row index.
* @param int j, column index.
* @return double, coeff(i,j)
*/
public double get(int i, int j) 
	{
	if (Debug.On) Debug.assert((i<n) && (j<n));
	if (j<i) 
		return a[i*(i+1)/2+j];
	else
		return a[j*(j+1)/2+i];
	
	}
/** 
* Set a coefficient of the matrix.
* @param int i, row index.
* @param int j, column index.
* @param double x, the value of coeff(i,j)
*/
public void set(int i, int j, double x) 
	{
	if (Debug.On) Debug.assert((i<n) && (j<n));
	if (j<i) 
		a[i*(i+1)/2+j] = x;
	else
		a[j*(j+1)/2+i] = x;
	}

/** 
* Make a copy of the current matrix.
* @return Matrix, the copy
*/
public SymMatrix copy () 
	{
	SymMatrix C = new SymMatrix(n);
	for (int i = 0; i < size; i++)  C.a[i] = a[i];
	return C;
	}
	
	
/**
* Add an another matrix A to the current matrix.
* @param Matrix A, an another matrix
* @return Matrix, the result
*/
public void add(SymMatrix A)
	{
	if (Debug.On) Debug.assert((n == A.n));
	for (int i=0;i<n;i++)
		for (int j=0;j<=i;j++)
			a[i*(i+1)/2+j]+=A.a[i*(i+1)/2+j];
	}

public DVect mul(DVect x)
	{
	if (Debug.On) Debug.assert(n == x.size());
	
	DVect y = new DVect(n);
	
/*	for (int i=0;i<n;i++)
		{
		double s=0.;
		for (int j=0;j<=i;j++)
			s+=a[i*(i+1)/2+j]*(x.get(j)+x.get(n-j-1)); //faux!!
		y.set(i,s);
		}*/
	for (int i=0;i<n;i++)
		{
		double s=0.;
		for (int j=0;j<=i;j++)
			s+=a[i*(i+1)/2+j]*x.get(j); 
		for (int j=i+1;j<n;j++)
			s+=a[j*(j+1)/2+i]*x.get(j); 
		y.set(i,s);
		}
		
	return y;
	}

/**
* Get the internal one-dimensional array.
* Type of storage :
* 0 
* 1 2
* 3 4 5
* ..... 
*  @return double[],    Pointer to the array of matrix elements.
*/

public double[] getInternalArray() 
	{
	return a;
	}
	
/**
* Get a copy of array of matrix elements.
*  @return double[],  the array of matrix elements.
*/

public double[] getArrayCopy() 
	{
	double[] tab = new double[size];
	for (int i = 0; i < size; i++) 
			tab[i] = a[i];
	return tab;
	}


/**
* Print the matrix.
*/
public String toString()
	{
	int i,j,cpt=0;
	double aff;
	StringBuffer s=new StringBuffer(n*n);
	
	for (i = 0; i <n; i++) 
		{
		for ( j = 0;j<=i;j++) 
			{
			aff=a[cpt++];
	     		s.append(aff);
			s.append(" ");
			}
		s.append("\n");
		}
	return s.toString();
	}
	
	
/**
* Creates and return a copy of this object.
* @return Object a copy of this object.
* @exception  OutOfMemoryError if not enough memory.
* @see        java.lang.Cloneable
* @since Opale-Matrix 0.11
*/
public Object clone()
	{
	try
		{
		SymMatrix copy = (SymMatrix) super.clone();
		copy.n = n;
		copy.size = size;
		copy.a = getArrayCopy();
		
		return copy;
		}
	catch (CloneNotSupportedException e) // ne devrait jamais arriver
		{
		throw new InternalError();
		}
   	}
	

	
	
public static void main(String[] arg)
	{
	SymMatrix A = new SymMatrix(3);
	A.set(0,0,1);
	A.set(0,2,2);
	A.set(2,1,-15);
	SymMatrix B = (SymMatrix) A.clone();
	A.set(1,2,10);
	
	System.out.println(A);

	System.out.println(B);
	}
	
}

