/*
Copyright (C) 2000  Olivier Cioni

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You can visit the web site http://www.opale.ovh.org to obtain more informations about this program and/or to contact the coders.
*/


package opale.m2d.ker;
import opale.tools.*;
import java.io.*;

/**
* Cette classe modlise un repre (O;I;J) quelconque dfinit soit dans un autre repre soit dans le repre absolu.
* En outre on trouve dans cette classe des mthodes de calculs pour les normes et les distances dans le repre.
* @since Opale-2d 0.1
* @author O.C.
*/
public final class OIJ implements  Cloneable
{
static public OIJ OIJ_ABSOLUTE = new OIJ();
static 
	{
	OIJ_ABSOLUTE.setName("OIJ_ABSOLUTE");
	}

private String name;	// le nom du repere.
private OIJ rep;
protected double oriX,oriY;	//les coordonnes du centre.
protected double[] I, J;	//les vecteurs de base.


/**
* Constructeur par dfaut. Construit un repre orthonorm dans le repre absolu.
*/ 
public OIJ()
	{
	name = new String("Sans nom");
	rep = OIJ.OIJ_ABSOLUTE;
	oriX = oriY = 0.;
	I = new double[2];
	J = new double[2];
	I[0]=1;
	I[1]=0;
	J[0]=0;
	J[1]=1;
	}
	
/**
* Constructeur qui prend en argument les caractristiques du repre  construire dans le repre absolu..
* @param double oriX, oriY les coordonnes du centre.
* @param double Ix, Iy les coordonnes du premier vecteur de base.
* @param double Jx, Jy les coordonnes du deuxime vecteur de base.
*/
public OIJ(double oriX, double oriY, double Ix, double Iy, double Jx, double Jy) throws OIJException
	{
	name = new String("Sans nom");
	this.rep = OIJ.OIJ_ABSOLUTE;
	this.oriX = oriX;
	this.oriY = oriY;
	if ( Math.abs(Ix*Jy - Iy*Jx) > 1.e-15 )
		{
		I = new double[2];
		J = new double[2];
		I[0]=Ix;
		I[1]=Iy;
		J[0]=Jx;
		J[1]=Jy;
		}
	else throw new OIJException("Erreur  la construction d'un repre !");
	}

/**
* Constructeur qui prend en argument les caractristiques du repre  construire.
* @param OIJ rep, le repre de base.
* @param double oriX, oriY les coordonnes du centre.
* @param double Ix, Iy les coordonnes du premier vecteur de base.
* @param double Jx, Jy les coordonnes du deuxime vecteur de base.
*/
public OIJ(OIJ rep,double oriX, double oriY, double Ix, double Iy, double Jx, double Jy) throws OIJException
	{
	name = new String("Sans nom");
	this.rep = rep;
	this.oriX = oriX;
	this.oriY = oriY;
	if ( Math.abs(Ix*Jy - Iy*Jx) > 1.e-15 )
		{
		I = new double[2];
		J = new double[2];
		I[0]=Ix;
		I[1]=Iy;
		J[0]=Jx;
		J[1]=Jy;
		}
	else throw new OIJException("Erreur  la construction d'un repre !");
	}

/**
* Constructeur de copie.
* @param OIJ rep, le repere  copier;
*/
public OIJ(OIJ rep)
	{
	this.rep = rep.rep;
	this.oriX = rep.oriX;
	this.oriY = rep.oriY;
	I = new double[2];
	J = new double[2];
	I[0]=rep.I[0];
	I[1]=rep.I[1];
	J[0]=rep.J[0];
	J[1]=rep.J[1];
	}
	
/**
* Renvoi le nom donn au repre.
* @return String, le nom.
*/
public final String getName()
	{
	return name;
	}

/**
* Change le nom du repre.
* @param String name, le nom  donner.
*/
public final void setName(String name)
	{
	this.name = new String(name);
	}

/**
* Cette mthode teste si le repere est gal  un objet pass en argument.
* @param Object obj un objet  comparer avec le repere.
* @return <code>true</code> si l'objet  comparer est une instance de OIJ et est gal au repere courant; <code>false</code> sinon.
*/
public boolean equals(Object obj)
{
if (obj instanceof OIJ)
	{
	OIJ rep = (OIJ) obj;
	Matrix2D mv01 = new Matrix2D();
	Matrix2D mv02 = new Matrix2D();

	matPassageAbs(rep,mv01);
	matPassageAbs(this,mv02);

	return mv01.equals(mv02);
	}
return false;
}

/**
* Renvoie le carre de la norme d'un vecteur dans le repre.
* @param double x, double y les coordonnes du vecteur.
* @return double le carre de la norme.
*/
public final double normSq(double x, double y)
	{
	return x*x*(I[0]*I[0]+I[1]*I[1]) + 2*x*y*(I[0]*J[0]+I[1]*J[1]) + y*y*(J[0]*J[0]+J[1]*J[1]);
	}

/**
* Renvoie la norme d'un vecteur dans le repre.
* @param double x, double y les coordonnes du vecteur.
* @return double la norme.
*/
public final double norm(double x, double y)
	{
	return Math.sqrt(normSq(x,y));
	}


/**
* Renvoie le carre de la distance entre deux points dans le repre.
* @param double x1, double y1 les coordonnes du premier point.
* @param double x2, double y2 les coordonnes du deuxime point.
* @return double le carre de la distance.
*/
public final double distSq(double x1, double y1, double x2, double y2)
	{
	return normSq(x2-x1,y2-y1);
	}


/**
* Renvoie la distance entre deux points dans le repre.
* @param double x1, double y1 les coordonnes du premier point.
* @param double x2, double y2 les coordonnes du deuxime point.
* @return double la distance.
*/
public final double dist(double x1, double y1, double x2, double y2)
	{
	return norm(x2-x1,y2-y1);
	}


/**
* Calcule la matrice de passage et le vecteur de passage du repere de dfinition au repere courant.
* @param double[][] mat, la matrice calcule.
* @param double[] vect, le vecteur calcul.
*/
public void matPassage(Matrix2D mv)
	{

	mv.set(0,0,I[0]);
	mv.set(1,0,I[1]);
	mv.set(0,1,J[0]);
	mv.set(1,1,J[1]);

	mv.setV(0,oriX);
	mv.setV(1,oriY);
	}


/**
* Calcule la matrice de passage et le vecteur de passage du repere spcifi en argument au repere courant.
* @param OIJ rep, le repere de dpart.
* @param double[][] mat, la matrice calcule.
* @param double[] vect, le vecteur calcul.
*/
public void matPassage(OIJ rep,Matrix2D mv)
	{
	Matrix2D mv01 =new Matrix2D();
	Matrix2D mv02 =new Matrix2D();

	double[][] tmp = new double[2][2];

	matPassageAbs(rep,mv01);
	matPassageAbs(this,mv02);
	double det=mv01.get(1,1)*mv01.get(0,0) - mv01.get(0,1)*mv01.get(1,0);
	tmp[0][0] = mv01.get(1,1)/det;
	tmp[0][1] = -mv01.get(0,1)/det;
	tmp[1][0] = -mv01.get(1,0)/det;
	tmp[1][1] = mv01.get(0,0)/det;

	mv.setV(0,tmp[0][0]*(mv02.getV(0)-mv01.getV(0))+tmp[0][1]*(mv02.getV(1)-mv01.getV(1)));
	mv.setV(1,tmp[1][0]*(mv02.getV(0)-mv01.getV(0))+tmp[1][1]*(mv02.getV(1)-mv01.getV(1)));

	mv.set(0,0,tmp[0][0]*mv02.get(0,0)+tmp[0][1]*mv02.get(1,0));
	mv.set(0,1,tmp[0][0]*mv02.get(0,1)+tmp[0][1]*mv02.get(1,1));
	mv.set(1,0,tmp[1][0]*mv02.get(0,0)+tmp[1][1]*mv02.get(1,0));
	mv.set(1,1,tmp[1][0]*mv02.get(0,1)+tmp[1][1]*mv02.get(1,1));

	}

//public static void matPassage(OIJ rep1, OIJ rep2,  double[][] mat, double[] vect);

private static void matPassageAbs(OIJ rep, Matrix2D mv)
	{
	if (rep != OIJ.OIJ_ABSOLUTE)
		{
		mv.set(0,0,rep.I[0]);
		mv.set(1,0,rep.I[1]);
		mv.set(0,1,rep.J[0]);
		mv.set(1,1,rep.J[1]);

		mv.setV(0,rep.oriX);
		mv.setV(1,rep.oriY);

		if (rep.rep != OIJ.OIJ_ABSOLUTE)
			{
			Matrix2D mv2 =new Matrix2D();
			double[][] tmp= new double[2][2];	
			tmp[0][0] = mv.get(0,0);
			tmp[0][1] = mv.get(0,1);
			tmp[1][0] = mv.get(1,0);
			tmp[1][1] = mv.get(1,1);

			matPassageAbs(rep.rep,mv2);
			
			mv.set(0,0,mv2.get(0,0)*tmp[0][0] + mv2.get(0,1)*tmp[1][0]);
			mv.set(0,1,mv2.get(0,0)*tmp[0][1] + mv2.get(0,1)*tmp[1][1]);
			mv.set(1,0,mv2.get(1,0)*tmp[0][0] + mv2.get(1,1)*tmp[1][0]);
			mv.set(1,1,mv2.get(1,0)*tmp[0][1] + mv2.get(1,1)*tmp[1][1]);

			tmp[0][0] = mv.getV(0);	

			mv.setV(0,mv2.get(0,0)*mv.getV(0) + mv2.get(0,1)*mv.getV(1) +
			mv2.getV(0));
			mv.setV(1,mv2.get(1,0)*tmp[0][0] + mv2.get(1,1)*mv.getV(1) + mv2.getV(1));
			}
		}
	else
		{
		mv.set(0,0,1);
		mv.set(1,0,0);
		mv.set(0,1,0);
		mv.set(1,1,1);	

		mv.setV(0,0);
		mv.setV(1,0);
		}
	}
public String toString()
	{
	return name;//+"\n"+I[0]+"\t"+J[0]+"\t"+oriX+"\n"+I[1]+"\t"+J[1]+"\t"+oriY+"\n";
	}
	
/**
* Fixe les coordonnes de l'origine.
* @param double x, labscisse de l'origine.
* @param double y, l'ordonne de l'origine.
*/
public void setOrigin(double x, double y)
	{
	oriX = x;
	oriY = y;
	}
	
	
/**
* Renvoie l'abscisse de l'origine.
* @return double ,l'abscisse.
*/	
public double getOriginX()
	{
	return oriX;
	}

/**
* Renvoie l'ordonne de l'origine.
* @return double ,l'ordonne.
*/
public double getOriginY()
	{
	return oriY;
	}
	
/**
* Fixe les coordonnes du premier vecteur de base.
* @param double x, l'abscisse.
* @param double y, l'ordonne.
*/
public void setI(double x, double y) throws OIJException
	{
	if ( Math.abs(x*J[1] - y*J[0]) > 1.e-15 )
		{
		I[0] = x;
		I[1] = y;
		}
	else throw new OIJException("erreur");
	}
	
/**
* Fixe les coordonnes du premier vecteur de base.
* @param double x, l'abscisse.
* @param double y, l'ordonne.
*/
public void setJ(double x, double y) throws OIJException
	{
	if ( Math.abs(x*I[1] - y*I[0]) > 1.e-15 )
		{
		J[0] = x;
		J[1] = y;
		}
	else throw new OIJException("erreur");
	}

/**
* Renvoie l'abscisse du premier  vecteur de base.
* @return double, l'abscisse.
*/
public double getIx()
	{
	return I[0];
	}

/**
* Renvoie l'ordonne du premier  vecteur de base.
* @return double, l'ordonne.
*/
public double getIy()
	{
	return I[1];
	}
	
/**
* Renvoie l'abscisse du deuxime vecteur de base.
* @return double, l'abscisse.
*/
public double getJx()
	{
	return J[0];
	}
		
		
/**
* Renvoie l'ordonne du deuxime vecteur de base.
* @return double, l'ordonne.
*/
public double getJy()
	{
	return J[1];
	}
	
/**
* Fixe le repre de base.
* @param mOIJ, le repre de base.
*/	
public void setOIJ(OIJ rep)
	{
	this.rep = rep;
	}
	

/**
* Renvoie le repre de base.
* @return OIJ, le repre de base.
*/
public OIJ getOIJ()
	{
	return rep;
	}
	
/**
* Ecrit ses informations dans un fichier.
* @WFile f, le fichier  ecrire.
*/
public void writeFile(WFile f,Pack2D p)
	{
	f.writeln("{");
	f.writeln("O " +oriX+" " +oriY);
	f.writeln("I " +I[0]+" "+I[1]);
	f.writeln("J " +J[0]+" "+J[1]);
	f.writeln("}");
	}

/**
* Lit ses informations dans un fichier.
* @RFile f, le fichier  lire.
*/
public void readFile(RFile f,Pack2D p) throws java.io.IOException,MalformedFileException
	{
	double i0=1,i1=0,j0=0,j1=1;
	int type;
	type=f.nextToken();	//on lit la premiere '{'
	if ( (type !=StreamTokenizer.TT_WORD) || !f.sval.equals("{") ) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
	type=f.nextToken();	
	do
		{
		if (f.sval.equals("O"))
			{
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
			oriX = f.nval;
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
			oriY = f.nval;
			}
		else if (f.sval.equals("I"))
			{
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
			i0 = f.nval;
	
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
			i1 = f.nval;
			}
		else if (f.sval.equals("J"))
			{
			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
			j0 = f.nval;

			type = f.nextToken();
			if (type != StreamTokenizer.TT_NUMBER) throw new MalformedFileException("Fichier de donnes au mauvais format !!");
			j1 = f.nval;
			}
		type =f.nextToken();
		}
	while ( (type != StreamTokenizer.TT_WORD) || !f.sval.equals("}"));
	
	try
		{
		setI(i0,i1);
		setJ(j0,j1);
		}
	catch(OIJException e)
		{
		throw new MalformedFileException("Impossible de construire un tel repre.");
		}
	}

/**
* Cre un nouvel objet de mme classe et de mme contenu.
* @return Object un clone de l'objet.
* @exception  OutOfMemoryError s'il n'y a pas assez de mmoire.
* @see        java.lang.Cloneable
* @since Opale-2d 0.11
*/
public Object clone()
	{
	try
		{
		OIJ p = (OIJ) super.clone();
		p.name = name;
		p.oriX = oriX;
		p.oriY = oriY;
		p.rep = rep;
		p.I[0] = I[0];
		p.I[1] = I[1];
		p.J[0] = J[0];
		p.J[1] = J[1];
	
		return p;
		}
	catch (CloneNotSupportedException e) // ne devrait jamais arriver
		{
		throw new InternalError();
		}
   	}
	

}



