/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.util;

import opale.soya.soya3d.*;
import java.lang.Math;

/**
 * Contains static definition for quaternion math methods.
 * A quaternion represent a direction.
 * 
 * Here, a matrix is a float[16], and a vector or a point a float[3].
 * 
 * Quaternions are used by the animation system, for interpolating between 2 positions.
 * (for animations, see soya.soya3d.animation).
 * 
 * Don't ask me more about those mathematic stuff! They work, and that's all I know.
 * 
 * @author Artiste on the Web
 */

public class Quaternion extends Object {
  private Quaternion() {  }

  /**
   * The value of PI in float.
   */
  public static final float PI = (float) java.lang.Math.PI;
  
  /**
   * Compare 2 quaternions.
   * @param a the first quaternion
   * @param b the second quaternion
   * @return true if a and b are equal (or very near)
   */
  public static final boolean quaternionEqual(float[] a, float[] b) {
    for(int i = 0; i < 4; i++) {
      if(Math.abs(a[i] - b[i]) > 0.001f) return false;
    }
    return true;
  }
  
  /**
   * Normalizes the given quaternion.
   * @param q the quaternion
   */
  public static final void quaternionNormalize(float[] q) {
    float l = (float) Math.sqrt((double) (q[0] * q[0] + q[1] * q[1] + q[2] * q[2] + q[3] * q[3]));
    q[0] = q[0] / l;
    q[1] = q[1] / l;
    q[2] = q[2] / l;
    q[3] = q[3] / l;
  }

  /**
   * Convert a quaternion into a matrix.
   * @param q the quaternion
   * @return the matrix
   */
  public static float[] matrixFromQuaternion(float[] q) {
    quaternionNormalize(q);
    float xx = q[0] * q[0], yy = q[1] * q[1], zz = q[2] * q[2],
          xy = q[0] * q[1], xz = q[0] * q[2], yz = q[1] * q[2],
          wx = q[3] * q[0], wy = q[3] * q[1], wz = q[3] * q[2];
    float[] m = { 1f - 2f * (yy + zz),      2f * (xy + wz),      2f * (xz - wy), 0f,
                       2f * (xy - wz), 1f - 2f * (xx + zz),      2f * (yz + wx), 0f,
                       2f * (xz + wy),      2f * (yz - wx), 1f - 2f * (xx + yy), 0f,
                                   0f,                  0f,                  0f, 1f };
    return m;
  }

  /**
   * Convert a matrix into a quaternion. Warning : the matrix must represent only rotations
   * and / or translations, but no scaling.
   * @param m the matrix
   * @return the quaternion
   */
  public static float[] quaternionFromMatrix(float[] m) {
    float s = (float) Math.sqrt(Math.abs(m[0] + m[5] + m[10] + m[15]));
    float[] q = { -(m[9] - m[6]) / (2f * s),
                  -(m[2] - m[8]) / (2f * s),
                  -(m[4] - m[1]) / (2f * s),
                  s / 2f };
    if(s == 0f) {
      if(Float.isInfinite(q[0])) {
        q[0] = 1f; q[1] = 0f; q[2] = 0f; q[3] = 0f; return q;
      }
      if(Float.isInfinite(q[1])) {
        q[0] = 0f; q[1] = 1f; q[2] = 0f; q[3] = 0f; return q;
      }
      if(Float.isInfinite(q[2])) {
        q[0] = 0f; q[1] = 0f; q[2] = 1f; q[3] = 0f; return q;
      }
    }
    quaternionNormalize(q);
    return q;
  }

  /**
   * Create a quaternion that correspond to the given rotation.
   * @param v the axe of the rotation : a vector in a float[3]. v must be normalized
   * @param angle the angle of the rotation in degree
   * @return the quaternion
   */
  public static float[] quaternionFromRotation(float[] v, float angle) {
    angle = (angle / 180f * PI) / 2f;
    float sinAngle = (float) Math.sin((double) angle);
    float[] q = { sinAngle * v[0],
                  sinAngle * v[1],
                  sinAngle * v[2],
                  (float) java.lang.Math.cos((double) angle) };
    quaternionNormalize(q);
    return q;
  }

  /**
   * Create a quaternion from the a graphical element.
   * @param ge the graphical element
   * @return the quaternion
   */
  public static float[] quaternionFromGraphical(GraphicalElement3D ge) {
    float[] m = Matrix.matrixScale(ge.getMatrix(), 1f / ge.getXFactor(), 1f / ge.getYFactor(), 1f / ge.getZFactor());
    return quaternionFromMatrix(m);
  }
  /*
  public static float[] quaternionFromMatrix(double[] m) {
    double s = Math.sqrt(Math.abs(m[0] + m[5] + m[10] + m[15]));
    float[] q = { (float) ((m[6] - m[9]) / (2d * s)),
                  (float) ((m[8] - m[2]) / (2d * s)),
                  (float) ((m[1] - m[4]) / (2d * s)),
                  (float) (s / 2d) };
    if(s == 0d) {
      if(Float.isInfinite(q[0])) {
        q[0] = 1f; q[1] = 0f; q[2] = 0f; q[3] = 0f; return q;
      }
      if(Float.isInfinite(q[1])) {
        q[0] = 0f; q[1] = 1f; q[2] = 0f; q[3] = 0f; return q;
      }
      if(Float.isInfinite(q[2])) {
        q[0] = 0f; q[1] = 0f; q[2] = 1f; q[3] = 0f; return q;
      }
    }
    
    double l = Math.sqrt((double) (q[0] * q[0] + q[1] * q[1] + q[2] * q[2] + q[3] * q[3]));
    q[0] = (float) (((double) q[0]) / l);
    q[1] = (float) (((double) q[1]) / l);
    q[2] = (float) (((double) q[2]) / l);
    q[3] = (float) (((double) q[3]) / l);
    
    return q;
  }
  */
  
  /**
   * Multiply 2 quaternions.
   * @param q1 the first quaternion
   * @param q2 the second quaternion
   * @return the resulting quaternion, that combine the rotation of both q1 and q2
   */
  public static float[] quaternionMultiply(float[] q1, float[] q2) {
    float[] r = { q2[3] * q1[0] + q2[0] * q1[3] + q2[1] * q1[2] + q2[2] * q1[1],
                  q2[3] * q1[1] + q2[1] * q1[3] + q2[2] * q1[0] + q2[0] * q1[2],
                  q2[3] * q1[2] + q2[2] * q1[3] + q2[0] * q1[1] + q2[1] * q1[0],
                  q2[3] * q1[3] + q2[0] * q1[0] + q2[1] * q1[1] + q2[2] * q1[2] };
    float d = (float) Math.sqrt((double) (r[0] * r[0] + r[1] * r[1] + r[2] * r[2] + r[3] * r[3]));
    r[0] = r[0] / d;
    r[1] = r[1] / d;
    r[2] = r[2] / d;
    r[3] = r[3] / d;
    quaternionNormalize(r);
    return r;
  }

  /**
   * Slerp 2 quaternions. 
   * @param q1 the first quaternion
   * @param q2 the second quaternion
   * @param alpha the weight of q1. The weight of q2 is 1 - alpha
   * @return the resulting quaternion, that combine the rotation of both q1 and q2
   */
  public static float[] quaternionSlerp(float[] q1, float[] q2, float alpha) {
    float scale1, scale2;
    float cosTheta = q1[0] * q2[0] + q1[1] * q2[1] + q1[2] * q2[2] + q1[3] * q2[3];
    if(cosTheta < 0) {
      q1 = (float[]) q1.clone();
      q1[0] = -q1[0];
      q1[1] = -q1[1];
      q1[2] = -q1[2];
      q1[3] = -q1[3];
      cosTheta = -cosTheta;
    }
    if(cosTheta + 1f > 0.05f) {
      if(1 - cosTheta < 0.05f) {
        scale1 = 1f - alpha;
        scale2 = alpha;
      }
      else {
        float theta = (float) Math.acos((double) cosTheta);
        float sinTheta = (float) Math.sin(theta);
        scale1 = (float) Math.sin((double) theta * (1f - alpha)) / sinTheta;
        scale2 = (float) Math.sin((double) theta * alpha) / sinTheta;
      }
    }
    else {
      q2 = (float[]) q2.clone();
      q2[0] = -q1[1];
      q2[1] =  q1[0];
      q2[2] = -q1[3];
      q2[3] =  q1[1];
      scale1 = (float) Math.sin((double) PI * (0.5f - alpha));
      scale2 = (float) Math.sin((double) PI * alpha);
    }

    float[] q = { scale1 * q1[0] + scale2 * q2[0],
                  scale1 * q1[1] + scale2 * q2[1],
                  scale1 * q1[2] + scale2 * q2[2],
                  scale1 * q1[3] + scale2 * q2[3] };
    quaternionNormalize(q);
    return q;
  }
  
  /**
   * Gets a string that describe the given quaternion.
   * @param q the quaternion
   * @return the description string
   */
  public static String quaternionToString(float[] q) {
    return "Quaternion {" + q[0] + ", " + q[1] + ", " + q[2] + ", " + q[3] + "}";
  }
}
