/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.util;

import java.util.Iterator;

/**
 * This interface extends the java standard collection interface by adding events.
 *
 * Also contains usefull iterators class.
 * 
 * @author Artiste on the Web
 */

public interface Collection extends java.util.Collection {
  /**
   * Create an iterator that iterates only of the given class of object (including subclasses).
   * @param clazz the class of object that will be iterated.
   * @return an class-specific iterator.
   */
  public Iterator iterator(Class clazz);

  /**
   * Create an iterator that iterates all elements and their sub-elements (if an element is a collection, its own elements.).
   * @return a recursive iterator.
   */
  public Iterator iteratorRecursive();

  /**
   * Create an iterator that iterates all elements and their sub-elements (if an element is a collection, its own elements.) of a specific class.
   * @param clazz the class of object that will be iterated.
   * @return a class specific recursive iterator.
   */
  public Iterator iteratorRecursive(Class clazz);

  /**
   * An iterator that iterates only a given class of object.
   */
  public static class ClassSpecificIterator implements Iterator {
    /**
     * Create An iterator that iterates only of the given class of object.
     * @param co the collection to iterate.
     * @param cl the class of object that will be iterated.
     */
    public ClassSpecificIterator(java.util.Collection co, Class cl) {
      this(co.iterator(), cl);
    }
    /**
     * Create An iterator that iterates only of the given class of object.
     * @param it an iterator from the collection to iterate. You can also use a custom iterator, like a RecursiveIterator.
     * @param cl the class of object that will be iterated.
     */
    public ClassSpecificIterator(Iterator it, Class cl) {
      i = it;
      clazz = cl;
    }

    private Iterator i;
    private Class clazz;
    private Object myNext;

    public boolean hasNext() {
      while(true) {
        if(!i.hasNext()) return false;
        myNext = i.next();
        if(clazz.isInstance(myNext)) return true;
      }
    }
    public Object next() { return myNext; }
    public void remove() { i.remove(); }
  }

  /**
   * A RecursiveIterator iterate all element of a collection, and if any of them if a Collection, it iterate also its element, and so on.
   */
  public static class RecursiveIterator implements Iterator {
    /**
     * Create A RecursiveIterator.
     * @param it an iterator from the collection to iterate. You can also use a custom iterator, like a ClassSpecificIterator.
     */
    public RecursiveIterator(Iterator it) { current = new RecursiveNode(it); }
    /**
     * Create A RecursiveIterator.
     * @param c the collection to iterate.
     */
    public RecursiveIterator(java.util.Collection c) { current = new RecursiveNode(c.iterator()); }
    
    private RecursiveNode current;
    private Object myNext;
    
    public boolean hasNext() {
      while(true) {
        current.makeRecursive();
        if(!current.hasNext()) return false;
        myNext = current.next();
        return true;
      }
    }
    public Object next() { return myNext; }
    public void remove() { current.remove(); }
    
    private class RecursiveNode implements Iterator {
      public RecursiveNode(Iterator it) { i = it; }
      public RecursiveNode(Iterator it, RecursiveNode prev) {
        i = it;
        previous = prev;
      }
      
      private Iterator previous;
      private Iterator i;
      private Object myNext;
      
      public void makeRecursive() {
        if(myNext instanceof java.util.Collection) {
          RecursiveIterator.this.current = new RecursiveNode(((java.util.Collection) myNext).iterator(), this);
        }
      }
      
      public boolean hasNext() {
        while(true) {
          if(!i.hasNext()) {
            if(previous == null) return false;
            else {
              RecursiveIterator.this.current = (RecursiveNode) previous;
              return previous.hasNext();
            }
          }
          myNext = i.next();
          return true;
        }
      }
      public Object next() { return myNext; }
      public void remove() { i.remove(); }
    }
  }

  /**
   * Add a listener for collection event.
   * @param listener the listener.
   */
  public void addCollectionListener(CollectionListener listener);
  /**
   * Remove a listener for collection event.
   * @param listener the listener.
   */
  public void removeCollectionListener(CollectionListener listener);

  /**
   * Get all the listeners.
   * @return a collection of listeners.
   */
  public java.util.Collection collectionListeners();

  /**
   * Raise a collection-add event on this object.
   * @param element the object added.
   */
  public void fireAdd(Object element);
  /**
   * Raise a collection-remove event on this object.
   * @param element the object removed.
   */
  public void fireRemove(Object element);
}

