/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.util;

import java.beans.*;

/**
 * An abstract class for BeanInfo that allow to recover the info from the parent class and the
 * implemented interfaces.
 * 
 * Just add the new properties, methods and events from any parent classes / interfaces will be
 * added.
 * 
 * Many examples can be found in the soya source code (soya.soya3d.*3D for example).
 * 
 * @author Artiste on the Web
 */

public abstract class AbstractBeanInfo extends java.beans.SimpleBeanInfo {
  /**
   * Creates a bean info. For each kind of element in the class (properties, events, methods),
   * you can choose to use default value from java.beans.Introspector, or to set them by yourself,
   * by calling the addPropertyDescriptors, addEventDescriptors or addMethodDescriptors that
   * correspond to the properties, events or methods of the documented class (but NOT the ones that
   * are specified into a super-class or an interface).
   * @param customProperties true if you will specify explicitely the properties.
   * @param customEvents     true if you will specify explicitely the events.
   * @param customMethods    true if you will specify explicitely the methods.
   */
  public AbstractBeanInfo(boolean customProperties, boolean customEvents, boolean customMethods) {
    this.customProperties = customProperties;
    this.customEvents = customEvents;
    this.customMethods = customMethods;
    if(customProperties) pd = new PropertyDescriptor[0];
    if(customEvents) ed = new EventSetDescriptor[0];
    if(customMethods) md = new MethodDescriptor[0];
    addDescriptorFromSuperclass();
    addDescriptorFromInterfaces();
  }
  
  protected boolean customProperties, customEvents, customMethods;

  /**
   * Gets the name of the class of the documented bean class, by removing "BeanInfo" at the end
   * of the name of the bean info.
   * If your bean info do not use this denomination, you must override this method and return the
   * good class name.
   * @return the bean class name
   */
  public String getBeanClassName() {
    String myClassName = getClass().getName();
    return myClassName.substring(0, myClassName.length() - "BeanInfo".length());
  }
  private Class beanClass;
  /**
   * Gets the class of the documented bean class, by removing "BeanInfo" at the end of the name
   * of the bean info.
   * If your bean info do not use this denomination, you must override this method and return the
   * good class.
   * @return the bean class
   */
  public Class getBeanClass() {
    if(beanClass == null) {
      try { beanClass = Class.forName(getBeanClassName()); }
      catch(Exception e) {
        System.out.println(getClass().getName() + " : error");
        e.printStackTrace();
        return null;
      }
    }
    return beanClass;
  }

  /**
   * Adds to this bean info all the descriptor of an onther bean class.
   * @param className another bean class name
   */
  protected synchronized void addDescriptorFromClass(String className) {
    try { addDescriptorFromClass(Class.forName(getBeanClassName())); }
    catch(Exception e) {
      System.out.println(getClass().getName() + " : error");
      e.printStackTrace();
    }
  }
  /**
   * Adds to this bean info all the descriptors of an onther bean class.
   * @param c another bean class
   */
  protected synchronized void addDescriptorFromClass(Class c) {
    try {
      BeanInfo bi = Introspector.getBeanInfo(c);
      addPropertyDescriptors(bi.getPropertyDescriptors());
      addEventSetDescriptors(bi.getEventSetDescriptors());
      addMethodDescriptors  (bi.getMethodDescriptors()  );
    }
    catch(Exception e) {
      System.out.println(getClass().getName() + " : error while adding descriptor from class " + c.getName());
      e.printStackTrace();
    }
  }
  private synchronized void addDescriptorFromSuperclass() {
    Class sup = getBeanClass().getSuperclass();
    if(sup != null) addDescriptorFromClass(sup);
  }
  private synchronized void addDescriptorFromInterfaces() {
    Class[] interfaces = getBeanClass().getInterfaces();
    for(int i = 0; i < interfaces.length; i++) addDescriptorFromClass(interfaces[i]);
  }

  private PropertyDescriptor[] pd;
  private EventSetDescriptor[] ed;
  private MethodDescriptor  [] md;
  public synchronized PropertyDescriptor[] getPropertyDescriptors() { return pd; }
  public synchronized EventSetDescriptor[] getEventSetDescriptors() { return ed; }
  public synchronized MethodDescriptor  [] getMethodDescriptors  () { return md; }
  
  /**
   * Adds to this bean info the given properties descriptors.
   * This is needed only if you have choosen to custom properties in the constructor.
   * @param add the properties descriptors
   */
  protected synchronized void addPropertyDescriptors(PropertyDescriptor[] add) {
    if(!customProperties) return;
    int pdLength = pd.length ;
    int addLength = add.length ;
    PropertyDescriptor[] newpd = new PropertyDescriptor[pdLength + addLength]; 
    System.arraycopy(pd, 0, newpd, 0,        pdLength);
    System.arraycopy(add, 0, newpd, pdLength, addLength);
    pd = newpd;
  }
  /**
   * Adds to this bean info the given events descriptors.
   * This is needed only if you have choosen to custom events in the constructor.
   * @param add the events descriptors
   */
  protected synchronized void addEventSetDescriptors(EventSetDescriptor[] add) {
    if(!customEvents) return;
    int edLength = ed.length ;
    int addLength = add.length ;
    EventSetDescriptor[] newed = new EventSetDescriptor[edLength + addLength];
    System.arraycopy(ed,  0, newed, 0,        edLength);
    System.arraycopy(add, 0, newed, edLength, addLength);
    ed = newed;
  }
  /**
   * Adds to this bean info the given methods descriptors.
   * This is needed only if you have choosen to custom methods in the constructor.
   * @param add the methods descriptors
   */
  protected synchronized void addMethodDescriptors(MethodDescriptor[] add) {
    if(!customMethods) return;
    int mdLength = md.length ;
    int addLength = add.length ;
    MethodDescriptor[] newmd = new MethodDescriptor[mdLength + addLength];
    System.arraycopy(md,  0, newmd, 0,        mdLength);
    System.arraycopy(add, 0, newmd, mdLength, addLength);
    md = newmd;
  }
}
