/*
 * Soya3D tutorial
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
Lesson 2 : modeling
You'll learn how to create a shape, like the cube of the lesson 1.
Those shapes can be saved and loaded. The soya GUI editor is the easiest
way to create them.
Here, we will build them by coding. 

For pedagogic purpose, the lesson 1's comments are removed from here... :-C
*/

package opale.soya.tutorial;

import opale.soya.*;              // contains the basic stuff (initialization,...).
import opale.soya.awt.*;          // contains the GUI stuff (RenderingCanvas, RenderingFrame,...).
import opale.soya.soya2d.*;       // contains the 2D stuff (Texturing and Material).
import opale.soya.soya3d.*;       // contains the 3D stuff (A lot of things).
import opale.soya.soya3d.model.*; // contains the modeling stuff (Shape,...).

/*
The Lesson2 is a rendering frame (in order to show our shape).
*/
public class Lesson2 extends RenderingFrame {
  public static void main(String[] args) { (new Lesson2()).setVisible(true); }

  public Lesson2() {
    super(320, 200);

    Soya.init(this);
    
    /*
    Use an environment instead of a world. Environment extends world, and adds environmental
    properties, like fog, ambiant lighting and color/image background.
    */
    scene = new Environment3D();
    
    /*
    Sets the ambiant lighting color to gray.
    */
    scene.setAmbientColor(.5f, .5f, .5f);

    /*
    buildShape is our new function.
    */
    Shape pyramid = buildShape();

    /*
    This constructor for volume create a volume with the specified shape.
    */
    Volume3D volume = new Volume3D(pyramid);
    scene.add(volume);

    /*
    No other lights are required, because of the ambient lighting.
    */
    
    Camera3D camera = new Camera3D();
    scene.add(camera);
    camera.move(0f, -0.1f, 1.3f);

    this.setRenderer(camera);

    repaint();
  }
  private Environment3D scene;

  private Shape buildShape() {
    /*
    Create a new Material. This object describes the property of the surface, such as
    color, shininess or texture.
    Materials can be saved or loaded exactely like shapes.
    */
    Material m = new Material();

    /*
    Create a new Texture for our material, from a gif image.
    */
    Texture t = null;
    try { t = new TextureRGB(Shape.path + "Autel.gif"); }
    catch(Exception e) { e.printStackTrace(); }

    /*
    Set the texture property of the material.
    */
    m.setTexture(t);
    m.setShininess(1f);

    /*
    Create a new Shape. A fragmented shape is a shape that contains a collection of
    ShapeElement, like Face, such as Triangle or Quad. We need to create each shape
    element, and to add all of them into the fragmented shape.
    A fragment is an optimized piece of a fragmented shape.
    A shape element is a "user-friendly" piece of a fragmented shape; it will be converted
    into fragment(s) before drawing, for speed boost.
    For example, here we use 4 triangles and 1 quad, and those are shape elements. While
    converting them into fragments, many optimizations will be performed : the 4 triangles
    will be regrouped and turn into a single fragment,...
    With this system, you can draw with "user-friendly" object, and have a good optimization,
    without having to take care of this optimization :-) !
    */
    FragmentedShape s = new FragmentedShape();

    /*
    Lock the updatement of the shape. Each time you add a shape element to the shape,
    it is updated. For performance boost, you can lock the shape, add many shape elements,
    and unlock it. Unlocking will update the shape once. But locking IS NOT obligatory.
    Locking twice will require to unlock twice before updating.
    */
    s.lock();

    /*
    Create all the faces for a pyramid. A triangle is created from 3 point and its material.
    A quad required 4 point.
    An AdvancedPoint is a 3D point that can have color or texture coordinates. They are created
    here from the 3 coordinates and the 2 texture coordinates.
    */
    s.add(new Triangle(new AdvancedPoint( .5f, 0f,  .5f, 1f, 1f),
                       new AdvancedPoint(-.5f, 0f,  .5f, 0f, 1f),
                       new AdvancedPoint(  0f, 1f,   0f, 0f, 0f),
                       m));
    s.add(new Triangle(new AdvancedPoint(-.5f, 0f, -.5f, 1f, 1f),
                       new AdvancedPoint( .5f, 0f, -.5f, 0f, 1f),
                       new AdvancedPoint(  0f, 1f,   0f, 0f, 0f),
                       m));
    s.add(new Triangle(new AdvancedPoint(-.5f, 0f,  .5f, 0f, 1f),
                       new AdvancedPoint(-.5f, 0f, -.5f, 1f, 1f),
                       new AdvancedPoint(  0f, 1f,   0f, 0f, 0f),
                       m));
    s.add(new Triangle(new AdvancedPoint( .5f, 0f, -.5f, 0f, 1f),
                       new AdvancedPoint( .5f, 0f,  .5f, 1f, 1f),
                       new AdvancedPoint(  0f, 1f,   0f, 0f, 0f),
                       m));
		
    s.add(new Quad(new AdvancedPoint(-.5f, 0f, -.5f, 0f, 0f),
                   new AdvancedPoint(-.5f, 0f,  .5f, 0f, 1f),
                   new AdvancedPoint( .5f, 0f,  .5f, 1f, 1f),
                   new AdvancedPoint( .5f, 0f, -.5f, 1f, 0f),
                   m));
    
    /*
    Set the side of the face that will be visible. For performance, OpenGL allows 
    face to be seen in one side instead of both. Soya can compute face of a closed-shape
    in order to see the face only from the exterior or the interior. Here, we want to see
    our shape from the exterior, so we use FaceVisibility.VISIBILITY_EXTERIOR.
    You can try with FaceVisibility.VISIBILITY_INTERIOR to see the difference.
    */
    s.setVisibility(FaceVisibility.VISIBILITY_EXTERIOR);

    /*
    Set the name of the shape.
    */
    s.setName("pyramid");

    /*
    Unlock and update the shape.
    */
    s.unlock();

    /*
    Save the shape. The file will be put in the Shape.path directory, with the filename
    pyramid.shape .
    The material have no name and is not saved in the material path, so it will be saved
    in the same file. If you want to save it, set the Material.path field, give a name 
    to the material and save it.
    */
    try { s.save(); }
    catch(Exception e) {  }

    /*
    The shape is finished!!! :-).
    As you can see, it is rather long... Use the soya GUI editor!!!!
    */
    return s;
  }
}
