/*
 * Soya3D tutorial
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
Lesson 101 : Particle system
Particle system is a great way to create impressive 3D effects, like fire, fume, water, magic
or blood...
The lesson 101 draws 2 shrines, one is a fire-shrine, and the other is a water-fountain.
*/

package opale.soya.tutorial;

/*
First, import a few soya packages.
*/
import opale.soya.*;                    // contains the basic stuff (initialization,...).
import opale.soya.awt.*;                // contains the GUI stuff (RenderingCanvas, RenderingFrame,...).
import opale.soya.soya2d.*;             // contains the 2D stuff (Texturing and Material).
import opale.soya.soya3d.*;             // contains the 3D stuff (A lot of things).
import opale.soya.soya3d.model.*;       // contains the modeling stuff, for the cube (Shape,...).
import opale.soya.soya3d.fx.*;          // contains the special effect stuff.
import opale.soya.soya3d.fx.particle.*; // contains the particle system stuff (ParticlesBunch3D,...).

import java.net.*;

/*
The Lesson101 is a rendering frame.
*/
public class Lesson101 extends RenderingFrame {
  /*
  Create a Lesson101 object and show it.
  */
  public static void main(String[] args) {
    (new Lesson101()).setVisible(true);
  }
  
  /*
  Builds the scene.
  */
  public Lesson101() {
    super(320, 200);
    /*
    inits soya and defines pathes.
    */
    Soya.init(this);
    Shape   .path = getClass().getResource("").getFile();
    Material.path = getClass().getResource("").getFile();
    
    /*
    Creates the scene (here an environment).
    */
    scene = new Environment3D();
    scene.setAmbientColor(.5f, .5f, .5f);
    
    /*
    Creates a light.
    */
    Light3D light = new Light3D();
    light.setColor(1f, .8f, .2f);
    light.move(0f, .5f, 0f);
    light.setAttenuation(0f, .01f, 0f);
    scene.add(light);
    
    /*
    Creates a camera.
    */
    Camera3D camera = new Camera3D();
    scene.add(camera);
    camera.move(0f, .4f, 3f);
    
    /*
    Creates 2 volumes, and load into them a fountain-like shape.
    */
    Volume3D volume1 = new Volume3D();
    try { volume1.setShape(Shape.get("fontaine")); }
    catch(Exception e) { e.printStackTrace(); }
    scene.add(volume1);
    volume1.scale(1.8f);
    volume1.move(-1f, -1.1f, 0f);
    
    Volume3D volume2 = (Volume3D) volume1.clone(); // Clone volume1. Any Element3D is cloneable.
    scene.add(volume2);
    volume2.move(1f, -1.1f, 0f);
    
    /*
    Loads a material to texture the particles.
    */
    Material material = null;
    try { material = Material.get("x-geser-1"); }
    catch(Exception e) { e.printStackTrace(); }
    
    
    /*
    Creates a flame-like particles bunch, that use the given material to texture the particles.
    In soya, a particle system is whatever object that extends ParticlesBunch3D, like Flame3D.
    A particle bunch is a graphical element, so you can use all the method from this class,
    like the move method.
    Here, we use a custom constructor from Flame3D, that creates a standard flame.
    */
    Flame3D bunch1 = new Flame3D(material);
    /*
    Sets the number of particles in the system. 80 is enought.
    */
    bunch1.setNumberOfParticles(80);
    scene.add(bunch1);
    bunch1.move(-1f, 0f, 0f);
    
    /*
    Creates a water-fountain-like particles bunch, that use the given material to texture the
    particles.
    Typicaly, particles bunches are created like below and not like above.
    A particles bunch is composed of several objects, each of them managing a part of the
    particles system :
      - The particles bunch itself manages the position, the speed and the movement of the
        particles.
      - A ParticlesLifes object manages the life of the particles. Each particle starts with a
        life of 1f, and die when its life fall below 0f. The life-speed represent how fast the
        life of the particle decreases, it is randomly choosen at the particle'birth, between
        a minimum and a maximum limits. Those limits are the 2 parameters of the
        ParticlesLifes' constructor.
      - A ParticlesColors object manages the color of the particles. Here we use a
        FadingParticlesColors. With such a ParticlesColors, all particles starts with the same
        color, and the alpha component decrease with the life of the particle. 
      - A ParticlesDrawer object is responsible for the drawing of the particles. For example,
        PointParticlesDrawer draws them as points, and SpriteParticlesDrawer draws particles
        as sprites.
    This system allow you to create the particle system of your dream, by combining a particles
    bunch (how the particles move), a particles lifes (how longer the particles last), a
    particles colors (what colors the particles have) and a particles drawer (how the particles
    are rendered).
    
    The main object is the particles bunch, and its particlesLifes, particlesColors and
    particlesDrawer properties correspond to the other 3 objects.
    
    The typical constructor for a particle bunch object (or its sub-class) has 3 parameters,
    one for each of those objects. It is the one used here.
    */
    Fountain3D bunch2 = new Fountain3D(
                                       new ParticlesLifes       (.006f, .01f     ), // The object that will manage the particles' lifes.
                                       new FadingParticlesColors(.8f, .9f, 1f, .3f), // The object that will manage the particles' colors.
                                       new SpriteParticlesDrawer (material        )  // The object that will draw the particles.
                                       //new PointParticlesDrawer (10f              )  // The object that will draw the particles.
                                       );
    bunch2.setNumberOfParticles(100);
    bunch2.setAngle(30f);
    //bunch2.setAcceleration(new Vector(  0f, -.006f,   0f, bunch2));
    scene.add(bunch2);
    bunch2.move(1f, -.3f, 0f);
    
    
    this.setRenderer(camera);
    
    /*
    Shows the FPS.
    */
    showFPS();
    
    while(true) {
      render();
    }
  }
  
  private Environment3D scene;
}
