/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.event.*;
import gl4java.*;
import java.util.Set;
import java.util.Map;
import java.util.HashSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Arrays;
import java.lang.Math;
import java.lang.ref.*;
import java.lang.reflect.*;
import java.io.*;
import java.beans.*;

/**
 * Abstract class for all shapes. A shape describe a 3D visible object, but has no position,...
 * and cannot be directly drawn in a rendering surface. To see a shape, you must put it into
 * a volume, which is a 3D element and so can be added into a world. Several volumes can share
 * the same shape and have different position, orientation or dimensions.
 * 
 * See fragmented shape for a concrete class.
 * 
 * @see opale.soya.soya3d.Volume3D
 * @see opale.soya.soya3d.model.FragmentedShape
 * @author Artiste on the Web
 */

public abstract class Shape extends AbstractBean implements StoredInPathObject, Lockable, Raypicking, DrawablesCollectorFiller, Dimension, CoordSyst, Transformable {
  private static final long serialVersionUID = 5169240283358815223l;
  /**
   * Creates a new shape.
   */
  public void Shape() {  }
  
  /**
   * Clones this shape.
   * @return the clone.
   */
  public synchronized Object clone() {
    Shape s = null;
    try { s = (Shape) getClass().newInstance(); }
    catch(Exception e) { System.out.println("Cannot create a new object : " + getClass().getName()); return null; }
    System.arraycopy(min , 0, s.min , 0, 3);
    System.arraycopy(max , 0, s.max , 0, 3);
    System.arraycopy(dims, 0, s.dims, 0, 3);
    s.name = name;
    return s;
  }
  
  /**
   * Writes this shape into the given output stream. If it has no name, it will be fully
   * written into the stream, else just the name will be written. It allow you to use the same
   * shape in different files.
   * Use read(ObjectOutputStream) to read a shape written by write(ObjectOutputStream).
   * @see opale.soya.soya3d.model.Shape#read
   * @param s the ObjectOutputStream
   */
  public void write(ObjectOutputStream s) throws IOException {
    if(hasName()) {
      s.writeInt(OBJECT_SAVED_IN_PATH);
      s.writeUTF(getName());
    }
    else {
      s.writeInt(OBJECT_SAVED_HERE);
      s.writeObject(this);
    }
  }
  /**
   * Writes the given shape into the given output stream. If it has no name, it will be fully
   * written into the stream, else just the name will be written. It allow you to use the
   * same shape in different files.
   * Use read(ObjectOutputStream) to read a shape written by write(ObjectOutputStream).
   * @see opale.soya.soya3d.model.Shape#read
   * @param s the ObjectOutputStream
   * @param shape the shape (may be null)
   */
  public static void write(ObjectOutputStream s, Shape shape) throws IOException {
    if(shape == null) s.writeInt(NO_OBJECT_SAVED);
    else shape.write(s);
  }
  /**
   * Reads a shape from the given output stream.
   * The shape must have been written by one of the write method.
   * If just the name of the shape was written, it can work only if the shape path contains a
   * shape file with the same name : the shape will be recovered exactely as the get method
   * does.
   * @see opale.soya.soya3d.model.Shape#write
   * @see opale.soya.soya3d.model.Shape#get
   * @param s the ObjectOutputStream
   * @return the shape (may be null)
   */
  public static Shape read(ObjectInputStream s) throws IOException, ClassNotFoundException {
    switch(s.readInt()) {
      case NO_OBJECT_SAVED: return null;
      case OBJECT_SAVED_IN_PATH: return get(s.readUTF());
      case OBJECT_SAVED_HERE: return (Shape) s.readObject();
    }
    return null;
  }

  public static String path;
  /**
   * Set the shape path.
   * @see opale.soya.soya3d.model.Shape#path
   * @param p the new path
   */
  public static void setPath(String p) {
    if(p.endsWith("/")) path = p;
    else path = p + "/";
  }
  public void save() throws java.io.IOException { save(path + name + ".shape"); }
  public void save(String fileName) throws java.io.IOException {
    ObjectOutputStream o = new ObjectOutputStream(new BufferedOutputStream(new FileOutputStream(fileName)));
    o.writeObject(this);
    o.flush();
    o.close();
  }
  public static void save(Shape s) throws java.io.IOException { s.save(); }
  public static void save(Shape s, String fileName) throws java.io.IOException { s.save(fileName); }
  public static Shape load(String fileName) throws java.io.IOException, java.lang.ClassNotFoundException {
    Shape s;
    ObjectInputStream i = new ObjectInputStream(new BufferedInputStream(new FileInputStream(fileName)));
    s = (Shape) i.readObject();
    i.close();
    return s;
  }
  private static final java.util.Set shapes = new java.util.HashSet();
  public static Shape get(String name) throws java.io.IOException, java.lang.ClassNotFoundException {
    Shape s;
    synchronized(shapes) {
      for(Iterator i = shapes.iterator(); i.hasNext(); ) {
        s = (Shape) ((SoftReference) i.next()).get();
        if(s == null) i.remove();
        else if(s.name.equals(name)) return s;
      }
    }
    return load(path + name + ".shape");
  }
  
  // Serializable :
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException { // Initialisation
    s.defaultReadObject();
    softReferenceToThis = new SoftReference(this);
    if(hasName()) shapes.add(softReferenceToThis);
  }

  private String name;
  public String getName() { return name; }
  public boolean hasName() {
    if(name == null) return false;
    if(name.length() == 0) return false;
    return true;
  }
  public void setName(String newName) {
    String oldName = name;
    boolean oldHasName = hasName();
    name = newName;
    if(hasName() && !oldHasName) shapes.add(softReferenceToThis);
    else {
      if(!hasName() && oldHasName) shapes.remove(softReferenceToThis);
    }
    firePropertyChange("name", oldName, name);
  }
  private transient SoftReference softReferenceToThis = new SoftReference(this);
  
  //public abstract void fillCollector(FragmentsCollector f, Renderer r, float[] currentMatrix);
  //public abstract void draw(Renderer r, GLFunc gl, GLUFunc glu); // Testing stuff.
  
  
  // Buildings :
  /**
   * Destroy and recreate all fragments. Called after a change.
   */
  public abstract void reBuild();
  /**
   * Computes dimensions into dims[], min[] and max[].
   */
  protected abstract void buildDimension();
  /**
   * Converts this shape into an array of fragments.
   * @return the fragments.
   */
  public abstract Fragment[] toFragments();
  
  // Dimension :
  protected final float[] min = new float[3], max = new float[3];
  public float[] getMin() { return min; }
  public float[] getMax() { return max; }

  protected final float[] dims = new float[3];
  public float getWidth () { return dims[0]; }
  public float getHeight() { return dims[1]; }
  public float getDepth () { return dims[2]; }

  public void setWidth (float f) { scale(f / getWidth (), 1f, 1f); }
  public void setHeight(float f) { scale(1f, f / getHeight(), 1f); }
  public void setDepth (float f) { scale(1f, 1f, f / getDepth ()); }
  public void setDims(float w, float h, float d) {
    scale(w / getWidth(), h / getHeight(), d / getDepth());
  }
  
  protected float radiusSquarred;
  public boolean ensureRaypickIsUsefull(float distanceSquarred) {
    return distanceSquarred < radiusSquarred;
  }
  
  public DimensionWrapper wrapper() {
    return (DimensionWrapper) new Box(new Point(min[0], min[1], min[2], this), new Point(max[0], max[1], max[2], this));
  }
  public float getXFactor() { return 1f; }
  public float getYFactor() { return 1f; }
  public float getZFactor() { return 1f; }
  public void setXFactor(float f) {  }
  public void setYFactor(float f) {  }
  public void setZFactor(float f) {  }
  
  /**
   * Default implementation based on scale(float, float, float).
   */
  public void scale(float f) { scale(f, f, f); }
  /**
   * Default implementation based on transform(float[]).
   */
  public void scale(float fx, float fy, float fz) { transform(Matrix.matrixScale(fx, fy, fz)); }

  public void fireResize() {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeResizeEvent(this));
  }
  public void fireResize(String propertyName) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeResizeEvent(this, propertyName));
  }
  public void fireResize(String propertyName, Object oldValue, Object newValue) {
    if(isWorthFiringEvent()) firePropertyChange(new PropertyChangeResizeEvent(this, propertyName, oldValue, newValue));
  }
  
  /**
   * Tries to optimize the shape. Optimization may vary according to the implementation.
   * Changing the shape may unoptimize it.
   */
  public abstract void optimize();
  
  // Raypicking :
  public boolean isInterior(Position p) { // You must assume that the Shape is closed.
    Vector v = new Vector((float) (Math.random() - 0.5f), (float) (Math.random() - 0.5f), (float) (Math.random() - 0.5f));  // Random direction to perform raypick tests.
    return raypickNumber(p, v, Raypicking.SENS_HALF_LINE, Raypicking.INTERSECTION_ALL, Float.POSITIVE_INFINITY) % 2 == 1;
  }
  
  
  // Lockable :
  protected transient int lockLevel;
  public boolean isLocked() { return lockLevel > 0; }
  public synchronized void lock() { lockLevel++; }
  public synchronized void unlock() {
    if(lockLevel > 0) {
      lockLevel--;
      if(lockLevel == 0) reBuild();
    }
  }
  
  // CoordSyst :
  public void convertPointTo   (float[] p) {  }  // No-op : the shape is always considered
  public void convertPointFrom (float[] p) {  } // as a root coordinates system
  public void convertVectorTo  (float[] p) {  }
  public void convertVectorFrom(float[] p) {  }
  
  private static final float[] matrix = Matrix.matrixIdentity();
  public float[] getMatrix            () { return matrix; }
  public float[] getRootMatrix        () { return matrix; }
  public float[] getInvertedRootMatrix() { return matrix; }
  
  public boolean isLeftHanded () { return true ; }
  public boolean isRightHanded() { return false; }
  
  public Vector x() { return new Vector(1f, 0f, 0f, this); }
  public Vector y() { return new Vector(0f, 1f, 0f, this); }
  public Vector z() { return new Vector(0f, 0f, 1f, this); }
  public Position origin() { return new Point(0f, 0f, 0f, this); }
  public CoordSyst getCoordSyst() { return null; }
  public CoordSyst getRootCoordSyst() { return null; }
}
