/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.Matrix;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import gl4java.*;
import java.util.Iterator;
import java.io.*;

class QuadsFragment extends AbstractFragment {
  private static final long serialVersionUID = -5711213316788350333l;
  
  public Fragment getOptimizedFragment() {
    OptimizedQuadsFragment f;
    try { f = new OptimizedQuadsFragment(this); }
    catch(UnsupportedOperationException e) {
      // Thrown when optimization is not worthing.
      return this;
    }
    return f;
  }
  
  // static method for a fragment class.
  public static Fragment[] fragmentsFromShapeElements(Shape newShape, FragmentedShapeElement[] ses) {
    Fragment[] fs = { new QuadsFragment(newShape, ses) };
    return fs;
  }
  
  public QuadsFragment() {  }
  public QuadsFragment(Shape newShape, FragmentedShapeElement[] ses) {
    shape = newShape;
    
    Face onese = (Face) ses[0];
    visibility = onese.getVisibility(); // All face should share the same option.
    material   = onese.getMaterial  ();
    smoothLit  = onese.isSmoothLit  ();
    staticLit  = onese.isStaticLit  ();
    
    nbFaces = ses.length;
    normals = new float[nbFaces * 3];
    nbPoints = nbFaces * nbPointsPerFace;
    points = new float[nbPoints * 3];
    
    usePointsNormals = smoothLit;
    if(usePointsNormals) pointsNormals = new float[nbPoints * 3];
    
    if(material.getTexture() != null) {
      useTexCoords = true;
      pointsTexCoords = new float[nbPoints * 2];
    }
    else useTexCoords = false;
    
    usePointsColors = false;
    for(int i = 0; i < nbFaces; i++) {
      if(((Face) ses[i]).getUseColor()) { // Use color.
        usePointsColors = true;
        pointsColors = new float[nbPoints * 4];
        break;
      }
    }
    class ArrayFiller {
      private int normalIndex, pointIndex, pointTexCoordIndex, pointColorIndex;
      public void addNormal(Vector v) {
        normals[normalIndex++] = v.getX();
        normals[normalIndex++] = v.getY();
        normals[normalIndex++] = v.getZ();
      }
      public void addPoint(Position p) {
        points[pointIndex++] = p.getX();
        points[pointIndex++] = p.getY();
        points[pointIndex++] = p.getZ();
        if(useTexCoords) {
          if(p instanceof Textured) {
            Textured tex = (Textured) p;
            pointsTexCoords[pointTexCoordIndex++] = tex.getTextureCoordX();
            pointsTexCoords[pointTexCoordIndex++] = tex.getTextureCoordY();
          }
          else { // Need to advance the index.
            pointsTexCoords[pointTexCoordIndex++] = 0;
            pointsTexCoords[pointTexCoordIndex++] = 0;
          }
        }
        if(usePointsColors) {
          float[] colo;
          if(p instanceof Colored) {
						Colored color = (Colored) p;
						if(color.getUseColor())	colo = color.getColor();
						else colo = material.getDiffuse();
					}
          else colo = material.getDiffuse();
          pointsColors[pointColorIndex++] = colo[0];
          pointsColors[pointColorIndex++] = colo[1];
          pointsColors[pointColorIndex++] = colo[2];
          pointsColors[pointColorIndex++] = colo[3];
        }
      }
    }
    ArrayFiller af = new ArrayFiller();
    java.util.Collection hiddenFaces = null; // Hidden faces are set at the end.
    for(int i = 0; i < nbFaces; i++) {
      Face f = (Face) ses[i];
      synchronized(f) {
        if(f.isHidden()) {
          if(hiddenFaces == null) hiddenFaces = new java.util.Vector();
          hiddenFaces.add(f);
        }
        else {
          af.addNormal(f.getNormal());
          af.addPoint(f.getPoint(0));
          af.addPoint(f.getPoint(1));
          af.addPoint(f.getPoint(2));
          af.addPoint(f.getPoint(3));
        }
      }
    }
    if(hiddenFaces != null) {
      for(Iterator i = hiddenFaces.iterator(); i.hasNext(); ) {
        Face f = (Face) i.next();
        synchronized(f) {
          af.addNormal(f.getNormal());
          af.addPoint(f.getPoint(0));
          af.addPoint(f.getPoint(1));
          af.addPoint(f.getPoint(2));
          af.addPoint(f.getPoint(3));
        }
      }
      nbNonHiddenFaces = nbFaces - hiddenFaces.size();
    }
    else nbNonHiddenFaces = nbFaces; // No hidden face.
    defineUseAlpha();
  }

  protected static final int nbPointsPerFace = 4;
  public int getNumberOfPointsPerFace() { return nbPointsPerFace; }

  // Buildings :
  public synchronized void buildVisibility() {
    if(visibility == FaceVisibility.VISIBILITY_NO_INTERIOR || visibility == FaceVisibility.VISIBILITY_ALL ) return;
    Raypicking r = (Raypicking) shape;
    Vector v = new Vector((float) (Math.random() - 0.5f), (float) (Math.random() - 0.5f), (float) (Math.random() - 0.5f));  // Random direction to perform raypick tests.

    float[] nxyz = new float[4];
    Vector o_n = new Vector();    // opposite of the normal.

    float[] pxyz = new float[4];  // Point very near the center of the face; suppose that no other face are between the current face and this point.
    Position p = new Point();

    int normalIndex = 0, pointIndex = 0;
    for(int i = 0; i < nbFaces; i++) {
      o_n.move(-normals[normalIndex], -normals[normalIndex + 1], -normals[normalIndex + 2]);
      
      p.move( // Very near to the center of the face, and in the side odf the normal vector.
             (points[pointIndex    ] + points[pointIndex + 3] + points[pointIndex + 6] + points[pointIndex +  9]) / 4 + normals[normalIndex    ] / 1000,
             (points[pointIndex + 1] + points[pointIndex + 4] + points[pointIndex + 7] + points[pointIndex + 10]) / 4 + normals[normalIndex + 1] / 1000,
             (points[pointIndex + 2] + points[pointIndex + 5] + points[pointIndex + 8] + points[pointIndex + 11]) / 4 + normals[normalIndex + 2] / 1000
             );
      
      if(r.raypickNumber(p, v, Raypicking.SENS_HALF_LINE, Raypicking.INTERSECTION_ALL, Float.POSITIVE_INFINITY) % 2 == 1) { // true if odd.
        // Odd number of intersection in any direction if the point is inside, even else.
        // p is inside.
        if(r.raypickBoolean(p, o_n, Raypicking.SENS_HALF_LINE, Raypicking.INTERSECTION_ONLY_VISIBLE, 0.005f)) {
          // true if a ray from p, directed on the face, can see it. The maxLenght is very small so no other faces can be hit.
          // visible from the inside point p.
          if(visibility == FaceVisibility.VISIBILITY_EXTERIOR) returnFace(i, normalIndex, pointIndex); // Return the triangle.
        }
        else {
          // invisible from the inside point p.
          if(visibility == FaceVisibility.VISIBILITY_INTERIOR) returnFace(i, normalIndex, pointIndex); // Return the triangle.
        }
      }
      else {
        // p is outside.
        if(r.raypickBoolean(p, o_n, Raypicking.SENS_HALF_LINE, Raypicking.INTERSECTION_ONLY_VISIBLE, 0.005f)) {
          // visible from the outside point p.
          if(visibility == FaceVisibility.VISIBILITY_INTERIOR) returnFace(i, normalIndex, pointIndex); // Return the triangle.
        }
        else {
          // invisible from the outside point p.
          if(visibility == FaceVisibility.VISIBILITY_EXTERIOR) returnFace(i, normalIndex, pointIndex); // Return the triangle.
        }
      }
      normalIndex = normalIndex + 3;
      pointIndex = pointIndex + 3 * nbPointsPerFace;
    }
  }
  private synchronized void returnFace(int fi, int ni, int pi) {
    float[] b = new float[4];

    normals[ni    ] = -normals[ni    ];
    normals[ni + 1] = -normals[ni + 1];
    normals[ni + 2] = -normals[ni + 2];

    System.arraycopy(points, pi + 3, b     , 0     , 3); // Exchange the 2nd and the 4th points.
    System.arraycopy(points, pi + 9, points, pi + 3, 3);
    System.arraycopy(b     , 0     , points, pi + 9, 3);

    if(usePointsNormals) {
      System.arraycopy(pointsNormals, pi + 3, b            , 0     , 3);
      System.arraycopy(pointsNormals, pi + 9, pointsNormals, pi + 3, 3);
      System.arraycopy(b            , 0     , pointsNormals, pi + 9, 3);
    }
    if(usePointsColors) {
      int ci = fi * nbPointsPerFace * 4;
      System.arraycopy(pointsColors, ci + 4 , b           , 0      , 4);
      System.arraycopy(pointsColors, ci + 12, pointsColors, ci + 4 , 4);
      System.arraycopy(b           , 0      , pointsColors, ci + 12, 4);
    }
    if(useTexCoords) {
      int ti = fi * nbPointsPerFace * 2;
      System.arraycopy(pointsTexCoords, ti + 2, b              , 0     , 2);
      System.arraycopy(pointsTexCoords, ti + 6, pointsTexCoords, ti + 2, 2);
      System.arraycopy(b              , 0     , pointsTexCoords, ti + 6, 2);
    }
  }
  public synchronized void returnAllFaces() {
    for(int fi = 0; fi < nbFaces; fi++) returnFace(fi, 3 * fi, 12 * fi);
  }


  public synchronized void buildSmoothLit(PointsIdentifier pi) {
    if(!usePointsNormals) return;
    int pointIndex = 0, normalIndex = 0;
    float x1, x2, y1, y2, z1, z2, angle0, angle1, angle2;
    PointRecord pr;
    for(int i = 0; i < nbFaces; i++) {
      x1 = points[pointIndex +  3] - points[pointIndex    ]; // First point.
      y1 = points[pointIndex +  4] - points[pointIndex + 1];
      z1 = points[pointIndex +  5] - points[pointIndex + 2];
      x2 = points[pointIndex +  9] - points[pointIndex    ];
      y2 = points[pointIndex + 10] - points[pointIndex + 1];
      z2 = points[pointIndex + 11] - points[pointIndex + 2];
      if(java.lang.Math.sqrt(x1 * x1 + y1 * y1 + z1 * z1) * java.lang.Math.sqrt(x2 * x2 + y2 * y2 + z2 * z2) == 0) angle0 = 0;
      else {
        x1 = x1 * x2 + y1 * y2 + z1 * z2;
        if(x1 >= 1) angle0 = 0;
        else { 
          if(x1 <= -1) angle0 = Matrix.PI;
          else angle0 = ((float) java.lang.Math.atan(-x1 / java.lang.Math.sqrt(-x1 * x1 + 1))) + 1.5707963267949f;
        }
      }
      pr = pi.searchOrCreate(points[pointIndex], points[pointIndex + 1], points[pointIndex + 2]);
      pr.addNormalComponent(normals[normalIndex], normals[normalIndex + 1], normals[normalIndex + 2], angle0);
      pr.addNormalComputationListener(new WriteInArrayListener(pointIndex));

      x1 = points[pointIndex    ] - points[pointIndex + 3]; // Second point.
      y1 = points[pointIndex + 1] - points[pointIndex + 4];
      z1 = points[pointIndex + 2] - points[pointIndex + 5];
      x2 = points[pointIndex + 6] - points[pointIndex + 3];
      y2 = points[pointIndex + 7] - points[pointIndex + 4];
      z2 = points[pointIndex + 8] - points[pointIndex + 5];
      if(java.lang.Math.sqrt(x1 * x1 + y1 * y1 + z1 * z1) * java.lang.Math.sqrt(x2 * x2 + y2 * y2 + z2 * z2) == 0) angle1 = 0;
      else {
        x1 = x1 * x2 + y1 * y2 + z1 * z2;
        if(x1 >= 1) angle1 = 0;
        else { 
          if(x1 <= -1) angle1 = Matrix.PI;
          else angle1 = ((float) java.lang.Math.atan(-x1 / java.lang.Math.sqrt(-x1 * x1 + 1))) + 1.5707963267949f;
        }
      }
      pr = pi.searchOrCreate(points[pointIndex + 3], points[pointIndex + 4], points[pointIndex + 5]);
      pr.addNormalComponent(normals[normalIndex], normals[normalIndex + 1], normals[normalIndex + 2], angle1);
      pr.addNormalComputationListener(new WriteInArrayListener(pointIndex + 3));

      x1 = points[pointIndex +  3] - points[pointIndex + 6]; // Third point.
      y1 = points[pointIndex +  4] - points[pointIndex + 7];
      z1 = points[pointIndex +  5] - points[pointIndex + 8];
      x2 = points[pointIndex +  9] - points[pointIndex + 6];
      y2 = points[pointIndex + 10] - points[pointIndex + 7];
      z2 = points[pointIndex + 11] - points[pointIndex + 8];
      if(java.lang.Math.sqrt(x1 * x1 + y1 * y1 + z1 * z1) * java.lang.Math.sqrt(x2 * x2 + y2 * y2 + z2 * z2) == 0) angle2 = 0;
      else {
        x1 = x1 * x2 + y1 * y2 + z1 * z2;
        if(x1 >= 1) angle2 = 0;
        else { 
          if(x1 <= -1) angle2 = Matrix.PI;
          else angle2 = ((float) java.lang.Math.atan(-x1 / java.lang.Math.sqrt(-x1 * x1 + 1))) + 1.5707963267949f;
        }
      }
      pr = pi.searchOrCreate(points[pointIndex + 6], points[pointIndex + 7], points[pointIndex + 8]);
      pr.addNormalComponent(normals[normalIndex], normals[normalIndex + 1], normals[normalIndex + 2], angle2);
      pr.addNormalComputationListener(new WriteInArrayListener(pointIndex + 6));

      pr = pi.searchOrCreate(points[pointIndex + 9], points[pointIndex + 10], points[pointIndex + 11]);
      pr.addNormalComponent(normals[normalIndex], normals[normalIndex + 1], normals[normalIndex + 2], (2f * Matrix.PI) - angle0 - angle1 - angle2);
      pr.addNormalComputationListener(new WriteInArrayListener(pointIndex + 9));

      pointIndex = pointIndex + 12;
      normalIndex = normalIndex + 3;
    }
  }
  public void drawFaces(GLFunc gl, GLUFunc glu) throws GLErrorException {
    if(usePointsNormals) gl.glDrawArrays(GLEnum.GL_QUADS, 0, nbNonHiddenFaces * nbPointsPerFace);
    else {
      int pointIndex = 0, normalIndex = 0;
      gl.glBegin(GLEnum.GL_QUADS);
      for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
        gl.glNormal3f(normals[normalIndex++], normals[normalIndex++], normals[normalIndex++]);
        gl.glArrayElement(pointIndex++);
        gl.glArrayElement(pointIndex++);
        gl.glArrayElement(pointIndex++);
        gl.glArrayElement(pointIndex++);
      }
      gl.glEnd();
    }
  }
  public void drawFacesWithoutArrays(GLFunc gl, GLUFunc glu) {
    int pointIndex = 0, normalIndex = 0;
    gl.glBegin(GLEnum.GL_QUADS);
    
    if(usePointsColors) {
      int colorIndex = 0;
      if(useTexCoords) {
        int texCoordIndex = 0;
        if(usePointsNormals) {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
        else {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(normals[normalIndex++], normals[normalIndex++], normals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
      }
      else {
        if(usePointsNormals) {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
        else {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(normals[normalIndex++], normals[normalIndex++], normals[normalIndex++]);
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glColor4f(pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++], pointsColors[colorIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
      }
    }
    else {
      if(useTexCoords) {
        int texCoordIndex = 0;
        if(usePointsNormals) {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
        else {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(normals[normalIndex++], normals[normalIndex++], normals[normalIndex++]);
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glTexCoord2f(pointsTexCoords[texCoordIndex++], pointsTexCoords[texCoordIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
      }
      else {
        if(usePointsNormals) {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            
            gl.glNormal3f(pointsNormals[normalIndex++], pointsNormals[normalIndex++], pointsNormals[normalIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
        else {
          for(int faceIndex = 0; faceIndex < nbNonHiddenFaces; faceIndex++) {
            gl.glNormal3f(normals[normalIndex++], normals[normalIndex++], normals[normalIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
            gl.glVertex3f(points[pointIndex++], points[pointIndex++], points[pointIndex++]);
          }
        }
      }
    }
    gl.glEnd();
  }
  public synchronized FragmentedShapeElement[] getFragmentedShapeElements() {
    FragmentedShapeElement[] faces = new FragmentedShapeElement[nbFaces];
    Position[] ps;
    int pointIndex = 0;

    if(usePointsColors) {
      float[] color = new float[4];
      int pointColorIndex = 0;
      if(useTexCoords) {
        int texIndex = 0;
        for(int i = 0; i < nbFaces; i++) {
          ps = new Position[nbPointsPerFace];
          for(int j = 0; j < nbPointsPerFace; j++) {
            System.arraycopy(pointsColors, pointColorIndex, color, 0, 4); // Color is copied by AdvancedPoint constructor, so we can reuse it.
            pointColorIndex = pointColorIndex + 4;
            ps[j] = (Position) new AdvancedPoint(points[pointIndex++], points[pointIndex++], points[pointIndex++], color, pointsTexCoords[texIndex++], pointsTexCoords[texIndex++]);
          }
          faces[i] = new Quad(ps, material, visibility, smoothLit, staticLit, i >= nbNonHiddenFaces);
        }
      }
      else {
        for(int i = 0; i < nbFaces; i++) {
          ps = new Position[nbPointsPerFace];
          for(int j = 0; j < nbPointsPerFace; j++) {
            System.arraycopy(pointsColors, pointColorIndex, color, 0, 4); // Color is copied by AdvancedPoint constructor, so we can reuse it.
            pointColorIndex = pointColorIndex + 4;
            ps[j] = (Position) new AdvancedPoint(points[pointIndex++], points[pointIndex++], points[pointIndex++], color, 0, 0);
          }
          faces[i] = new Quad(ps, material, visibility, smoothLit, staticLit, i >= nbNonHiddenFaces);
        }
      }
    }
    else {
      if(useTexCoords) {
        int texIndex = 0;
        for(int i = 0; i < nbFaces; i++) {
          ps = new Position[nbPointsPerFace];
          for(int j = 0; j < nbPointsPerFace; j++) {
            ps[j] = (Position) new AdvancedPoint(points[pointIndex++], points[pointIndex++], points[pointIndex++], Material.NO_COLOR, pointsTexCoords[texIndex++], pointsTexCoords[texIndex++]);
          }
          faces[i] = new Quad(ps, material, visibility, smoothLit, staticLit, i >= nbNonHiddenFaces);
        }
      }
      else {
        for(int i = 0; i < nbFaces; i++) {
          ps = new Position[nbPointsPerFace];
          for(int j = 0; j < nbPointsPerFace; j++) {
            ps[j] = (Position) new AdvancedPoint(points[pointIndex++], points[pointIndex++], points[pointIndex++], Material.NO_COLOR, 0, 0);
          }
          faces[i] = new Quad(ps, material, visibility, smoothLit, staticLit, i >= nbNonHiddenFaces);
        }
      }
    }
    return faces;
  }

  // Raypicking : a bunch of mathematics :
  // Each quad is divided into 2 triangles... :-) Have a better idea?    ...Certainly Optimizable.
  public synchronized Position raypick(Position origin, Vector direction, int sens, int intersection) {
    if(material.isWireframed()) return null; // No raypick if wireframe.
    
    int j;
    Position best = null;
    double bestLenght = Float.POSITIVE_INFINITY, currentLenght;  // Squarred lenght used for speed boost.
    float ox = origin.getX(), oy = origin.getY(), oz = origin.getZ();
    float[] m = new float[16];
    m[ 8] = direction.getX(); // Triangle-indepedant values.
    m[ 9] = direction.getY();
    m[10] = direction.getZ();
    m[15] = 1f;
    float[] mi;
    float vx, vy, vz;
    float px, py, pz;
    float anx, any, anz;
    float nx, ny, nz;
    
    if((intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) && (visibility == FaceVisibility.VISIBILITY_ALL)) intersection = Raypicking.INTERSECTION_ALL;
    
    for(int i = 0; i < nbFaces; i++) {
      j = 12 * i;
      
      m[0] = points[j + 3] - points [j    ]; // First.
      m[1] = points[j + 4] - points [j + 1];
      m[2] = points[j + 5] - points [j + 2];
      
      m[4] = points[j + 6] - points [j    ];
      m[5] = points[j + 7] - points [j + 1];
      m[6] = points[j + 8] - points [j + 2];
      
      mi = Matrix.matrixInvert(m);
      if(mi != null) {
        vx = ox - points[j    ];
        vy = oy - points[j + 1];
        vz = oz - points[j + 2];
        
        //p = Matrix.pointMultiplyByMatrix(mi, v); // p coordinate in the triangle.
        px = vx * mi[0] + vy * mi[4] + vz * mi[ 8] + mi[12];
        py = vx * mi[1] + vy * mi[5] + vz * mi[ 9] + mi[13];
        pz = vx * mi[2] + vy * mi[6] + vz * mi[10] + mi[14];
        
        if(px >= 0f && py >= 0f && px + py <= 1f) { // Intersection on the triangle.
          boolean ok = true;
          if((sens == Raypicking.SENS_HALF_LINE) && (pz > 0f)) ok = false; // Not in the good direction. Discard.
          else {
            if(intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) {
              anx = m[1] * m[6] - m[2] * m[5];
              any = m[2] * m[4] - m[0] * m[6];
              anz = m[0] * m[5] - m[1] * m[2];
              //n = Matrix.pointMultiplyByMatrix(mi, n);
              //nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              //ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = anx * mi[2] + any * mi[6] + anz * mi[10] + mi[14]; // nx, ny not used.
              if(nz > 0) ok = false; // Face normal and direction vector have the same direction. Discard.
            }
          }
          if(ok) {
            pz = 0f; // Project p on the triangle.
            //p = Matrix.pointMultiplyByMatrix(m, p); // Convert in space coordinate.
            nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
            ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
            nz = px * m[2] + py * m[6] + pz * m[10] + m[14]  + points[j + 2];
            
            currentLenght = Matrix.pow2(ox - nx) + Matrix.pow2(oy - ny) + Matrix.pow2(oz - nz);
            if(currentLenght < bestLenght) {
              bestLenght = currentLenght;
              if(best == null) best = new Point(nx, ny, nz);
              else best.move(nx, ny, nz);
            }
          }
        }
      }
    }

    for(int i = 0; i < nbFaces; i++) {
      j = 12 * i;

      m[0] = points[j +  6] - points [j    ];
      m[1] = points[j +  7] - points [j + 1];
      m[2] = points[j +  8] - points [j + 2];

      m[4] = points[j +  9] - points [j    ];
      m[5] = points[j + 10] - points [j + 1];
      m[6] = points[j + 11] - points [j + 2];

      mi = Matrix.matrixInvert(m);
      if(mi != null) {
        vx = ox - points[j    ];
        vy = oy - points[j + 1];
        vz = oz - points[j + 2];
        
        //p = Matrix.pointMultiplyByMatrix(mi, v); // p coordinate in the triangle.
        px = vx * mi[0] + vy * mi[4] + vz * mi[ 8] + mi[12];
        py = vx * mi[1] + vy * mi[5] + vz * mi[ 9] + mi[13];
        pz = vx * mi[2] + vy * mi[6] + vz * mi[10] + mi[14];
        
        if(px >= 0f && py >= 0f && px + py <= 1f) { // Intersection on the triangle.
          boolean ok = true;
          if((sens == Raypicking.SENS_HALF_LINE) && (pz > 0f)) ok = false; // Not in the good direction. Discard.
          else {
            if(intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) {
              anx = m[1] * m[6] - m[2] * m[5];
              any = m[2] * m[4] - m[0] * m[6];
              anz = m[0] * m[5] - m[1] * m[2];
              //n = Matrix.pointMultiplyByMatrix(mi, n);
              //nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              //ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = anx * mi[2] + any * mi[6] + anz * mi[10] + mi[14]; // nx, ny not used.
              if(nz > 0) ok = false; // Face normal and direction vector have the same direction. Discard.
            }
          }
          if(ok) {
            pz = 0f; // Project p on the triangle.
            //p = Matrix.pointMultiplyByMatrix(m, p); // Convert in space coordinate.
            nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
            ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
            nz = px * m[2] + py * m[6] + pz * m[10] + m[14]  + points[j + 2];
            
            currentLenght = Matrix.pow2(ox - nx) + Matrix.pow2(oy - ny) + Matrix.pow2(oz - nz);
            if(currentLenght < bestLenght) {
              bestLenght = currentLenght;
              if(best == null) best = new Point(nx, ny, nz);
              else best.move(nx, ny, nz);
            }
          }
        }
      }
    }
    return best;
  }
  public synchronized int raypickNumber(Position origin, Vector direction, int sens, int intersection, float maxLength) {
    if(material.isWireframed()) return 0; // No raypick if wireframe.
    
    if(maxLength != Float.POSITIVE_INFINITY) maxLength = Matrix.pow2(maxLength);
    int nb = 0;
    int j;
    float ox = origin.getX(), oy = origin.getY(), oz = origin.getZ();
    float[] m = new float[16];
    m[ 8] = direction.getX(); // Triangle-indepedant values.
    m[ 9] = direction.getY();
    m[10] = direction.getZ();
    m[15] = 1f;
    float[] mi;
    float vx, vy, vz;
    float px, py, pz;
    float anx, any, anz;
    float nx, ny, nz;
    
    if((intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) && (visibility == FaceVisibility.VISIBILITY_ALL)) intersection = Raypicking.INTERSECTION_ALL;

    for(int i = 0; i < nbFaces; i++) {
      j = 12 * i;
      
      m[0] = points[j + 3] - points [j    ]; // first.
      m[1] = points[j + 4] - points [j + 1];
      m[2] = points[j + 5] - points [j + 2];
      
      m[4] = points[j + 6] - points [j    ];
      m[5] = points[j + 7] - points [j + 1];
      m[6] = points[j + 8] - points [j + 2];
      
      mi = Matrix.matrixInvert(m);
      if(mi != null) {
        vx = ox - points[j    ];
        vy = oy - points[j + 1];
        vz = oz - points[j + 2];
        
        //p = Matrix.pointMultiplyByMatrix(mi, v); // p coordinate in the triangle.
        px = vx * mi[0] + vy * mi[4] + vz * mi[ 8] + mi[12];
        py = vx * mi[1] + vy * mi[5] + vz * mi[ 9] + mi[13];
        pz = vx * mi[2] + vy * mi[6] + vz * mi[10] + mi[14];
        
        if(px >= 0f && py >= 0f && px + py <= 1f) { // Intersection on the triangle.
          boolean ok = true;
          if((sens == Raypicking.SENS_HALF_LINE) && (pz > 0f)) ok = false; // Not in the good direction. Discard.
          else {
            if(intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) {
              anx = m[1] * m[6] - m[2] * m[5];
              any = m[2] * m[4] - m[0] * m[6];
              anz = m[0] * m[5] - m[1] * m[2];
              //n = Matrix.pointMultiplyByMatrix(mi, n);
              //nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              //ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = anx * mi[2] + any * mi[6] + anz * mi[10] + mi[14]; // nx, ny not used.
              if(nz > 0) ok = false; // Face normal and direction vector have the same direction. Discard.
            }
          }
          if(ok) {
            if(maxLength == Float.POSITIVE_INFINITY) {
              nb++;
              continue; // useless to test the second triangle of the quad.
            }
            else {
              pz = 0f; // Project p on the triangle.
              //p = Matrix.pointMultiplyByMatrix(m, p); // Convert in space coordinate.
              nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = px * m[2] + py * m[6] + pz * m[10] + m[14]  + points[j + 2];
              
              if(Matrix.pow2(ox - nx) + Matrix.pow2(oy - ny) + Matrix.pow2(oz - nz) <= maxLength) {
                nb++;
                continue; // useless to test the second triangle of the quad.
              }
            }
          }
        }
      }
    }
    for(int i = 0; i < nbFaces; i++) {
      j = 12 * i;
      
      m[0] = points[j +  6] - points [j    ];
      m[1] = points[j +  7] - points [j + 1];
      m[2] = points[j +  8] - points [j + 2];
      
      m[4] = points[j +  9] - points [j    ];
      m[5] = points[j + 10] - points [j + 1];
      m[6] = points[j + 11] - points [j + 2];
      
      mi = Matrix.matrixInvert(m);
      if(mi != null) {
        vx = ox - points[j    ];
        vy = oy - points[j + 1];
        vz = oz - points[j + 2];
        
        //p = Matrix.pointMultiplyByMatrix(mi, v); // p coordinate in the triangle.
        px = vx * mi[0] + vy * mi[4] + vz * mi[ 8] + mi[12];
        py = vx * mi[1] + vy * mi[5] + vz * mi[ 9] + mi[13];
        pz = vx * mi[2] + vy * mi[6] + vz * mi[10] + mi[14];
        
        if(px >= 0f && py >= 0f && px + py <= 1f) { // Intersection on the triangle.
          boolean ok = true;
          if((sens == Raypicking.SENS_HALF_LINE) && (pz > 0f)) ok = false; // Not in the good direction. Discard.
          else {
            if(intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) {
              anx = m[1] * m[6] - m[2] * m[5];
              any = m[2] * m[4] - m[0] * m[6];
              anz = m[0] * m[5] - m[1] * m[2];
              //n = Matrix.pointMultiplyByMatrix(mi, n);
              //nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              //ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = anx * mi[2] + any * mi[6] + anz * mi[10] + mi[14]; // nx, ny not used.
              if(nz > 0) ok = false; // Face normal and direction vector have the same direction. Discard.
            }
          }
          if(ok) {
            if(maxLength == Float.POSITIVE_INFINITY) {
              nb++;
              continue;
            }
            else {
              pz = 0f; // Project p on the triangle.
              //p = Matrix.pointMultiplyByMatrix(m, p); // Convert in space coordinate.
              nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = px * m[2] + py * m[6] + pz * m[10] + m[14]  + points[j + 2];
              
              if(Matrix.pow2(ox - nx) + Matrix.pow2(oy - ny) + Matrix.pow2(oz - nz) <= maxLength) {
                nb++;
                continue;
              }
            }
          }
        }
      }
    }

    return nb;
  }
  public synchronized boolean raypickBoolean(Position origin, Vector direction, int sens, int intersection, float maxLength) {
    if(material.isWireframed()) return false; // No raypick if wireframe.
    
    if(maxLength != Float.POSITIVE_INFINITY) maxLength = Matrix.pow2(maxLength);
    int nb = 0;
    int j;
    float ox = origin.getX(), oy = origin.getY(), oz = origin.getZ();
    float[] m = new float[16];
    m[ 8] = direction.getX(); // Triangle-indepedant values.
    m[ 9] = direction.getY();
    m[10] = direction.getZ();
    m[15] = 1f;
    float[] mi;
    float vx, vy, vz;
    float px, py, pz;
    float anx, any, anz;
    float nx, ny, nz;
    
    if((intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) && (visibility == FaceVisibility.VISIBILITY_ALL)) intersection = Raypicking.INTERSECTION_ALL;
    
    for(int i = 0; i < nbFaces; i++) {
      j = 12 * i;
      
      m[0] = points[j + 3] - points [j    ]; // first.
      m[1] = points[j + 4] - points [j + 1];
      m[2] = points[j + 5] - points [j + 2];
      
      m[4] = points[j + 6] - points [j    ];
      m[5] = points[j + 7] - points [j + 1];
      m[6] = points[j + 8] - points [j + 2];
      
      mi = Matrix.matrixInvert(m);
      if(mi != null) {
        vx = ox - points[j    ];
        vy = oy - points[j + 1];
        vz = oz - points[j + 2];
        
        //p = Matrix.pointMultiplyByMatrix(mi, v); // p coordinate in the triangle.
        px = vx * mi[0] + vy * mi[4] + vz * mi[ 8] + mi[12];
        py = vx * mi[1] + vy * mi[5] + vz * mi[ 9] + mi[13];
        pz = vx * mi[2] + vy * mi[6] + vz * mi[10] + mi[14];
        
        if(px >= 0f && py >= 0f && px + py <= 1f) { // Intersection on the triangle.
          boolean ok = true;
          if((sens == Raypicking.SENS_HALF_LINE) && (pz > 0f)) ok = false; // Not in the good direction. Discard.
          else {
            if(intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) {
              anx = m[1] * m[6] - m[2] * m[5];
              any = m[2] * m[4] - m[0] * m[6];
              anz = m[0] * m[5] - m[1] * m[2];
              //n = Matrix.pointMultiplyByMatrix(mi, n);
              //nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              //ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = anx * mi[2] + any * mi[6] + anz * mi[10] + mi[14]; // nx, ny not used.
              if(nz > 0) ok = false; // Face normal and direction vector have the same direction. Discard.
            }
          }
          if(ok) {
            if(maxLength == Float.POSITIVE_INFINITY) return true;
            else {
              pz = 0f; // Project p on the triangle.
              //p = Matrix.pointMultiplyByMatrix(m, p); // Convert in space coordinate.
              nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = px * m[2] + py * m[6] + pz * m[10] + m[14]  + points[j + 2];
              
              if(Matrix.pow2(ox - nx) + Matrix.pow2(oy - ny) + Matrix.pow2(oz - nz) <= maxLength)
                return true;
            }
          }
        }
      }
    }
    
    for(int i = 0; i < nbFaces; i++) {
      j = 12 * i;
      
      m[0] = points[j +  6] - points [j    ];
      m[1] = points[j +  7] - points [j + 1];
      m[2] = points[j +  8] - points [j + 2];
      
      m[4] = points[j +  9] - points [j    ];
      m[5] = points[j + 10] - points [j + 1];
      m[6] = points[j + 11] - points [j + 2];
      
      mi = Matrix.matrixInvert(m);
      if(mi != null) {
        vx = ox - points[j    ];
        vy = oy - points[j + 1];
        vz = oz - points[j + 2];
        
        //p = Matrix.pointMultiplyByMatrix(mi, v); // p coordinate in the triangle.
        px = vx * mi[0] + vy * mi[4] + vz * mi[ 8] + mi[12];
        py = vx * mi[1] + vy * mi[5] + vz * mi[ 9] + mi[13];
        pz = vx * mi[2] + vy * mi[6] + vz * mi[10] + mi[14];
        
        if(px >= 0f && py >= 0f && px + py <= 1f) { // Intersection on the triangle.
          boolean ok = true;
          if((sens == Raypicking.SENS_HALF_LINE) && (pz > 0f)) ok = false; // Not in the good direction. Discard.
          else {
            if(intersection == Raypicking.INTERSECTION_ONLY_VISIBLE) {
              anx = m[1] * m[6] - m[2] * m[5];
              any = m[2] * m[4] - m[0] * m[6];
              anz = m[0] * m[5] - m[1] * m[2];
              //n = Matrix.pointMultiplyByMatrix(mi, n);
              //nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              //ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = anx * mi[2] + any * mi[6] + anz * mi[10] + mi[14]; // nx, ny not used.
              if(nz > 0) ok = false; // Face normal and direction vector have the same direction. Discard.
            }
          }
          if(ok) {
            if(maxLength == Float.POSITIVE_INFINITY) return true;
            else {
              pz = 0f; // Project p on the triangle.
              //p = Matrix.pointMultiplyByMatrix(m, p); // Convert in space coordinate.
              nx = px * m[0] + py * m[4] + pz * m[ 8] + m[12]  + points[j    ];
              ny = px * m[1] + py * m[5] + pz * m[ 9] + m[13]  + points[j + 1];
              nz = px * m[2] + py * m[6] + pz * m[10] + m[14]  + points[j + 2];
              
              if(Matrix.pow2(ox - nx) + Matrix.pow2(oy - ny) + Matrix.pow2(oz - nz) <= maxLength)
                return true;
            }
          }
        }
      }
    }
    return false;
  }
}
