/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.event.*;
import java.io.*;
import java.beans.*;
import java.util.Iterator;

/**
 * Abstract class for all polygon-like shape element.
 * 
 * See Triangle and Quad for concrete class.
 * 
 * @author Artiste on the Web
 */

public abstract class Face extends AbstractBean implements FragmentedShapeElement, FaceVisibility, Lit, Lockable {
  /**
   * Creates a new face.
   */
  public Face() {  }
  /**
   * Creates a new face, from the given array of points.
   * @param p the points
   */
  public Face(Position[] p) { this(p, null); }
  /**
   * Creates a new face, from the given array of points and material.
   * @param p the points
   * @param m the material
   */
  public Face(Position[] p, Material m) { this(p, m, FaceVisibility.VISIBILITY_ALL, false); }
  /**
   * Creates a new face, from the given array of points, material, visibility and lit.
   * @param p the points
   * @param m the material
   * @param newVisibility the visibility
   * @param newSmoothLit true for smooth-lighting
   */
  public Face(Position[] ps, Material m, int newVisibility, boolean newSmoothLit) {
    this(ps, m, newVisibility, newSmoothLit, false);
  }
  /**
   * Creates a new face, from the given array of points, material, visibility, lit and hidden.
   * @param ps the points
   * @param m the material
   * @param newVisibility the visibility
   * @param newSmoothLit true for smooth-lighting
   * @param newHidden true for hidden
   */
  public Face(Position[] ps, Material m, int newVisibility, boolean newSmoothLit, boolean newHidden) {
    visibility = newVisibility;
    smoothLit = newSmoothLit;

    for(int i = 0; i < getNumberOfPoints(); i++) {
      Position pos = ps[i];
      points[i] = pos;
      if(pos != null) pos.addPropertyChangeListener(pointsListener);
    }
    material = m;
    hidden = newHidden;
  }
  /**
   * Creates a new face, from the given array of points, material, visibility, lit and hidden.
   * @param ps the points
   * @param m the material
   * @param newVisibility the visibility
   * @param newSmoothLit true for smooth-lighting
   * @param newStaticLit true if static lit
   * @param newHidden true for hidden
   */
  public Face(Position[] ps, Material m, int newVisibility, boolean newSmoothLit, boolean newStaticLit, boolean newHidden) {
    visibility = newVisibility;
    smoothLit = newSmoothLit;
    staticLit = newStaticLit;
    
    for(int i = 0; i < getNumberOfPoints(); i++) {
      Position pos = ps[i];
      points[i] = pos;
      if(pos != null) pos.addPropertyChangeListener(pointsListener);
    }
    material = m;
    hidden = newHidden;
  }

  /**
   * Gets the number of points in this face. It must return a constant value.
   * @return the number of points
   */
  public abstract int getNumberOfPoints();

  private final Position[] points = new Position[getNumberOfPoints()];
  /**
   * Gets the point of the given index.
   * @param index the index, from 0 to getNumberOfPoints() - 1
   * @return the point
   */
  public Position getPoint(int index) { return points[index]; }
  private synchronized void putPoint(int index, Position p) {
    Position oldP = points[index];
    if(oldP != null) oldP.removePropertyChangeListener(pointsListener);
    points[index] = p;
    if(p != null) p.addPropertyChangeListener(pointsListener);
  }
  /**
   * Sets the point of the given index.
   * @param index the index, from 0 to getNumberOfPoints() - 1
   * @param p the new point
   */
  public void setPoint(int index, Position p) {
    putPoint(index, p);
    firePropertyChange("point");
  }
  /**
   * Gets all the points.
   * @return an array of points
   */
  public Position[] getPoints() { return points; }
  /**
   * Sets all the points.
   * @param ps the new points, the array must have at least getNumberOfPoints() elements
   */
  public void setPoints(Position[] ps) {
    for(int i = 0; i < getNumberOfPoints(); i++) putPoint(i, ps[i]);
    firePropertyChange("point");
  }
  /**
   * Adds the given point at the first null point of this face.
   * @param p the new point
   */
  public void addPoint(Position p) {
    synchronized(this) {
      for(int i = 0; i < getNumberOfPoints(); i++) {
        if(points[i] == null) {
          putPoint(i, p);
          firePropertyChange("point");
          break;
        }
      }
    }
  }

  /**
   * Gets the normal vector of this face. The vector is normalized.
   * @return the normal vector
   */
  public Vector getNormal() { // Compute the normal from the 3 first points (a face should be plane).
    float x0, y0, z0;
    float x1, y1, z1;
    float x2, y2, z2;
    Position p;

    p = points[0];
    x0 = p.getX();
    y0 = p.getY();
    z0 = p.getZ();

    p = points[1];
    x1 = p.getX() - x0;
    y1 = p.getY() - y0;
    z1 = p.getZ() - z0;

    p = points[2];
    x2 = p.getX() - x0;
    y2 = p.getY() - y0;
    z2 = p.getZ() - z0;

    Vector v = new Vector(y1 * z2 - z1 * y2, z1 * x2 - x1 * z2, x1 * y2 - y1 * x2);
    v.normalize();
    return v;
  }

  private Position center;
  private class CenterMoveListener implements /*MoveListener*/ PropertyChangeListener {
    private Position oldCenter = new Point(center);
    public synchronized void propertyChange(PropertyChangeEvent e) {
      if((e instanceof MoveEvent) && (!updating)) {
        addVector(new Vector(oldCenter, center));
        oldCenter.move(center);
      }
    }
    private boolean updating;
    public synchronized void update(float x, float y, float z) {
      updating = true ;
      center   .move(x, y, z);
      oldCenter.move(x, y, z);
      updating = false;
    }
  }
  private CenterMoveListener centerMoveListener;
  /**
   * Gets the center of this face. In order to move all the face, you can move the center.
   * @return the center
   */
  public synchronized Position getCenter() {
    if(center == null) {
      center = new Point();
      centerMoveListener = new CenterMoveListener();
      center.addPropertyChangeListener(centerMoveListener);
      updateCenter();
    }
    return center;
  }
  /**
   * Update the position of the center of the face. Should be call if a point is moved.
   */
  private synchronized void updateCenter() {
    if(center != null) {
      float x = 0f, y = 0f, z = 0f;
      int nb = points.length;
      for(int i = 0; i < nb; i++) {
        x = x + points[i].getX();
        y = y + points[i].getY();
        z = z + points[i].getZ();
      }
      x = x / nb;
      y = y / nb;
      z = z / nb;
      
      centerMoveListener.update(x, y, z);
    }
  }
  /**
   * Translate the face (=all the points in the face) with the given vector.
   * @param v the vector
   */
  public void addVector(Vector v) {
    for(int i = 0; i < points.length; i++) points[i].addVector(v);
  }

  protected transient Material material = Material.WHITE_MATERIAL;
  /**
   * Gets the material of this face.
   * @return the material
   */
  public Material getMaterial() { return material; }
  /**
   * Sets the material of this face.
   * @param m the new material
   */
  public void setMaterial(Material m) {
    if(material != m) {
      if(changeListeners != null) {
        synchronized(changeListeners) {
          for(Iterator i = changeListeners.iterator(); i.hasNext(); ) material.removePropertyChangeListener((PropertyChangeListener) i.next());
        }
      }
      material = m;
      if(changeListeners != null) {
        synchronized(changeListeners) {
          for(Iterator i = changeListeners.iterator(); i.hasNext(); ) material.addPropertyChangeListener((PropertyChangeListener) i.next());
        }
      }
      firePropertyChange("material");
    }
  }

  /**
   * Checks if this face uses alpha. It uses alpha if one of its points does.
   * @return true if uses alpha
   */
  public synchronized boolean getUseAlpha() {
    if(material.getUseAlpha()) return true;
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) {
        if(((Colored) points[i]).getUseAlpha()) return true;
      }
    }
    return false;
  }
  /**
   * Checks if this face uses color (in addition to its material color). It uses color if one
   * of its points does.
   * @return true if uses color
   */
  public boolean getUseColor() {
    for(int i = 0; i < getNumberOfPoints(); i++) {
      if(points[i] instanceof Colored) {
        if(((Colored) points[i]).getUseColor()) return true;
      }
    }
    return false;
  }

  protected boolean smooth; // TODO
  protected boolean smoothLit;
  public boolean isSmoothLit() { return smoothLit; }
  public synchronized void setSmoothLit(boolean b) {
    if(smoothLit != b) {
      smoothLit = b;
      firePropertyChange("smoothLit");
    }
  }

  protected boolean staticLit;
  public boolean isStaticLit() { return staticLit; }
  public synchronized void setStaticLit(boolean b) {
    if(staticLit != b) {
      staticLit = b;
      firePropertyChange("staticLit");
    }
  }
  /** Not supported. */
  public void applyStaticLighting(Light3D[] lights, CoordSyst fromCoordSyst) {
    throw new UnsupportedOperationException();
  }
  
  protected int visibility;
  public int getVisibility() { return visibility; }
  public synchronized void setVisibility(int b) {
    if(visibility != b) {
      visibility = b;
      firePropertyChange("visibility");
    }
  }
  public boolean canShareFragmentWith(FragmentedShapeElement se) {
    if(se.getClass    () != getClass()) return false;
    Face f = (Face) se;
    if(f.material        != material     ) return false;
    if(f.visibility      != visibility   ) return false;
    if(f.smoothLit       != smoothLit    ) return false;
    if(f.staticLit       != staticLit    ) return false;
    if(f.getUseAlpha  () != getUseAlpha()) return false;
    if(f.getUseColor  () != getUseColor()) return false;
    return true;
  }

  /**
   * Scale this face.
   * @param f the scale factor
   */
  public void scale(float f) {
    int nb = points.length;
    for(int i = 0; i < nb; i++) {
      points[i].move(points[i].getX() * f, points[i].getY() * f, points[i].getZ() * f);
    }
  }
  /**
   * Scale this face.
   * @param fx the x scale factor
   * @param fy the y scale factor
   * @param fz the z scale factor
   */
  public void scale(float fx, float fy, float fz) {
    int nb = points.length;
    for(int i = 0; i < nb; i++) {
      points[i].move(points[i].getX() * fx, points[i].getY() * fy, points[i].getZ() * fz);
    }
  }
  
  public synchronized void revert() {
    Position[] ps = (Position[]) points.clone();
    int nb = points.length;
    for(int i = 0; i < nb; i++) points[i] = ps[nb - i - 1];
    firePropertyChange("points");
  }
  
  private boolean hidden;
  /**
   * Checks if this face is hidden. Default is false.
   * A hidden face is never drawn, but it is taken into account for raypicking, or
   * visibility, ...
   * @return true if hidden
   */
  public boolean isHidden() { return hidden; }
  /**
   * Sets if this face is hidden.
   * @param true if hidden
   */
  public void setHidden(boolean b) {
    hidden = b;
    firePropertyChange("hidden");
  }
  
  // Transformable :
  public synchronized void preTransform(float[] m) {
    lock(); // Avoid a lot of event.
    int nb = points.length;
    for(int i = 0; i < nb; i++) if(points[i] != null) points[i].transform(m);
    unlock();
  }
  
  // Serializable :
  private void writeObject(ObjectOutputStream s) throws IOException {
    s.defaultWriteObject();
    material.write(s);
  }
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
    s.defaultReadObject();
    material = Material.read(s);
    pointsListener = new MyListener();
    int nb = points.length;
    for(int i = 0; i < nb; i++) if(points[i] != null) points[i].addPropertyChangeListener(pointsListener);
  }
  
  // Lockable :
  protected transient int lockLevel;
  /**
   * Checks if this face is locked.
   * @return true if locked
   */
  public boolean isLocked() { return lockLevel > 0; }
  /**
   * Locks this face. A locked face will send no event until it will be unlocked. If you
   * intend to change or move a lot this face, you should lock it, change it and unlock it.
   * Only a single event will be sent, when unlocking, so there may be a performance boost
   * (in particular if the event treatment is complex). But locking element for nothing can
   * waste time.
   * Notice that locking is cumulative : if you lock twice an object, you need to unlock it
   * twice before it can be refreshed or rebuilt.
   * The lock-state is not saved in serialization.
   */
  public synchronized void lock() { lockLevel++; }
  /**
   * Unlocks this face.
   */
  public synchronized void unlock() {
    if(lockLevel > 0) {
      lockLevel--;
      if(lockLevel == 0) {
        updateCenter();
        firePropertyChange();
      }
    }
  }

  // Overrides :
  /**
   * Checks if it is really necessary to fire an event. This method returns false if there
   * is no listener or if the object is lock, else it returns true.
   * @return true if firing an event if necessary
   */
  protected boolean isWorthFiringEvent() {
    if(lockLevel > 0) return false;
    return super.isWorthFiringEvent();
  }
  
  public void setColor(float[] color) {
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) ((Colored) points[i]).setColor(color);
    }
  }
  public void setColor(float red, float green, float blue, float alpha) {
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) ((Colored) points[i]).setColor(red, green, blue, alpha);
    }
  }
  public void setRed(float f) {
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) {
        Colored c = (Colored) points[i];
        if(!c.getUseColor()) c.setAlpha(1f);
        c.setRed(f);
      }
    }
  }
  public void setGreen(float f) {
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) {
        Colored c = (Colored) points[i];
        if(!c.getUseColor()) c.setAlpha(1f);
        c.setGreen(f);
      }
    }
  }
  public void setBlue(float f) {
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) {
        Colored c = (Colored) points[i];
        if(!c.getUseColor()) c.setAlpha(1f);
        c.setBlue(f);
      }
    }
  }
  public void setAlpha(float f) {
    for(int i = 0; i < points.length; i++) {
      if(points[i] instanceof Colored) {
        Colored c = (Colored) points[i];
        c.setAlpha(f);
      }
    }
  }
  
  // Event :
  public void addPropertyChangeListener(PropertyChangeListener l) {
    super.addPropertyChangeListener(l);
    material.addPropertyChangeListener(l);
  }
  public void removePropertyChangeListener(PropertyChangeListener l) {
    super.removePropertyChangeListener(l);
    material.removePropertyChangeListener(l);
  }

  private class MyListener implements PropertyChangeListener {
    public void propertyChange(PropertyChangeEvent e) {
      if(lockLevel == 0) {
        if(e instanceof MoveEvent) updateCenter();
        Face.this.firePropertyChange(e);
      }
    }
  }
  protected void firePropertyChange(PropertyChangeEvent e) {
    super.firePropertyChange(e);
  }
  
  private transient MyListener pointsListener = new MyListener();
}
