/*
 * Soya3D
 * Copyright (C) 1999-2000  Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.model;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import java.io.*;
import java.beans.*;

/**
 * A point that is colored and textured.
 * 
 * Advanced points are very often, if not always, used with modeling, into face for example.
 * 
 * @author Artiste on the Web
 */

public class AdvancedPoint extends Point implements Colored, Textured {
  /**
   * Creates a new advanced point.
   */
  public AdvancedPoint() { super(); }
  /**
   * Creates a new advanced point, at the given location.
   * @param x the x coordinate
   * @param y the y coordinate
   * @param z the z coordinate
   */
  public AdvancedPoint(float x, float y, float z) { super(x, y, z, null); }
  /**
   * Creates a new advanced point, at the given location.
   * @param x the x coordinate
   * @param y the y coordinate
   * @param z the z coordinate
   * @param frame the frame where the coordinates are defined
   */
  public AdvancedPoint(float x, float y, float z, CoordSyst coordSyst) { super(x, y, z, coordSyst); }
  /**
   * Creates a new advanced point, at the given location.
   * @param x the x coordinate
   * @param y the y coordinate
   * @param z the z coordinate
   * @param newColor the point color
   * @param texX the x texture coordinate
   * @param texY the y texture coordinate
   */
  public AdvancedPoint(float x, float y, float z, float[] newColor, float texX, float texY) {
    super(x, y, z, null);
    texCoordX = texX;
    texCoordY = texY;
    System.arraycopy(newColor, 0, color, 0, 4);
  }
  /**
   * Creates a new advanced point, at the given location.
   * @param x the x coordinate
   * @param y the y coordinate
   * @param z the z coordinate
   * @param texX the x texture coordinate
   * @param texY the y texture coordinate
   */
  public AdvancedPoint(float x, float y, float z, float texX, float texY) {
    super(x, y, z, null);
    texCoordX = texX;
    texCoordY = texY;
  }
  /**
   * Creates a new advanced point, at the given location.
   * @param p the position
   */
  public AdvancedPoint(Position p) { super(p); }
  /**
   * Creates a new advanced point, at the given location.
   * @param p the position
   * @param newColor the point color
   * @param texX the x texture coordinate
   * @param texY the y texture coordinate
   */
  public AdvancedPoint(Position p, float[] newColor, float texX, float texY) {
    super(p);
    texCoordX = texX;
    texCoordY = texY;
    System.arraycopy(newColor, 0, color, 0, 4);
  }

  private final float[] color = (float[]) Material.NO_COLOR.clone();
  public float[] getColor() {	return color; }
  public void setColor(float red, float green, float blue) { setColor(red, green, blue, 1f); }
  public void setColor(float red, float green, float blue, float alpha) {
    color[0] = red;
    color[1] = green;
    color[2] = blue;
    color[3] = alpha;
    firePropertyChange("color");
  }
  public void setColor(float[] c) {
    System.arraycopy(c, 0, color, 0, 4);
    firePropertyChange("color");
  }
  public boolean getUseAlpha() { return ((color[3] != -1) && (color[3] != 1)); }
  public boolean getUseColor() { return (color[3] != -1); }
  public void setUseColor(boolean b) {
    if(b != getUseColor()) {
      if(b) {
        color[3] = 1;
        firePropertyChange("color");
      }
      else setColor(Material.NO_COLOR);
    }
    firePropertyChange("useColor");
  }
  public float getRed  () { return color[0]; }
  public float getGreen() { return color[1]; }
  public float getBlue () { return color[2]; }
  public float getAlpha() { return color[3]; }
  public void setRed  (float f) {
    color[0] = f;
    firePropertyChange("color");
  }
  public void setGreen(float f) {
    color[1] = f;
    firePropertyChange("color");
  }
  public void setBlue (float f) {
    color[2] = f;
    firePropertyChange("color");
  }
  public void setAlpha(float f) {
    color[3] = f;
    firePropertyChange("color");
  }

  private float texCoordX, texCoordY;
  public float getTextureCoordX() { return texCoordX; }
  public void setTextureCoordX(float newCoord) {
    texCoordX = newCoord;
    firePropertyChange("textureCoordX");
  }
  public float getTextureCoordY() { return texCoordY; }
  public void setTextureCoordY(float newCoord) {
    texCoordY = newCoord;
    firePropertyChange("textureCoordY");
  }
  public void setTextureCoords(float newCoordX, float newCoordY) {
    texCoordX = newCoordX;
    texCoordY = newCoordY;
    firePropertyChange("textureCoordX");
    firePropertyChange("textureCoordY");
  }

  //Overrides :
  public boolean equals(Object o) {
    if(o instanceof AdvancedPoint) {
      if(!super.equals(o)) return false;
      AdvancedPoint ap = (AdvancedPoint) o;
      float[] apColor = ap.getColor();
      if((texCoordX == ap.getTextureCoordX()) && (texCoordY == ap.getTextureCoordY()) && (color[0] == apColor[0]) && (color[1] == apColor[1]) && (color[2] == apColor[2]) && (color[3] == apColor[3])) return true;
    }
    return false;
  }
}
