/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx.particle;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * A particles drawer that draws particles as sprites.
 *
 * @author Artiste on the Web
 */

public class SpriteParticlesDrawer extends ParticlesDrawer {
  private static final long serialVersionUID = -407006984161228544l;
  /**
   * Create a sprites particles drawer.
   */
  public SpriteParticlesDrawer() {
    super();
  }
  /**
   * Create a sprites particles drawer.
   * @param m the sprite's material
   */
  public SpriteParticlesDrawer(Material m) {
    this();
    setMaterial(m);
  }
  /**
   * Create a particles drawer.
   * @param m the sprite's material
   * @param x1 x1
   * @param y1 y1
   * @param x2 x2
   * @param y2 y2
   */
  public SpriteParticlesDrawer(Material m, float x1, float y1, float x2, float y2) {
    this();
    setMaterial(m);
    this.x1 = x1;
    this.y1 = y1;
    this.x2 = x2;
    this.y2 = y2;
  }
  
  /**
   * Clones this particles drawer.
   * @return the clone
   */
  public Object clone() {
    SpriteParticlesDrawer o = (SpriteParticlesDrawer) super.clone();
    o.material = material;
    o.x1 = x1;
    o.y1 = y1;
    o.x2 = x2;
    o.y2 = y2;
    
    return o;
  }
  
  // Overrides :
  protected transient float[] matrix0;
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) {
    matrix0 = (float[]) mat.clone();
    
    mat[ 0] = 1;
    mat[ 1] = 0;
    mat[ 2] = 0;
    
    mat[ 4] = 0;
    mat[ 5] = 1;
    mat[ 6] = 0;
    
    mat[ 8] = 0;
    mat[ 9] = 0;
    mat[10] = 1;
    super.fillCollector(f, r, mat);
  }
  protected void drawBunch(Renderer r, GLFunc gl, GLUFunc glu) {
    gl.glDisable(GLEnum.GL_LIGHTING);
    gl.glEnable (GLEnum.GL_COLOR_MATERIAL);
    gl.glDisable(GLEnum.GL_CULL_FACE);
    bunch.colors.makeColorCurrent(gl, glu);
    
    float x0, y0, z0;
    float x , y , z ;
    
    gl.glBegin(GLEnum.GL_QUADS);
    for(int i = 0; i < nbParticles; i++) {
      if(bunch.lifes.lifes[i] > 0f) {
        x0 = bunch.positions[i * 3    ];
        y0 = bunch.positions[i * 3 + 1];
        z0 = bunch.positions[i * 3 + 2];
        x  = x0 * matrix0[0] + y0 * matrix0[4] + z0 * matrix0[ 8];
        y  = x0 * matrix0[1] + y0 * matrix0[5] + z0 * matrix0[ 9];
        z  = x0 * matrix0[2] + y0 * matrix0[6] + z0 * matrix0[10];
        bunch.colors.makeParticleColorCurrent(gl, glu, i);
        gl.glTexCoord2f(0f, 0f); gl.glVertex3f(x + x1, y + y1, z);
        gl.glTexCoord2f(1f, 0f); gl.glVertex3f(x + x2, y + y1, z);
        gl.glTexCoord2f(1f, 1f); gl.glVertex3f(x + x2, y + y2, z);
        gl.glTexCoord2f(0f, 1f); gl.glVertex3f(x + x1, y + y2, z);
      }
    }
    gl.glEnd();
    
    gl.glEnable (GLEnum.GL_LIGHTING);
    gl.glEnable (GLEnum.GL_CULL_FACE);
    gl.glDisable(GLEnum.GL_COLOR_MATERIAL);
    material.makeColorCurrent(gl, glu);
  }
  
  // Particles properties :
  
  // A few properties.
  protected transient Material material = Material.WHITE_MATERIAL;
  /**
   * Gets the material used for drawing the sprites.
   * Default is Material.WHITE_MATERIAL.
   * @return the sprites' material
   */
  public Material getMaterial() { return material; }
  /**
   * Sets the material used for drawing the sprites.
   * @param m the new sprites' material
   */
  public void setMaterial(Material m) { 
    if(m == null) material = Material.WHITE_MATERIAL;
    else          material = m;
    firePropertyChange("material");
  }
  
  protected float x1 = -.5f, y1 = -.5f, x2 = .5f, y2 = .5f;
  /**
   * Gets the sprites' x1 value. x1 represents the left coordinate of the sprite, relative to
   * the particle coordinates. The sprite's width is x2 - x1.
   * Default is -0.5f .
   * @return the x1 value
   */
  public float getX1() { return x1; }
  /**
   * Gets the sprites' y1 value. y1 represents the top coordinate of the sprite, relative to
   * the particle coordinates. The sprite's height is y2 - y1.
   * Default is -0.5f .
   * @return the y1 value
   */
  public float getY1() { return y1; }
  /**
   * Gets the sprites' x2 value. x2 represents the right coordinate of the sprite, relative to
   * the particle coordinates. The sprite's width is x2 - x1.
   * Default is 0.5f .
   * @return the x2 value
   */
  public float getX2() { return x2; }
  /**
   * Gets the sprites' y2 value. y2 represents the bottom coordinate of the sprite, relative to
   * the particle coordinates. The sprite's height is y2 - y1.
   * Default is 0.5f .
   * @return the y2 value
   */
  public float getY2() { return y2; }
  /**
   * Sets the sprites' x1 value.
   * @param f the new x1 value
   */
  public void setX1(float f) {
    x1 = f;
    firePropertyChange("x1");
  }
  /**
   * Sets the sprites' y1 value.
   * @param f the new y1 value
   */
  public void setY1(float f) {
    y1 = f;
    firePropertyChange("y1");
  }
  /**
   * Sets the sprites' x2 value.
   * @param f the new x2 value
   */
  public void setX2(float f) {
    x2 = f;
    firePropertyChange("x2");
  }
  /**
   * Sets the sprites' y2 value.
   * @param f the new y2 value
   */
  public void setY2(float f) {
    y2 = f;
    firePropertyChange("y2");
  }
  
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
    s.defaultReadObject();
    material = Material.read(s);
  }
  private void writeObject(ObjectOutputStream s) throws IOException {
    s.defaultWriteObject();
    material.write(s);
  }
}
