/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx.particle;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.fx.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * A bunch of particles. This is an abstract class for all particles system.
 * 
 * The particles bunch manages the positions of all the particles.
 * 
 * @author Artiste on the Web
 */

public abstract class ParticlesBunch3D extends FragmentElement3D implements Enable, ParticlesStuff {
  private static final long serialVersionUID = -7369089538627803626l;
  /**
   * Create a particles bunch.
   */
  public ParticlesBunch3D() { super(); }
  /**
   * Create a particles bunch.
   * @param l the object that manages the particles' lifes
   * @param c the object that manages the particles' colors
   * @param d the object that draws the particles
   */
  public ParticlesBunch3D(ParticlesLifes l, ParticlesColors c, ParticlesDrawer d) {
    this();
    lifes  = l; lifes .bunch = this;
    colors = c; colors.bunch = this;
    drawer = d; drawer.bunch = this;
  }
  
  /**
   * Clones this particles bunch.
   * @return the clone
   */
  public Object clone() {
    ParticlesBunch3D o = (ParticlesBunch3D) super.clone();
    o.enabled = enabled;
    o.setParticlesLifes  ((ParticlesLifes ) lifes .clone());
    o.setParticlesColors ((ParticlesColors) colors.clone());
    o.setParticlesDrawer ((ParticlesDrawer) drawer.clone());
    
    o.setNumberOfParticles(nbParticles);
    
    return o;
  }
  
  protected ParticlesLifes lifes;
  /**
   * Gets the object that manages the particles' lifes.
   * @return the particles' lifes object
   */
  public ParticlesLifes getParticlesLifes() { return lifes; }
  /**
   * Sets the object that manages the particles' lifes.
   * @param l the new particles' lifes object
   */
  public void setParticlesLifes(ParticlesLifes l) {
    lifes = l;
    lifes.bunch = this;
    firePropertyChange("particlesLifes");
  }
  
  protected ParticlesColors colors;
  /**
   * Gets the object that manages the particles' colors.
   * @return the particles' colors object
   */
  public ParticlesColors getParticlesColors() { return colors; }
  /**
   * Sets the object that manages the particles' colors.
   * @param d the new particles' colors object
   */
  public void setParticlesColors(ParticlesColors d) {
    colors = d;
    colors.bunch = this;
    firePropertyChange("particlesColors");
  }
  
  protected ParticlesDrawer drawer;
  /**
   * Gets the object that draws the particles.
   * @return the particles drawer
   */
  public ParticlesDrawer getParticlesDrawer() { return drawer; }
  /**
   * Sets the object that draws the particles.
   * @param d the new particles drawer
   */
  public void setParticlesDrawer(ParticlesDrawer d) {
    drawer = d;
    drawer.bunch = this;
    firePropertyChange("particlesDrawer");
  }
  
  // Overrides : hackish :
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) { // Optimizable
    if(visible) {
      if(leftHanded) f.invertConfiguration();
      if(coordSyst == this) mat = Matrix.matrixMultiply(mat, m);
      else {
        if(coordSyst == null)
          mat = Matrix.matrixMultiply(r.modelMatrix(), getRootParent().getRootMatrix());
        else
          mat = Matrix.matrixMultiply(r.modelMatrix(), coordSyst.getRootMatrix()      );
      }
      drawer.fillCollector(f, r, mat);
      if(leftHanded) f.invertConfiguration();
    }
  }
  public void draw(Renderer r, GLFunc gl, GLUFunc glu) { // Draw the sphere with the glu function.
    advanceBunch(r.fpsFactor() * speed);
    drawer.drawBunch(r, gl, glu);
  }
  
  public void initParticle(int id) {
    lifes .initParticle(id);
    colors.initParticle(id);
    drawer.initParticle(id);
    nbParticlesAlive++;
  }
  /**
   * Inits the position of the particle of the given ID.
   * @param id the particle ID
   * @param position the new particle position
   */
  protected void initParticlePosition(int id, Position position) {
    if((position.getCoordSyst() != this) && (position.getCoordSyst() != null)) position = position.clone(this);
    positions[id * 3    ] = position.getX();
    positions[id * 3 + 1] = position.getY();
    positions[id * 3 + 2] = position.getZ();
  }
  /**
   * Inits the position of the particle of the given ID.
   * @param id the particle ID
   * @param x the new particle x coordinate
   * @param y the new particle y coordinate
   * @param z the new particle z coordinate
   */
  protected void initParticlePosition(int id, float x, float y, float z) {
    id = id * 3;
    if(coordSyst == this) {
      positions[id    ] = x;
      positions[id + 1] = y;
      positions[id + 2] = z;
    }
    else {
      if(coordSyst == null) {
        float[] m = getRootMatrix();
        positions[id    ] = x * m[0] + y * m[4] + z * m[ 8] + m[12];
        positions[id + 1] = x * m[1] + y * m[5] + z * m[ 9] + m[13];
        positions[id + 2] = x * m[2] + y * m[6] + z * m[10] + m[14];
        
      }
      else {
        Position p = new Point(x, y, z, this);
        p.setCoordSyst(coordSyst);
        positions[id    ] = p.getX();
        positions[id + 1] = p.getY();
        positions[id + 2] = p.getZ();
      }
    }
  }
  protected int nbParticlesCreated;
  public void advanceBunch(float factor) {
    nbParticlesCreated = 0;
    for(int i = 0; i < nbParticles; i++) advanceParticle(i, factor);
  }
  public void advanceParticle(int id, float factor) {
    if(lifes.isAlive(id)) {
      lifes .advanceParticle(id, factor);
      colors.advanceParticle(id, factor);
      drawer.advanceParticle(id, factor);
    }
    else {
      if(enabled && (nbParticlesCreated < maximumParticlesCreatedPerRendering)) {
        nbParticlesCreated++;
        initParticle(id);
      }
    }
  }
  public void deleteParticle(int id) {
    lifes .deleteParticle(id);
    colors.deleteParticle(id);
    drawer.deleteParticle(id);
    
    nbParticlesAlive--;
    //if(enabled) initParticle(id);
  }
  
  public ParticlesBunch3D getBunch() { return this; }
  
  // Particles properties :
  protected float[] positions;
  /**
   * Gets the x coordinate of the particle of the given ID.
   * @return the x coordinate
   */
  public float getParticleX(int id) { return positions[id * 3    ]; }
  /**
   * Gets the y coordinate of the particle of the given ID.
   * @return the y coordinate
   */
  public float getParticleY(int id) { return positions[id * 3 + 1]; }
  /**
   * Gets the z coordinate of the particle of the given ID.
   * @return the z coordinate
   */
  public float getParticleZ(int id) { return positions[id * 3 + 2]; }
  
  // A few properties.
  protected int nbParticlesAlive;
  public int numberOfParticlesAlive() { return nbParticlesAlive; }
  
  protected int nbParticles;
  public int getNumberOfParticles() { return nbParticles; }
  public void setNumberOfParticles(int i) {
    int first;
    if(positions == null) {
      positions = new float[3 * i];
      first = 0;
    }
    else {
      float[] positions2 = new float[3 * i];
      if(nbParticles < i)
        System.arraycopy(positions, 0, positions2, 0, 3 * nbParticles);
      else
        System.arraycopy(positions, 0, positions2, 0, 3 * i);
      positions = positions2;
      first = nbParticles;
    }
    nbParticles = i;
    
    lifes .setNumberOfParticles(i);
    colors.setNumberOfParticles(i);
    drawer.setNumberOfParticles(i);
    
    if(enabled) for(int j = first; j < i; j++) initParticle(j);
    
    firePropertyChange("numberOfParticles");
  }
  
  protected boolean enabled = true;
  /**
   * Checks if this particles bunch is enabled. If so, dead particles will be automaticaly
   * re-created.
   * Default is true.
   * @return true if enabled
   */
  public boolean isEnabled() { return enabled; }
  public void setEnabled(boolean b) {
    enabled = b;
    firePropertyChange("enabled");
  }
  
  protected float speed = 1f;
  /**
   * Gets the speed factor. Default is 1f.
   * @return the speed factor
   */
  public float getSpeed() { return speed; }
  /**
   * Sets the speed factor.
   * @param f the new speed factor
   */
  public void setSpeed(float f) {
    speed = f;
    firePropertyChange("speed");
  }
  
  protected int maximumParticlesCreatedPerRendering = 10;
  /**
   * Gets the maximum number of particles that can be created each rendering.
   * Default is 10.
   * @return the maximum number of particles created each rendering
   */
  public int getMaximumParticlesCreatedPerRendering() { return maximumParticlesCreatedPerRendering; }
  /**
   * Sets the maximum number of particles that can be created each rendering.
   * @param i the new maximum number of particles created each rendering
   */
  public void setMaximumParticlesCreatedPerRendering(int i) {
    maximumParticlesCreatedPerRendering = i;
    firePropertyChange("maximumParticlesCreatedPerRendering");
  }
  
  /*
  protected GraphicalElement3D coordSyst = this;
  public GraphicalElement3D getParticlesCoordSyst() { return coordSyst; }
  public void setParticlesCoordSyst(GraphicalElement3D ge) {
    coordSyst = ge;
    firePropertyChange("particlesCoordSyst");
  }
  */
  protected CoordSyst coordSyst = this;
  /**
   * Gets the coordinate system where the particles' positions are defined.
   * Default is this.
   * You might want to use a different value, for example if you want to move the particle
   * bunch in order to move the particles' source, without moving all the particles.
   * If this property is null, the root parent will be used (A usefull tips if you don't want
   * to save the root parent with this object !).
   * @return the particles' coordinate system
   */
  public CoordSyst getParticlesCoordSyst() { return coordSyst; }
  /**
   * Sets the coordinate system where the particles' positions are defined.
   * @param cs the new particles' coordinate system
   */
  public void setParticlesCoordSyst(CoordSyst cs) {
    coordSyst = cs;
    firePropertyChange("particlesCoordSyst");
  }
  /**
   * Checks if the particles' coordinates are relative to the particle bunch or absolute,
   * that is to say if the particlesCoordSyst property is == to this. In this case, moving
   * the particles bunch will move all the particles; else it will move only newly created
   * ones.
   * @return true if particlesCoordSyst == this
   */
  public boolean areParticleCoordinatesRelative() { return coordSyst == this; }
  /**
   * Sets if the particles' coordinates are relative to the particle bunch or absolute.
   * setParticleCoordinatesRelative(true) is equivalent to setParticlesCoordSyst(this), and
   * setParticleCoordinatesRelative(false) is equivalent to setParticlesCoordSyst(null).
   * @param b true if relative
   */
  public void setParticleCoordinatesRelative(boolean b) {
    if(b) setParticlesCoordSyst(this);
    else  setParticlesCoordSyst(null);
  }
  
  /**
   * Deletes all the particles.
   */
  public void deleteAllParticles() {
    for(int i = 0; i < nbParticles; i++) deleteParticle(i);
  }
  
  /**
   * Creates all the particles.
   */
  public void createAllParticles() {
    for(int i = 0; i < nbParticles; i++) {
      if(!lifes.isAlive(i)) initParticle(i);
    }
  }
  
  public Material getMaterial() { return drawer.getMaterial(); }
  public boolean getUseAlpha() { return colors.getUseAlpha() || drawer.getUseAlpha(); }
  
  public float getNaturalWidth () { return 0f; }
  public float getNaturalHeight() { return 0f; }
  public float getNaturalDepth () { return 0f; }
  public DimensionWrapper wrapper() {
    return new Box(new Point(0f, 0f, 0f, this), new Point(0f, 0f, 0f, this));
  }
}
