/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx.particle;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import opale.soya.soya3d.fx.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * A bunch of moving particles, like a flame.
 * 
 * Flame particles bunch uses its own particles colors manager, so you don'thave to specify
 * yours.
 * 
 * Default acceleration is new Vector(0f, .0005f, 0f, this), so the particles go up.
 * 
 * @author Artiste on the Web
 */

public class Flame3D extends Fountain3D /* implements */ {
  private static final long serialVersionUID = -898967425184737726l;
  /**
   * Create a flame particles bunch.
   */
  public Flame3D() {
    super();
    createParticlesColors();
    minSpeed = .02f;
    maxSpeed = .03f;
    angle    = 100f; //120f;
    acceleration.move(0f, .00005f, 0f);
  }
  private void createParticlesColors() {
    colors = new FadingParticlesColors() {
      public void initParticle(int id) {  } // No-op. Color is speed-dependant, and defined elsewhere
    };
    colors.setAlpha2(0f);
    setParticlesColors(colors);
  }
  /**
   * Create a flame particles bunch, with a standard particles' lifes and a sprites particles
   * drawer. This is the easiest way to create a nice fire effect.
   * @param material the material for sprite drawing.
   */
  public Flame3D(Material material) {
    this();
    setParticlesLifes (new ParticlesLifes       (.016f, .022f              ));
    setParticlesDrawer(new SpriteParticlesDrawer(material, -.2f, -.2f, .2f, .2f));
    setNumberOfParticles(80);
  }
  /**
   * Create a flame particles bunch.
   * @param l the object that manages the particles' lifes
   * @param d the object that draws the particles
   */
  public Flame3D(ParticlesLifes l, ParticlesDrawer d) {
    this();
    setParticlesLifes (l);
    setParticlesDrawer(d);
  }
  
  protected FadingParticlesColors colors;
  
  /**
   * Clones this fountain particles bunch.
   * @return the clone
   */
  public Object clone() {
    Flame3D o = (Flame3D) super.clone();
    System.arraycopy(color1, 0, o.color1, 0, 4);
    System.arraycopy(color2, 0, o.color2, 0, 4);
    
    return o;
  }
  
  public void initParticle(int id) {
    //super.initParticle(id);
    lifes .initParticle(id);
    colors.initParticle(id);
    drawer.initParticle(id);
    
    initParticlePosition(id, 0f, 0f, 0f);
    
    double angle1 = Math.random() * 6.28318531d; // 0 to 360
    double angle2 = .0174533d * (Math.random() * (double) angle) + .35d;
    //double angle2 = .0174533d * (Math.random() * (double) (angle - 20f)) + .35d;
    float  length = ((float) Math.random()) * (maxSpeed - minSpeed) + minSpeed;
    
    double sin = Math.sin(angle2);
    
    initParticleSpeed(id,
                      length * (float) (sin * Math.cos(angle1)),
                      length * (float) (      Math.cos(angle2)),
                      length * (float) (sin * Math.sin(angle1))
                      );
    
    float f1 = Math.abs(((float) angle2) - 1.57f), f2 = 1f - f1;
    
    colors.colors[id * 4    ] =   f1 * color1[0] + f2 * color2[0];
    colors.colors[id * 4 + 1] =   f1 * color1[1] + f2 * color2[1];
    colors.colors[id * 4 + 2] =   f1 * color1[2] + f2 * color2[2];
    colors.colors[id * 4 + 3] =   f1 * color1[3] + f2 * color2[3];
    
    if(angle2 > 1.2d) lifes.lifeSpeeds[id] = 2f * lifes.lifeSpeeds[id];
  }
  
  private transient float centerX, centerZ;
  public void advanceBunch(float factor) {
    if(coordSyst == this)
      centerX = centerZ = 0f;
    else {
      if(coordSyst == null) {
        float[] m = getRootMatrix();
        centerX = m[12];
        centerZ = m[14];
      }
      else {
        Position p = new Point(0f, 0f, 0f, this);
        p.setCoordSyst(coordSyst);
        centerX = p.getX();
        centerZ = p.getZ();
      }
    }
    
    super.advanceBunch(factor);
  }
  public void advanceParticle(int id, float factor) {
    super.advanceParticle(id, factor);
    
    id = id * 3;
    if(positions[id    ] > centerX)
      speeds[id    ]    = speeds[id    ] - factor * horizontalDeceleration;
    else
      speeds[id    ]    = speeds[id    ] + factor * horizontalDeceleration;
    if(positions[id + 2] > centerZ)
      speeds[id + 2]    = speeds[id + 2] - factor * horizontalDeceleration;
    else
      speeds[id + 2]    = speeds[id + 2] + factor * horizontalDeceleration;
  }
  
  // Particles properties :
  
  // Bunch properties :
  protected float horizontalDeceleration = .0005f;
  /**
   * Gets the horizontal deceleration of the particles. All particles share the same valuefor
   * this property. The horizontal deceleration represent how the particles horizontal
   * (= x and z) speed is diminushed with time. The higher it is, the more the flame is conic
   * and thin.
   * Default is .0005f.
   * @return the horizontal deceleration.
   */
  public float getHorizontalDeceleration() { return horizontalDeceleration; }
  /**
   * Sets the horizontal deceleration of the particles.
   * @param f the new horizontal deceleration.
   */
  public void setHorizontalDeceleration(float f) {
    horizontalDeceleration = f;
    firePropertyChange("horizontalDeceleration");
  }
    
  protected final float[] color1 = { .9f, .9f, .2f, .7f }, color2 = { 1f, 0.01f, .05f, .7f };
  public float[] getColor1() {	return color1; }
  public float[] getColor2() {	return color2; }
  public void setColor1(float red, float green, float blue) { setColor1(red, green, blue, 1f); }
  public void setColor2(float red, float green, float blue) { setColor2(red, green, blue, 1f); }
  public void setColor1(float red, float green, float blue, float alpha) {
    color1[0] = red;
    color1[1] = green;
    color1[2] = blue;
    color1[3] = alpha;
    firePropertyChange("color1");
  }
  public void setColor2(float red, float green, float blue, float alpha) {
    color2[0] = red;
    color2[1] = green;
    color2[2] = blue;
    color2[3] = alpha;
    firePropertyChange("color2");
  }
  public void setColor1(float[] c) {
    System.arraycopy(c, 0, color1, 0, 4);
    firePropertyChange("color1");
  }
  public void setColor2(float[] c) {
    System.arraycopy(c, 0, color2, 0, 4);
    firePropertyChange("color2");
  }
  public float getRed1  () { return color1[0]; }
  public float getRed2  () { return color2[0]; }
  public float getGreen1() { return color1[1]; }
  public float getGreen2() { return color2[1]; }
  public float getBlue1 () { return color1[2]; }
  public float getBlue2 () { return color2[2]; }
  public float getAlpha1() { return color1[3]; }
  public float getAlpha2() { return color2[3]; }
  public void setRed1  (float f) {
    color1[0] = f;
    firePropertyChange("color1");
  }
  public void setRed2  (float f) {
    color2[0] = f;
    firePropertyChange("color2");
  }
  public void setGreen1(float f) {
    color1[1] = f;
    firePropertyChange("color1");
  }
  public void setGreen2(float f) {
    color2[1] = f;
    firePropertyChange("color2");
  }
  public void setBlue1 (float f) {
    color1[2] = f;
    firePropertyChange("color1");
  }
  public void setBlue2 (float f) {
    color2[2] = f;
    firePropertyChange("color2");
  }
  public void setAlpha1(float f) {
    color1[3] = f;
    firePropertyChange("color1");
  }
  public void setAlpha2(float f) {
    color2[3] = f;
    firePropertyChange("color2");
  }
}
