/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.fx;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.io.*;
import java.beans.*;

/**
 * A fragment / element that is a speed effect. For weapons like swords,...
 *
 * @author Artiste on the Web
 */

public class SpeedEffect3D extends FragmentElement3D implements Colored, Enable {
  private static final long serialVersionUID = 2705588478323608469l;
  
  /**
   * Create a speed effect.
   */
  public SpeedEffect3D() { super(); init(); }
  /**
   * Create a speed effect with the given name.
   * @param name the name
   */
  public SpeedEffect3D(String name) { super(name); init(); }
  
  /**
   * Clones this speed effect.
   * @return the clone
   */
  public Object clone() {
    SpeedEffect3D o = (SpeedEffect3D) super.clone();
    o.enabled = enabled;
    System.arraycopy(color, 0, o.color, 0, 4);
    o.p1.setCoordSyst(null); o.p1.move(p1); o.p1.setCoordSyst(o);
    o.p2.setCoordSyst(null); o.p2.move(p2); o.p2.setCoordSyst(o);
    o.material = material;
    o.setLength(length);
    return o;
  }
  
  // Overrides : hackish :
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) { // Optimizable
    if(visible && (first != length)) {
      if(leftHanded) f.invertConfiguration();
      mat = Matrix.matrixMultiply(r.modelMatrix(), getRootParent().getRootMatrix());
      f.collect(this, mat);
      if(leftHanded) f.invertConfiguration();
    }
  }
  
  public void draw(Renderer r, GLFunc gl, GLUFunc glu) { // Draw the sphere with the glu function.
    if((p1 == null) || (p2 == null)) return;
    
    float f = 0f, af = 0f;
    int j;
    
    advance();
    
    gl.glDisable(GLEnum.GL_CULL_FACE);
    gl.glDisable(GLEnum.GL_LIGHTING);
    gl.glEnable (GLEnum.GL_COLOR_MATERIAL);
    gl.glBlendFunc(GLEnum.GL_SRC_ALPHA, GLEnum.GL_ONE);
    
    gl.glColor4fv(color);
    gl.glBegin(GLEnum.GL_QUADS);
    
    if(material == Material.WHITE_MATERIAL) {
      for(int i = first; i < last - 1; i++) {
        j = 3 * i;
        af = f;
        f = ((float) (i + 1)) / ((float) length);
        
        gl.glColor4f(color[0], color[1], color[2], 0f);
        gl.glTexCoord2f(af,  1f); gl.glVertex3f(ps1 [j    ], ps1 [j + 1], ps1 [j + 2]);
        gl.glColor4f(color[0], color[1], color[2], color[3] * (1f - af));
        gl.glTexCoord2f(af, .5f); gl.glVertex3f(ps12[j    ], ps12[j + 1], ps12[j + 2]);
        gl.glColor4f(color[0], color[1], color[2], color[3] * (1f -  f));
        gl.glTexCoord2f( f, .5f); gl.glVertex3f(ps12[j + 3], ps12[j + 4], ps12[j + 5]);
        gl.glColor4f(color[0], color[1], color[2], 0f);
        gl.glTexCoord2f( f,  1f); gl.glVertex3f(ps1 [j + 3], ps1 [j + 4], ps1 [j + 5]);
        
        gl.glColor4f(color[0], color[1], color[2], 0f);
        gl.glTexCoord2f(af,  0f); gl.glVertex3f(ps2 [j    ], ps2 [j + 1], ps2 [j + 2]);
        gl.glColor4f(color[0], color[1], color[2], color[3] * (1f - af));
        gl.glTexCoord2f(af, .5f); gl.glVertex3f(ps12[j    ], ps12[j + 1], ps12[j + 2]);
        gl.glColor4f(color[0], color[1], color[2], color[3] * (1f -  f));
        gl.glTexCoord2f( f, .5f); gl.glVertex3f(ps12[j + 3], ps12[j + 4], ps12[j + 5]);
        gl.glColor4f(color[0], color[1], color[2], 0f);
        gl.glTexCoord2f( f,  0f); gl.glVertex3f(ps2 [j + 3], ps2 [j + 4], ps2 [j + 5]);
      }
    }
    else {
      for(int i = first; i < last - 1; i++) {
        j = 3 * i;
        gl.glTexCoord2f(f, 1f); gl.glVertex3f(ps1[j    ], ps1[j + 1], ps1[j + 2]);
        gl.glTexCoord2f(f, 0f); gl.glVertex3f(ps2[j    ], ps2[j + 1], ps2[j + 2]);
        
        f = ((float) (i + 1)) / ((float) length);
        
        gl.glColor4f(color[0], color[1], color[2], color[3] * (1f - f));
        gl.glTexCoord2f(f, 0f); gl.glVertex3f(ps2[j + 3], ps2[j + 4], ps2[j + 5]);
        gl.glTexCoord2f(f, 1f); gl.glVertex3f(ps1[j + 3], ps1[j + 4], ps1[j + 5]);
      }
    }
    gl.glEnd();
    
    gl.glEnable (GLEnum.GL_CULL_FACE);
    gl.glEnable (GLEnum.GL_LIGHTING);
    gl.glBlendFunc(GLEnum.GL_SRC_ALPHA, GLEnum.GL_ONE_MINUS_SRC_ALPHA);
    gl.glDisable(GLEnum.GL_COLOR_MATERIAL);
    material.makeColorCurrent(gl, glu);
  }
  protected void advance() {
    System.arraycopy(ps1 , 0, ps1 , 3, (length - 1) * 3);
    System.arraycopy(ps2 , 0, ps2 , 3, (length - 1) * 3);
    System.arraycopy(ps12, 0, ps12, 3, (length - 1) * 3);
    
    if(enabled) {
      World3D root = getRootParent();
      Position p = p1.clone(root);
      ps1[0] = p.getX();
      ps1[1] = p.getY();
      ps1[2] = p.getZ();
      p.setCoordSyst(null);
      p.move(p2);
      p.setCoordSyst(p2.getCoordSyst());
      p.setCoordSyst(root);
      ps2[0] = p.getX();
      ps2[1] = p.getY();
      ps2[2] = p.getZ();
      
      ps12[0] = (ps1[0] + ps2[0]) / 2f;
      ps12[1] = (ps1[1] + ps2[1]) / 2f;
      ps12[2] = (ps1[2] + ps2[2]) / 2f;
      
      if(last  != length) last ++;
    }
    else {
      if(first != length) first++;
    }
  }
  
  // A few properties.
  protected boolean enabled = true;
  protected int first, last;
  public boolean isEnabled() { return enabled; }
  public void setEnabled(boolean b) {
    if(enabled == b) return;
    enabled = b;
    if(b) {
      first = 0;
      last  = 0;
    }
    else first = 0;
    firePropertyChange("enabled");
  }
  
  protected int length = 10;
  public int getLength() { return length; }
  public void setLength(int i) {
    length = i;
    if(first > length) first = length;
    if(last  > length) last  = length;
    init();
    firePropertyChange("length");
  }
  
  protected Position p1 = new Point(0f, 0f, 0f, this), p2 = new Point(0f, 0f, -1f, this);
  public Position getPoint1() { return p1; }
  public Position getPoint2() { return p2; }
  public void setPoint1(Position p) {
    p1 = p;
    firePropertyChange("point1");
  }
  public void setPoint2(Position p) {
    p2 = p;
    firePropertyChange("point2");
  }
  public float getPoint1X() { return p1.getX(); }
  public float getPoint1Y() { return p1.getY(); }
  public float getPoint1Z() { return p1.getZ(); }
  public float getPoint2X() { return p2.getX(); }
  public float getPoint2Y() { return p2.getY(); }
  public float getPoint2Z() { return p2.getZ(); }
  public void setPoint1X(float f) { p1.setX(f); }
  public void setPoint1Y(float f) { p1.setY(f); }
  public void setPoint1Z(float f) { p1.setZ(f); }
  public void setPoint2X(float f) { p2.setX(f); }
  public void setPoint2Y(float f) { p2.setY(f); }
  public void setPoint2Z(float f) { p2.setZ(f); }
  
  protected float[] ps1, ps2, ps12;
  protected void init() {
    ps1  = new float[3 * length];
    ps2  = new float[3 * length];
    ps12 = new float[3 * length];
  }
  
  protected final float[] color = (float[]) Material.WHITE_COLOR.clone();
  public float[] getColor() {	return color; }
  public void setColor(float red, float green, float blue) { setColor(red, green, blue, 1f); }
  public void setColor(float red, float green, float blue, float alpha) {
    color[0] = red;
    color[1] = green;
    color[2] = blue;
    color[3] = alpha;
    firePropertyChange("color");
  }
  public void setColor(float[] c) {
    System.arraycopy(c, 0, color, 0, 4);
    firePropertyChange("color");
  }
  public boolean getUseColor() { return true; }
  public float getRed  () { return color[0]; }
  public float getGreen() { return color[1]; }
  public float getBlue () { return color[2]; }
  public float getAlpha() { return color[3]; }
  public void setRed  (float f) {
    color[0] = f;
    firePropertyChange("color");
  }
  public void setGreen(float f) {
    color[1] = f;
    firePropertyChange("color");
  }
  public void setBlue (float f) {
    color[2] = f;
    firePropertyChange("color");
  }
  public void setAlpha(float f) {
    color[3] = f;
    firePropertyChange("color");
  }
  
  protected transient Material material = Material.WHITE_MATERIAL;
  public Material getMaterial() { return material; }
  public void setMaterial(Material m) {
    material = m;
    reBuild();
    firePropertyChange("material");
  }
  public boolean getUseAlpha() { return true; }
  
  public float getNaturalWidth () { return 0f; }
  public float getNaturalHeight() { return 0f; }
  public float getNaturalDepth () { return 0f; }
  public DimensionWrapper wrapper() {
    return new Box(new Point(0f, 0f, 0f, this), new Point(0f, 0f, 0f, this));
  }
  
  private void readObject(ObjectInputStream s) throws IOException, ClassNotFoundException {
    s.defaultReadObject();
    material = Material.read(s);
    init();
  }
  private void writeObject(ObjectOutputStream s) throws IOException {
    s.defaultWriteObject();
    material.write(s);
  }
}
