/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d.animation;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.*;
import java.io.*;
import java.beans.*;
import java.util.Iterator;

public class Animation extends AbstractBean implements NamedObject {
  private static final long serialVersionUID = 9095055258270805046l;
  public Animation() {  }
  public Animation(String name) { setName(name); }
  
  public Object clone() { // TODO
    Animation a = new Animation();
    a.name = name;
    a.firstTime = firstTime;
    a.lastTime  = lastTime ;
    a.cyclicLap = cyclicLap;
    int nb = movements.length;
    a.movements = new Movement[nb];
    for(int i = 0; i < nb; i++) a.movements[i] = (Movement) movements[i].clone();
    return a;
  }
  
  private String name;
  public String getName() { return name; }
  public boolean hasName() { return ((name != null) && (!"".equals(name))); }
  public synchronized void setName(String n) { 
    String oldName = name;
    name = n; 
    firePropertyChange("name", oldName, name);
  }

  public boolean isEmpty() { return movements.length == 0; }

  private float firstTime = Float.POSITIVE_INFINITY, lastTime = Float.NEGATIVE_INFINITY;
  public float getFirstTime() { return firstTime; }
  public float getLastTime () { return lastTime ; }
  public float getFirstAvalaibleTime() { return firstTime; }
  public float getLastAvalaibleTime () { return lastTime + cyclicLap; }

  private float cyclicLap = 0f;
  public float getCyclicLap() { return cyclicLap; }
  public boolean isCyclic() { return cyclicLap > 0; }
  public synchronized void setCyclicLap(float f) { 
    cyclicLap = f;
    int l = movements.length;
    for(int i = 0; i < l; i++) movements[i].setCyclicLap(cyclicLap);
    firePropertyChange("cyclicLap");
  }

  private Movement[] movements = new Movement[0];
  public synchronized float[] times() {
    if(movements.length == 0) return new float[0];
    else return movements[0].times();
  }
  public synchronized Movement getMovement(GraphicalElement3D e) { 
    int l = movements.length;
    Movement m = e.getMovement();
    if(m != null) {
      for(int i = 0; i < l; i++) if(movements[i] == m) return m;
    }
    String n = e.getName();
    if(n != null && n != "") {
      m = getMovement(n);
      if(m != null) return m;
    }
    return null;
  }
  public synchronized Movement getMovement(String elementName) {
    int l = movements.length;
    for(int i = 0; i < l; i++) {
      if(movements[i].getElementName().equals(elementName)) return movements[i];
    }
    return null;
  }
  public synchronized void addCurrentState(float time, World3D w) {
    if(time < firstTime) firstTime = time;
    if(time > lastTime ) lastTime  = time;
    
    Movement m;
    if(w.getAnimation() == this) {
      for(Iterator i = w.graphicalsRecursive(); i.hasNext(); ) {
        GraphicalElement3D e = (GraphicalElement3D) i.next();
        m = getMovement(e);
        if(m == null) { // New, should linked it to the element.
          m = addNewMovement(e);
          m.addState(time, e);
          e.setMovement(m);
        }
        else m.addState(time, e);
      }
    }
    else {
      for(Iterator i = w.graphicalsRecursive(); i.hasNext(); ) {
        GraphicalElement3D e = (GraphicalElement3D) i.next();
        if(!e.hasName()) throw new UnsupportedOperationException("Soya can only animate element that have a name (all elements should have a different name), or that are already in an animated world.");
        m = getMovement(e.getName()); // Always search by name.
        if(m == null) addNewMovement(e).addState(time, e);
        else m.addState(time, e);
      }
    }
    firePropertyChange("movements");
  }
  private synchronized Movement addNewMovement(GraphicalElement3D e) {
    Movement m = new Movement(e.getName());
    Movement[] movements2 = new Movement[movements.length + 1];
    System.arraycopy(movements, 0, movements2, 0, movements.length);
    movements2[movements.length] = m;
    movements = movements2;
    return m;
  }
  public synchronized void removeState(float time) {
    for(int i = 0; i < movements.length; i++) {
      movements[i].removeState(time);
    }
    if(time == firstTime || time == lastTime) redefineBoundTimes();
    firePropertyChange("movements");
  }
  private synchronized void redefineBoundTimes() {
    float[] times = times();
    if(times.length == 0) {
      firstTime = Float.POSITIVE_INFINITY;
      lastTime  = Float.NEGATIVE_INFINITY;
    }
    else {
      firstTime = times[0];
      lastTime  = times[times.length - 1];
    }
  }
  public synchronized void clear() {
    firstTime = Float.POSITIVE_INFINITY;
    lastTime  = Float.NEGATIVE_INFINITY;
    movements = new Movement[0];
  }
  
  public void scale(float t) {
    if(t == 0f) return;
    for(int i = 0; i < movements.length; i++) movements[i].scale(t);
    firstTime = firstTime * t;
    lastTime  = lastTime  * t;
    cyclicLap = cyclicLap * t;
    firePropertyChange("movements");
  }
  public void changeTime(float oldTime, float newTime) {
    for(int i = 0; i < movements.length; i++) movements[i].changeTime(oldTime, newTime);
    if(firstTime == oldTime) firstTime = newTime;
    if(lastTime  == oldTime) lastTime  = newTime;
    firePropertyChange("movements");
  }
  
  public String toString() {
    String s;
    s = "Animation : name : " + name + "\n";
    s = s + movements.length + " movements : {\n";
    for(int i = 0; i < movements.length; i++) s = s + movements[i].toString() + "\n";
    s = s + "}";
    return s;
  }
}
