/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;

/**
 * An orientation is an object that can rotate and that have a front direction (=its -z axis,
 * if applicable).
 * 
 * @author Atrtiste on the Web
 */

public interface Orientation extends Bean, Rotable {
  /**
   * Constants for rotation type.
   */
  public final static int ROTATION_TYPE_INTERNAL = 1;
  public final static int ROTATION_TYPE_EXTERNAL = 2;
  
  /**
   * Returns the coordinates system in witch this orientation is defined.
   * @return the coordinates system.
   */
  public CoordSyst getCoordSyst();
  
  /**
   * Gets the rotation type (a ROTATION_TYPE_* constant). It is used by some rotations
   * (but not by look at) methods. It defines in which coordinates system the x, y or z
   * axis are defined, for rotateLateral, rotateVertical and rotateIncline method.
   * If rotation type is set to ROTATION_TYPE_INTERNAL, it use the orientation's axis.
   * If it is set to ROTATION_TYPE_EXTERNAL, it use its parent's axis.
   * The default is ROTATION_TYPE_INTERNAL.
   * ROTATION_TYPE_EXTERNAL is often usefull for camera.
   * @return the rotation type
   */
  public int getRotationType();
  /**
   * Sets the rotation type.
   * @param newRotationType the new rotation type
   */
  public void setRotationType(int newRotationType);
  
  /**
   * Resets this orientation.
   */
  public void resetOrientation();
  
  /**
   * Rotates this orientation laterally : around the y axis. The effect depends on the
   * rotation type.
   * @param angle the angle of the rotation
   */
  public void rotateLateral(float angle);
  /**
   * Rotates this orientation vertically : around the x axis. The effect depends on the
   * rotation type.
   * @param angle the angle of the rotation
   */
  public void rotateVertical(float angle);
  /**
   * Rotates this orientation around the z axis. The effect depends on the rotation type.
   * @param angle the angle of the rotation
   */
  public void rotateIncline(float angle);
  /**
   * Rotates this orientation around the given axis. The effect does not depend on the
   * rotation type.
   * @param axe the axe of the rotation
   * @param angle the angle of the rotation
   */
  public void rotate(Vector axe, float angle);
  
  /**
   * Rotates this orientation in order that it will look at the given position. The front
   * of the orientation is considered to be the -z axis. The lookAt always maintain the y
   * axis of the orientation as the upper axis.
   * @param p the position to look at
   */
  public void lookAt(Position p);
  /**
   * Rotates this orientation in order that it will look in the direction of the given
   * Vector. The front of the orientation is considered to be the -z axis. The lookAt
   * always maintain the y axis of the orientation as the upper axis.
   * @param v the vector to look at
   */
  public void lookAt(Vector v);
  /**
   * Gets the position at which this orientation looks at. This is a new position and not
   * a property : it may return different position if you call this method twice; and you
   * cannot move the returned position to change the look-at-direction. Use lookAt.
   * @return the position to look at
   */
  public Position target();
  
  /**
   * Fires a rotate property change event. Such an event is automatically fired when the
   * orientation has changed; you shoud not have to use this method.
   * @see opale.soya.soya3d.event.OrientateEvent
   */
  public void fireOrientate();
  /**
   * Fires a rotate property change event for the given property name. Such an event is
   * automatically fired when the orientation has changed; you shoud not have to use this
   * method.
   * @see opale.soya.soya3d.event.OrientateEvent
   */
  public void fireOrientate(String propertyName);
  /**
   * Fires a rotate property change event for the given property name, and the given old
   * and new values. Such an event is automatically fired when the orientation has changed;
   * you shoud not have to use this method.
   * @see opale.soya.soya3d.event.OrientateEvent
   */
  public void fireOrientate(String propertyName, Object oldValue, Object newValue);
}
