/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.io.*;

public class Light3D extends GraphicalElement3D implements Colored, DrawablesCollectorFiller {
  private static final long serialVersionUID = -6529334610866936288L;
  /**
   * Creates a new lignt.
   */
  public Light3D()                                      { super(); }
  /**
   * Creates a new lignt of the given color.
   * @param c the color in a float[4]
   */
  public Light3D(float[] c)                             { super(); System.arraycopy(c, 0, color, 0, 4); }
  /**
   * Creates a new lignt of the given name.
   * @param newName the name
   */
  public Light3D(String newName)                        { super(newName); }
  /**
   * Creates a new lignt of the given name and color.
   * @param newName the name
   * @param c the color in a float[4]
   */
  public Light3D(String newName, float[] c)             { super(newName); System.arraycopy(c, 0, color, 0, 4); }
  
  /**
   * Clones this light.
   * @see opale.soya.soya3d.GraphicalElement3D#clone
   * @see opale.soya.soya3d.Element3D#clone
   * @return the clone
   */
  public Object clone() {
    Light3D l = (Light3D) super.clone();
    System.arraycopy(coords   , 0, l.coords   , 0, 4);
    System.arraycopy(direction, 0, l.direction, 0, 4);
    System.arraycopy(color    , 0, l.color    , 0, 4);
    l.constant = constant;
    l.linear = linear;
    l.quadratic = quadratic;
    l.type = type;
    l.spotAngle = spotAngle;
    l.spotExponent = spotExponent;
    l.dynamic = dynamic;
    return l;
  }
  
  // FragmentsCollectorFiller : (Light3D is not really a filler, but it must be called before each rendering...
  private final float[] coords = { 0f, 0f, 0f, 1f };
  private static final float[] direction = { 0f, 0f, -1f, 1f };
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) { // Optimizable.
    if(visible && !pureStatic) {
      GLFunc  gl  = r.getGLFunc ();
      GLUFunc glu = r.getGLUFunc();
      
      int id = r.newLightID(dynamic);
      mat = Matrix.matrixMultiply(mat, m);
      gl.glLoadMatrixf(mat);
      if(id == 0) return;
      gl.glLightfv(id, GLEnum.GL_POSITION      , coords      );
      gl.glLightfv(id, GLEnum.GL_SPOT_DIRECTION, direction   );
      gl.glLightf (id, GLEnum.GL_SPOT_EXPONENT , spotExponent);
      gl.glLightf (id, GLEnum.GL_SPOT_CUTOFF   , spotAngle   );
      gl.glLightfv(id, GLEnum.GL_DIFFUSE       , color       );
      //gl.glLightfv(id, GLEnum.GL_AMBIENT       , color       );
      gl.glLightfv(id, GLEnum.GL_SPECULAR      , color       );
      //gl.glLightfv(id, GLEnum.GL_SPECULAR      , ones        );
      gl.glLightf (id, GLEnum.GL_CONSTANT_ATTENUATION , constant );
      gl.glLightf (id, GLEnum.GL_LINEAR_ATTENUATION   , linear   );
      gl.glLightf (id, GLEnum.GL_QUADRATIC_ATTENUATION, quadratic);
      gl.glEnable(id);
    }
  }
  
  // Dimension :
  public float getWidth () { return 0f; }
  public float getHeight() { return 0f; }
  public float getDepth () { return 0f; }
  public DimensionWrapper wrapper() { return new Box(new Point(0f, 0f, 0f, this), new Point(0f, 0f, 0f, this)); }
  
  private float constant, linear = 1f, quadratic;
  /**
   * Gets the attenuation's constant factor. The light is attenuated by this factor.
   * Default is 0f .
   * @return the constant factor of attenuation
   */
  public float getConstantAttenuation() { return constant; }
  /**
   * Sets the attenuation's constant factor.
   * @param f the new constant factor of attenuation
   */
  public void setConstantAttenuation(float f) {
    constant = f;
    firePropertyChange("constantAttenuation");
  }
  /**
   * Gets the attenuation's linear factor. The light is attenuated by this factor * the
   * distance of the light source.
   * Default is 1f .
   * @return the linear factor of attenuation
   */
  public float getLinearAttenuation() { return linear; }
  /**
   * Sets the attenuation's linear factor.
   * @param f the new linear factor of attenuation
   */
  public void setLinearAttenuation(float f) {
    linear = f;
    firePropertyChange("linearAttenuation");
  }
  /**
   * Gets the attenuation's quadratic factor. The light is attenuated by this factor * (the
   * distance of the light source).
   * Default is 0f .
   * @return the quadratic factor of attenuation
   */
  public float getQuadraticAttenuation() { return quadratic; }
  /**
   * Sets the attenuation's quadratic factor.
   * @param f the new quadratic factor of attenuation
   */
  public void setQuadraticAttenuation(float f) {
    quadratic = f;
    firePropertyChange("quadraticAttenuation");
  }
  /**
   * Sets all the attenuation's factors.
   * @param c the new constant factor of attenuation
   * @param l the new linear factor of attenuation
   * @param q the new quadratic factor of attenuation
   */
  public void setAttenuation(float c, float l, float q) {
    constant = c;
    linear = l;
    quadratic = q;
    firePropertyChange("attenuation");
  }
  
  /** Constant for point light type. Those lights are luminous points, like a fire. */
  public static final int TYPE_POINT = 0;
  /** Constant for directionnal light type, like the sun. */
  public static final int TYPE_DIRECTIONAL = 1;
  /** Constant for spot light type. */
  public static final int TYPE_SPOT = 2;
  protected int type = TYPE_POINT;
  /**
   * Gets the type of this light.
   * Default is TYPE_POINT .
   * @return a TYPE_* constant
   */
  public int getType() { return type; }
  /**
   * Sets the type of this light.
   * @param i a TYPE_* constant
   */
  public void setType(int i) {
    if(type != i) {
      type = i;
      switch(type) {
        case TYPE_POINT:
          spotAngle = 180f;
          coords[2] =   0f;
          coords[3] =   1f;
          break;
        case TYPE_DIRECTIONAL:
          spotAngle = 180f;
          coords[2] =   1f;
          coords[3] =   0f;
          break;
        case TYPE_SPOT:
          if(spotAngle == 180) spotAngle = 45f;
          coords[2] =   0f;
          coords[3] =   1f;
          break;
        default: throw new UnsupportedOperationException("Light3D : type must be a TYPE_* constant.");
      }
      firePropertyChange("type");
    }
  }

  private float spotAngle = 180f;
  /**
   * Gets the spot angle of this light. The spot angle is 180f for non-spot light; for spot
   * light, the default is 45f . The higher it is, the more the spot will be opened.
   * @return the spot angle in degrees
   */
  public float getSpotAngle () { return spotAngle ; }
  /**
   * Sets the spot angle of this light. If the new value is different from 180f, the light
   * type will be set to TYPE_SPOT .
   * @param f the spot angle in degrees
   */
  public void setSpotAngle(float f) {
    if(spotAngle != f) {
      spotAngle = f;
      if(spotAngle != 180 && type != TYPE_SPOT) setType(TYPE_SPOT);
      firePropertyChange("spotAngle");
    }
  }

  private float spotExponent = 0f;
  /**
   * Gets the spotExponent of this light. The spotExponent only deals with spot light; it
   * represents the difference of intensity between the middle and the side of the spot.
   * Default is 0f .
   * @return the spotExponent
   */
  public float getSpotExponent() { return spotExponent; }
  /**
   * Sets the spotExponent of this light.
   * @param f the spotExponent
   */
  public void setSpotExponent(float f) {
    if(spotExponent != f) {
      spotExponent = f;
      firePropertyChange("spotExponent");
    }
  }
  
  private boolean dynamic = false;
  /**
   * Checks if this light is static. A static light is applied on static-lit shapes at their
   * creation, and not after.
   * Default is true.
   * @see opale.soya.soya3d.World3D#toShape
   * @return true if static
   */
  public boolean isStatic() { return dynamic; }
  /**
   * Sets if this light is static.
   * @param b true if static
   */
  public void setStatic(boolean b) {
    if(b != dynamic) {
      dynamic = b;
      firePropertyChange("static");
    }
  }
  
  private boolean pureStatic;
  /**
   * Checks if this light is pure static. A pure static light is used for computing static
   * lighting, but do not light normal object. Default is false.
   * This property has only a meaning if the dynamic one is true (sic!).
   * @return true is pure static
   */
  public boolean isPureStatic() { return pureStatic; }
  /**
   * Sets if this light is pure static.
   * @param b true if pure static
   */
  public void setPureStatic(boolean b) {
    if(b != pureStatic) {
      pureStatic = b;
      firePropertyChange("pureStatic");
    }
  }
  
  /**
   * Gets the attenuation factor of this light at the given position, as defined by openGL.
   * @param p the position
   * @return the attenuation factor
   */
  public float attenuationFactorAt(Position p) {
    if(type == TYPE_DIRECTIONAL) return 1f;
    p = convertToThisCoordSyst(p);
    float distance = (float) Math.sqrt(p.getX() * p.getX() + p.getY() * p.getY() + p.getZ() * p.getZ());
    return 1f / (constant + linear * distance + quadratic * distance * distance);
  }
  /**
   * Gets the attenuation factor of this light at the given position, as defined by openGL.
   * @param x the x coordinate of the position in the light coordinates system
   * @param y the y coordinate of the position in the light coordinates system
   * @param z the z coordinate of the position in the light coordinates system
   * @return the attenuation factor
   */
  public float attenuationFactorAt(float x, float y, float z) {
    if(type == TYPE_DIRECTIONAL) return 1f;
    float distance = (float) Math.sqrt((double) (x * x + y * y + z * z));
    return 1f / (constant + linear * distance + quadratic * distance * distance);
  }
  /**
   * Gets the attenuation factor of this light at the given distance, as defined by openGL.
   * @param distance the distance
   * @return the attenuation factor
   */
  public float attenuationFactorAt(float distance) {
    if(type == TYPE_DIRECTIONAL) return 1f;
    return 1f / (constant + linear * distance + quadratic * distance * distance);
  }
  /**
   * Gets the spot light effect of this light at the given position, as defined by openGL.
   * @param p the position
   * @return the spot light effect
   */
  public float spotLightEffectAt(Position p) {
    if(type != TYPE_SPOT) return 1f;
    float z = convertToThisCoordSyst(p).getZ();
    float cos = (float) Math.cos((double) (spotAngle * Matrix.PI / 180f));
    float f   = (z - getZ()) * -1f; // Dot product, where an x and an y is 0f.
    if(f < 0f) f = 0f;
    if(f < cos) return 0f;
    return (float) Math.pow((double) f, spotExponent);
  }
  /**
   * Gets the spot light effect of this light at the given position, as defined by openGL.
   * @param x the x coordinate of the position in the light coordinates system
   * @param y the y coordinate of the position in the light coordinates system
   * @param z the z coordinate of the position in the light coordinates system
   * @return the spot light effect
   */
  public float spotLightEffectAt(float x, float y, float z) {
    if(type != TYPE_SPOT) return 1f;
    float cos = (float) Math.cos((double) (spotAngle * Matrix.PI / 180f));
    float f   = (z - getZ()) * -1f; // Dot product, where an x and an y is 0f.
    if(f < 0f) f = 0f;
    if(f < cos) return 0f;
    return (float) Math.pow(f, spotExponent);
  }
  /**
   * Gets the diffuse color due to this light, at the given position.
   * Notice that the specular is not used here.
   * @param x the x coordinate of the position in the light coordinates system
   * @param y the y coordinate of the position in the light coordinates system
   * @param z the z coordinate of the position in the light coordinates system
   * @param nx the x coordinate of the normal vector in the light coordinates system
   * @param ny the y coordinate of the normal vector in the light coordinates system
   * @param nz the z coordinate of the normal vector in the light coordinates system
   * @param materialColor the material color
   * @return the color in a float[4]
   */
  public float[] diffuseColorAt(float x, float y, float z, float nx, float ny, float nz, float[] materialColor) {
    float distance = (float) Math.sqrt((double) (x * x + y * y + z * z));
    float[] c = new float[4];
    float factor = attenuationFactorAt(distance) * spotLightEffectAt(x, y, z);
    if(factor < 0.0001f) return c;
    float diffuseFactor = -x / distance * nx + -y / distance * ny + -z / distance * nz;
    if(diffuseFactor < 0f) diffuseFactor = 0f;
    c[0] = factor * diffuseFactor * materialColor[0] * color[0];
    c[1] = factor * diffuseFactor * materialColor[1] * color[1];
    c[2] = factor * diffuseFactor * materialColor[2] * color[2];
    c[3] = factor * diffuseFactor * materialColor[3] * color[3];
    return c;
  }
  /**
   * Gets the diffuse color due to this light, at the given position.
   * Notice that the specular is not used here.
   * @param p the position
   * @param n the normal vector
   * @param materialColor the material color
   * @return the color in a float[4]
   */
  public float[] diffuseColorAt(Position p, Vector n, float[] materialColor) {
    p = convertToThisCoordSyst(p);
    n = (Vector) convertToThisCoordSyst(n);
    return diffuseColorAt(p.getX(), p.getY(), p.getZ(), n.getX(), n.getY(), n.getZ(), materialColor);
  }
  private Position convertToThisCoordSyst(Position p) {
    CoordSyst f2 = p.getCoordSyst();
    if((this != f2) && (f2 != null)) return p.clone(this);
    else return p; // No conversion required.
  }
  
  // Colored :
  protected final float[] color = (float[]) Material.WHITE_COLOR.clone();
  public float[] getColor() { return color; }
  public void setColor(float red, float green, float blue) {
    setColor(red, green, blue, 1f);
  }
  public void setColor(float red, float green, float blue, float alpha) {
    color[0] = red;
    color[1] = green;
    color[2] = blue;
    color[3] = alpha;
    firePropertyChange("color");
  }
  public void setColor(float[] i) {
    System.arraycopy(i, 0, color, 0, 4);
    firePropertyChange("color");
  }
  public float getRed  () { return color[0]; }
  public float getGreen() { return color[1]; }
  public float getBlue () { return color[2]; }
  public float getAlpha() { return color[3]; }
  public void setRed  (float f) {
    color[0] = f;
    firePropertyChange("color");
  }
  public void setGreen(float f) {
    color[1] = f;
    firePropertyChange("color");
  }
  public void setBlue (float f) {
    color[2] = f;
    firePropertyChange("color");
  }
  public void setAlpha(float f) {
    color[3] = f;
    firePropertyChange("color");
  }
  public boolean getUseAlpha() { return false; }
  public boolean getUseColor() { return true; }
}
