/*
 * Soya3D
 * Copyright (C) 1999-2000 Jean-Baptiste LAMY (Artiste on the web)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package opale.soya.soya3d;

import opale.soya.*;
import opale.soya.util.*;
import opale.soya.soya2d.*;
import opale.soya.soya3d.model.*;
import gl4java.*;
import java.util.*;
import java.io.*;
import java.beans.*;

/**
 * Abstract class for graphical 3D elements that can act either as fragments or as 3D
 * elements.
 * 
 * This class is design to be extended by many other element / fragment. You just need to
 * implements a few methods :
 *  - getNaturalWidth(), getNaturalHeight(), getNaturalDepth() : returns the natural
 * dimensions : without any scaling, ... just the size of what you draw in the draw method.
 *  - wrapper() : returns a dimension wrapper (like a Box).
 *  - getMaterial() : returns the material; you should add setMaterial(Material m) if needed.
 *  - draw(Renderer r, GLFunc gl, GLUFunc glu) : this is your drawing method.
 * 
 * You may want to override the following methods :
 *  - getUseAlpha() : if your fragment element use alpha blending, or if you want it to be drawn
 * at the end, in a second pass. The default implementation returns true if and only if the
 * material (returned by getMaterial()) use alpha.
 *  - fillCollector(FragmentsCollector f, Renderer r, float[] mat) : to perform special optimization
 * (= do not collect the fragment element if not needed).
 *  - reBuild() : if you want to do something each time the fragment element is modified.
 * 
 * Add also your own properties and method; whenever the fragment element is changed, you
 * must call the reBuild() method.
 * 
 * If your fragment element do not change its drawing, except when its properties are
 * changed, you should consider to use rather a FixedFragmentElement3D.
 * 
 * @see opale.soya.soya3d.model.FragmentFromElement
 * @see opale.soya.soya3d.model.FragmentedShapeElement
 * @see opale.soya.soya3d.Box
 * @see opale.soya.soya3d.FixedFragmentElement3D
 * @author Artiste on the Web
 */

public abstract class FragmentElement3D extends GraphicalElement3D implements Fragmentable, Drawable, FragmentedShapeElement, PreTransformable {
  private static final long serialVersionUID = -631315446376864609l;
  
  /**
   * Creates a new fragment element.
   */
  public FragmentElement3D() {  }
  /**
   * Creates a new fragment element, with the given name.
   */
  public FragmentElement3D(String newName) { super(newName); }
  
  /**
   * Clones this fragment element.
   * @return the clone.
   */
  public synchronized Object clone() {
    FragmentElement3D o = (FragmentElement3D) super.clone();
    return o;
  }
  
  // FragmentedShapeElement :
  public Class getFragmentClass() { return opale.soya.soya3d.model.FragmentFromElement.class; }
  public boolean canShareFragmentWith(FragmentedShapeElement se) { return false; }
  public Position getCenter() { return this; }
  
  protected FragmentFromElement fragment;
  public FragmentFromElement getFragment() {
    if(fragment == null) fragment = new FragmentFromElement(this); // Create it.
    return fragment;
  }
  
  // Fragmentable :
  public Fragment[] toFragments() {
    FragmentElement3D fe = (FragmentElement3D) clone();
    System.out.println(fe);
    fe.printMatrix();
    Fragment[] fs = { fe.getFragment() };
    //Fragment[] fs = { ((FragmentElement3D) clone()).getFragment() };
    return fs;
  }
  
  // FragmentsCollectorFiller :
  public void fillCollector(DrawablesCollector f, Renderer r, float[] mat) { // Optimizable
    if(visible) {
      mat = Matrix.matrixMultiply(mat, m);
      if(leftHanded) f.invertConfiguration();
      if(specialEffect == null) f.collect(this, mat);
      else {
        f.   addSpecialEffect(specialEffect);
        f.collect(this, mat);
        f.removeSpecialEffect(specialEffect);
      }
      if(leftHanded) f.invertConfiguration();
    }
  }
  
  // PreTransformable :
  public void preTransform(float[] mat) { fragment.preTransform(mat); }
  
  // Dimension :
  public float getWidth () { return getNaturalWidth () * factorX; }
  public float getHeight() { return getNaturalHeight() * factorY; }
  public float getDepth () { return getNaturalDepth () * factorZ; }
  /**
   * Gets the width dimension, without the scaling.
   * @return the width with no scaling
   */
  protected abstract float getNaturalWidth ();
  /**
   * Gets the height dimension, without the scaling.
   * @return the height with no scaling
   */
  protected abstract float getNaturalHeight();
  /**
   * Gets the depth dimension, without the scaling.
   * @return the depth with no scaling
   */
  protected abstract float getNaturalDepth ();
  /**
   * Gets a dimension wrapper for this dimension. Suach a wrapper can be used for converting
   * dimensions from a coordinates system to another. The exact class of wrapper returned
   * depends on the dimension class.
   * This is not a property, and calling twice this method may not return the same wrapper.
   * @see opale.soya.soya3d.Box
   * @return a dimension wrapper
   */
  public abstract DimensionWrapper wrapper();
  
  /**
   * Gets this fragment's material. A fragment element can use only a single material.
   * @return the material
   */
  public abstract Material getMaterial();
  /**
   * Returns true if the shape element use alpha blending or alpha function.
   * Should be overriden if you use it. Alpha-user fragment are rendered at the end.
   * Default implementation return true if the material use alpha.
   * @return true if use alpha.
   */
  public boolean getUseAlpha() { return getMaterial().getUseAlpha(); }
  
  /**
   * This method must be called when a property has changed.
   */
  public void reBuild() {  }
  
  /**
   * Draws the fragment element.
   * @param r the renderer
   * @param gl  the gl
   * @param glu the glu
   */
  public abstract void draw(Renderer r, GLFunc gl, GLUFunc glu);
  
  public void draw(Renderer r, GLFunc gl, GLUFunc glu, SpecialEffect[] specialEffects) {
    for(int i = 0; i < specialEffects.length; i++) specialEffects[i].  makeCurrent(gl, glu);
    draw(r, gl, glu);
    for(int i = 0; i < specialEffects.length; i++) specialEffects[i].unmakeCurrent(gl, glu);
  }
  
  // CoordSyst : overrides :
  public CoordSyst getCoordSyst() {
    if(fragment != null) return fragment;
    else return parent;
  }
  public CoordSyst getRootCoordSyst() {
    if(fragment != null) return fragment.getRootCoordSyst();
    else return super.getRootCoordSyst();
  }
}
